<?php

declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0'); // prod: 0; dev: 1

/**
 * SOL - Sistema de Operaciones Logísticas
 * Endpoint: api/control/conteo_layout.php
 *
 * Endpoints:
 *   GET  ?meta=depositos
 *   GET  ?meta=racks&deposito_id=#
 *   GET  (principal) deposito_id, rack[, only_active=1][, fecha=YYYY-MM-DD]
 *   GET  ?detail=1&fondo_id=#&fecha=YYYY-MM-DD
 *   POST ?save=1  (body JSON con conteoPayload)
 */

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

// === helpers ================================================================

function read_json_body(): array
{
  $raw = file_get_contents('php://input');
  if (!$raw) return [];
  $j = json_decode($raw, true);
  return (is_array($j) ? $j : []);
}

function jerr(string $msg, ?Throwable $e = null, int $code = 400): void
{
  http_response_code($code);
  echo json_encode([
    'ok'     => false,
    'error'  => $msg,
    'detail' => (env('APP_ENV') === 'local' && $e) ? $e->getMessage() : null
  ], JSON_UNESCAPED_UNICODE);
  exit;
}

/** Valida fecha Y-m-d y devuelve string o null */
function normalize_date(?string $s): ?string
{
  $s = trim((string)$s);
  if ($s === '') return null;
  $dt = DateTime::createFromFormat('Y-m-d', $s);
  if (!$dt) return null;
  $errors = DateTime::getLastErrors();
  if ($errors && ($errors['warning_count'] > 0 || $errors['error_count'] > 0)) return null;
  return $dt->format('Y-m-d');
}

try {
  $pdo = get_pdo();

  // --------------------------------------------------------------------------
  // POST SAVE: guarda el conteo de una posición
  //   POST ?save=1
  //   Body JSON:
  //     fondo_id (int) → position_id
  //     fecha (YYYY-MM-DD)
  //     nota (string|null)  [opcional]
  //     items: [ { producto_id, lote, vencimiento, uc_esperadas, uv_esperadas, uc_contadas, uv_contadas, obs } ... ]
  // --------------------------------------------------------------------------
  if ((isset($_GET['save']) && $_GET['save'] == '1') || (isset($_POST['save']) && $_POST['save'] == '1')) {
    try {
      $in = read_json_body();
      if (!$in) {
        $in = $_POST;
        if (isset($in['items']) && is_string($in['items'])) {
          $dec = json_decode($in['items'], true);
          if (is_array($dec)) $in['items'] = $dec;
        }
      }

      $position_id = (int)($in['fondo_id'] ?? 0);
      $fecha       = normalize_date($in['fecha'] ?? null) ?: date('Y-m-d');
      $nota        = isset($in['nota']) ? trim((string)$in['nota']) : null;
      $items       = (isset($in['items']) && is_array($in['items'])) ? $in['items'] : [];

      if ($position_id <= 0) jerr('Parámetro requerido: fondo_id', null, 422);

      $pdo->beginTransaction();

      // Si ya existe conteo para esa fecha+posición, lo actualizamos (reemplazando items)
      $stmt = $pdo->prepare("SELECT id FROM wh_counts WHERE fecha = :f AND position_id = :p LIMIT 1");
      $stmt->execute([':f' => $fecha, ':p' => $position_id]);
      $row = $stmt->fetch(PDO::FETCH_ASSOC);

      if ($row) {
        $count_id = (int)$row['id'];
        $pdo->prepare("DELETE FROM wh_count_items WHERE count_id = :id")->execute([':id' => $count_id]);
        $pdo->prepare("UPDATE wh_counts SET nota = :n, updated_at = NOW() WHERE id = :id")
          ->execute([':n' => $nota, ':id' => $count_id]);
      } else {
        $stmt = $pdo->prepare("
          INSERT INTO wh_counts (fecha, position_id, user_id, nota, created_at)
          VALUES (:f, :p, :u, :n, NOW())
        ");
        $stmt->execute([
          ':f' => $fecha,
          ':p' => $position_id,
          ':u' => null, // TODO: setear user_id si hay sesión
          ':n' => $nota,
        ]);
        $count_id = (int)$pdo->lastInsertId();
      }

      if ($items) {
        $ins = $pdo->prepare("
          INSERT INTO wh_count_items
            (count_id, producto_id, lote, vencimiento, uc_esperadas, uv_esperadas, uc_contadas, uv_contadas, obs, created_at)
          VALUES
            (:c, :prod, :lote, :venc, :uce, :uve, :ucc, :uvc, :obs, NOW())
        ");
        foreach ($items as $it) {
          $ins->execute([
            ':c'   => $count_id,
            ':prod' => isset($it['producto_id']) ? (int)$it['producto_id'] : null,
            ':lote' => isset($it['lote']) ? (string)$it['lote'] : null,
            ':venc' => !empty($it['vencimiento']) ? substr((string)$it['vencimiento'], 0, 10) : null,
            ':uce' => isset($it['uc_esperadas']) ? (int)$it['uc_esperadas'] : 0,
            ':uve' => isset($it['uv_esperadas']) ? (int)$it['uv_esperadas'] : 0,
            ':ucc' => isset($it['uc_contadas']) ? (int)$it['uc_contadas'] : 0,
            ':uvc' => isset($it['uv_contadas']) ? (int)$it['uv_contadas'] : 0,
            ':obs' => isset($it['obs']) ? (string)$it['obs'] : null,
          ]);
        }
      }

      $pdo->commit();

      echo json_encode([
        'ok'            => true,
        'count_id'      => $count_id,
        'position_id'   => $position_id,
        'fecha'         => $fecha,
        'items'         => count($items),
        'message'       => 'Conteo registrado',
        'show_expected' => false,
      ], JSON_UNESCAPED_UNICODE);
      exit;
    } catch (Throwable $e) {
      if ($pdo->inTransaction()) $pdo->rollBack();
      jerr('Error guardando conteo: ' . $e->getMessage(), null, 500);
    }
  }

  // --------------------------------------------------------------------------
  // GET DETAIL: datos para el modal de conteo (?detail=1)
  // --------------------------------------------------------------------------
  if (isset($_GET['detail'])) {
    $fondo_id = (int)($_GET['fondo_id'] ?? 0);
    if ($fondo_id <= 0) jerr('Parámetro requerido: fondo_id', null, 422);

    $fecha = normalize_date($_GET['fecha'] ?? null);

    // 1) Posición
    $stmt = $pdo->prepare("
      SELECT p.id, p.deposito_id, p.rack, p.columna, p.nivel, p.fondo
      FROM wh_positions p
      WHERE p.id = :id
      LIMIT 1
    ");
    $stmt->execute([':id' => $fondo_id]);
    $pos = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$pos) jerr('Posición no encontrada', null, 404);

    $depId = (int)$pos['deposito_id'];
    $rkId  = (int)$pos['rack'];
    $col   = (int)$pos['columna'];
    $niv   = (int)$pos['nivel'];
    $depth = (int)$pos['fondo'];

    // 2) Pallets en la posición
    $stmt = $pdo->prepare("
      SELECT id, codigo, estado
      FROM wh_pallets
      WHERE position_id = :pid
      ORDER BY id
    ");
    $stmt->execute([':pid' => $fondo_id]);
    $pallets = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $palletIds   = array_map(fn($r) => (int)$r['id'], $pallets);
    $palletCount = count($palletIds);

    $palletCodeHdr = null;
    if ($palletCount === 1) $palletCodeHdr = $pallets[0]['codigo'] ?? null;
    elseif ($palletCount >  1)  $palletCodeHdr = $palletCount . ' pallets';

    $estadoHdr = $palletCount > 0 ? 'ACTIVE' : 'EMPTY';

    $hdr = [
      'deposito_code' => 'DEP' . $depId,
      'rack_code'     => 'R' . str_pad((string)$rkId, 2, '0', STR_PAD_LEFT),
      'pos_code'      => sprintf('C%02d-N%02d-F%02d', $col, $niv, $depth),
      'pallet_code'   => $palletCodeHdr,
      'estado'        => $estadoHdr,
    ];

    // 3) Ítems de los pallets
    $items = [];
    if ($palletCount > 0) {
      $placeholders = implode(',', array_fill(0, $palletCount, '?'));

      $sqlWithJoin = "
        SELECT
          i.id, i.pallet_id, i.producto_id, i.lote,
          i.fecha_vencimiento, i.uv_cajas, i.uc_por_caja, i.uc_sueltas,
          i.uc_total_cache, i.estado, i.nota,
          p.sku AS prod_sku, p.denominacion AS prod_den
        FROM wh_pallet_items i
        LEFT JOIN para_productos p ON p.id = i.producto_id
        WHERE i.pallet_id IN ($placeholders)
        ORDER BY i.pallet_id, i.id
      ";

      $sqlNoJoin = "
        SELECT
          i.id, i.pallet_id, i.producto_id, i.lote,
          i.fecha_vencimiento, i.uv_cajas, i.uc_por_caja, i.uc_sueltas,
          i.uc_total_cache, i.estado, i.nota
        FROM wh_pallet_items i
        WHERE i.pallet_id IN ($placeholders)
        ORDER BY i.pallet_id, i.id
      ";

      try {
        $stmt = $pdo->prepare($sqlWithJoin);
        $stmt->execute($palletIds);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
      } catch (Throwable $e) {
        $stmt = $pdo->prepare($sqlNoJoin);
        $stmt->execute($palletIds);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        foreach ($rows as &$r) {
          if (!array_key_exists('prod_sku', $r)) $r['prod_sku'] = null;
          if (!array_key_exists('prod_den', $r)) $r['prod_den'] = null;
        }
        unset($r);
      }

      foreach ($rows as $r) {
        $uv   = (int)($r['uv_cajas'] ?? 0);
        $ucpc = (int)($r['uc_por_caja'] ?? 0);
        $ucs  = (int)($r['uc_sueltas'] ?? 0);
        $uc_calc = $uv * $ucpc + $ucs;
        $uc = (int)($r['uc_total_cache'] ?? 0);
        if ($uc <= 0) $uc = $uc_calc;

        $items[] = [
          'producto_id'   => (int)($r['producto_id'] ?? 0),
          'sku'           => $r['prod_sku'] ?? null,
          'denominacion'  => $r['prod_den'] ?? null,
          'lote'          => $r['lote'] ?? null,
          'vencimiento'   => isset($r['fecha_vencimiento']) && $r['fecha_vencimiento'] !== null
            ? substr((string)$r['fecha_vencimiento'], 0, 10) : null,
          'uc_esperadas'  => $uc,
          'uv_esperadas'  => $uv,
          'uc_contadas'   => 0,
          'uv_contadas'   => 0,
          'obs'           => $r['nota'] ?? null,
        ];
      }
    }

    echo json_encode([
      'ok'            => true,
      'date'          => $fecha ?: null,
      'header'        => $hdr,
      'items'         => $items,
      'show_expected' => false, // por ahora ocultamos columnas esperadas en UI
    ], JSON_UNESCAPED_UNICODE);
    exit;
  }

  // --------------------------------------------------------------------------
  // 0) Fecha seleccionada (eco)
  // --------------------------------------------------------------------------
  $fechaParam = normalize_date($_GET['fecha'] ?? null);

  // --------------------------------------------------------------------------
  // GET META: depósitos
  // --------------------------------------------------------------------------
  if (isset($_GET['meta']) && (string)$_GET['meta'] === 'depositos') {
    $rows = $pdo->query("
      SELECT DISTINCT deposito_id AS id
      FROM wh_positions
      ORDER BY deposito_id
    ")->fetchAll(PDO::FETCH_ASSOC);

    $out = array_map(function ($r) {
      $id = (int)$r['id'];
      return ['id' => $id, 'code' => 'DEP' . $id, 'name' => null];
    }, $rows);

    $resp = ['depositos' => $out];
    if ($fechaParam) $resp['date'] = $fechaParam;
    echo json_encode($resp, JSON_UNESCAPED_UNICODE);
    exit;
  }

  // --------------------------------------------------------------------------
  // GET META: racks por depósito
  // --------------------------------------------------------------------------
  if (isset($_GET['meta']) && (string)$_GET['meta'] === 'racks') {
    $deposito_id = (int)($_GET['deposito_id'] ?? 0);
    if ($deposito_id <= 0) jerr('Parámetro requerido: deposito_id', null, 422);

    $stmt = $pdo->prepare("
      SELECT DISTINCT rack
      FROM wh_positions
      WHERE deposito_id = :dep
      ORDER BY rack
    ");
    $stmt->execute([':dep' => $deposito_id]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $out = array_map(function ($r) {
      $rk = (int)$r['rack'];
      return [
        'id'               => $rk,
        'code'             => 'R' . str_pad((string)$rk, 2, '0', STR_PAD_LEFT),
        'name'             => null,
        'orientacion_code' => null,
        'lado_code'        => null
      ];
    }, $rows);

    $resp = ['racks' => $out];
    if ($fechaParam) $resp['date'] = $fechaParam;
    echo json_encode($resp, JSON_UNESCAPED_UNICODE);
    exit;
  }

  // --------------------------------------------------------------------------
  // GET PRINCIPAL: posiciones de un rack (por depósito + rack)
  // Devuelve flag 'counted' según wh_counts(fecha, position_id)
  // --------------------------------------------------------------------------
  $deposito_id = (int)($_GET['deposito_id'] ?? 0);
  $rackParam   = trim((string)($_GET['rack'] ?? ''));

  if ($deposito_id <= 0 || $rackParam === '') {
    jerr('Parámetros requeridos: deposito_id y rack', null, 422);
  }

  if (preg_match('/^\s*[Rr]\s*0*([0-9]+)\s*$/', $rackParam, $m)) $rack = (int)$m[1];
  else $rack = (int)$rackParam;
  if ($rack <= 0) jerr('Parámetro rack inválido', null, 422);

  $only_active = (int)($_GET['only_active'] ?? 0) === 1;

  // Trae posiciones
  $sql = "
    SELECT id, deposito_id, rack, columna, nivel, fondo, orientacion, activo
    FROM wh_positions
    WHERE deposito_id = :dep AND rack = :rk
  ";
  if ($only_active) $sql .= " AND activo = 1";
  $sql .= " ORDER BY columna, nivel, fondo";

  $stmt = $pdo->prepare($sql);
  $stmt->bindValue(':dep', $deposito_id, PDO::PARAM_INT);
  $stmt->bindValue(':rk',  $rack,        PDO::PARAM_INT);
  $stmt->execute();
  $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

  // Mapa de posiciones contadas para la fecha (si hay fecha)
  $countsMap = [];
  if ($fechaParam) {
    $posIds = array_map(fn($r) => (int)$r['id'], $rows);
    if ($posIds) {
      $ph = implode(',', array_fill(0, count($posIds), '?'));
      $q  = "SELECT position_id FROM wh_counts WHERE fecha = ? AND position_id IN ($ph)";
      $stmt2 = $pdo->prepare($q);
      $stmt2->execute(array_merge([$fechaParam], $posIds));
      foreach ($stmt2->fetchAll(PDO::FETCH_COLUMN, 0) as $pid) {
        $countsMap[(int)$pid] = true;
      }
    }
  }

  // Mapeo a contrato del layout
  $positions = array_map(function (array $r) use ($countsMap): array {
    $id     = (int)$r['id'];
    $dep    = (int)$r['deposito_id'];
    $rk     = (int)$r['rack'];
    $col    = (int)$r['columna'];
    $niv    = (int)$r['nivel'];
    $fondo  = (int)$r['fondo'];
    $activo = (int)$r['activo'] === 1;

    $col_code = 'C' . str_pad((string)$col, 2, '0', STR_PAD_LEFT);
    $niv_code = 'N' . str_pad((string)$niv, 2, '0', STR_PAD_LEFT);
    $rk_code  = 'R' . str_pad((string)$rk,  2, '0', STR_PAD_LEFT);
    $dep_code = 'DEP' . $dep;

    return [
      'id'             => $id,
      'col_code'       => $col_code,
      'niv_code'       => $niv_code,
      'depth_index'    => $fondo,
      'code_full'      => "{$dep_code}-{$rk_code}-{$col_code}-{$niv_code}-F" . str_pad((string)$fondo, 2, '0', STR_PAD_LEFT),
      'status'         => $activo ? 'ACTIVE' : 'INACTIVE',
      'is_pick_face'   => ($fondo === 1) ? 1 : 0,
      'blocked_reason' => null,
      'fondo_id'       => $id,
      'counted'        => !empty($countsMap[$id]),   // ← NUEVO: ya contada en esa fecha
    ];
  }, $rows);

  $resp = [
    'ok'       => true,
    'date'     => $fechaParam ?: null,
    'deposito' => ['id' => $deposito_id, 'code' => 'DEP' . $deposito_id, 'name' => null],
    'rack'     => ['id' => $rack, 'code' => 'R' . str_pad((string)$rack, 2, '0', STR_PAD_LEFT), 'name' => null, 'orientacion_code' => null],
    'positions' => $positions,
  ];

  echo json_encode($resp, JSON_UNESCAPED_UNICODE);
  exit;
} catch (Throwable $e) {
  jerr('Error cargando layout de conteo', $e, 500);
}
