<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';

if (empty($_SESSION['usuario_id'])) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'No autorizado']);
  exit;
}

try {
  $input = json_decode(file_get_contents('php://input'), true);
  
  $id = (int)($input['id'] ?? 0);
  $codigo = trim((string)($input['codigo'] ?? ''));
  $fecha_conteo = trim((string)($input['fecha_conteo'] ?? ''));
  $deposito_id = (int)($input['deposito_id'] ?? 0);
  $cliente_id = !empty($input['cliente_id']) ? (int)$input['cliente_id'] : null;
  $tipo = trim((string)($input['tipo'] ?? 'GENERAL'));
  $estado = trim((string)($input['estado'] ?? 'BORRADOR'));
  $observaciones = trim((string)($input['observaciones'] ?? ''));
  $user_id = (int)$_SESSION['usuario_id'];

  // Validaciones
  if (empty($codigo) || empty($fecha_conteo) || $deposito_id <= 0) {
    throw new InvalidArgumentException('Datos incompletos');
  }

  if (!in_array($tipo, ['GENERAL', 'CICLICO', 'PARCIAL'])) {
    throw new InvalidArgumentException('Tipo inválido');
  }

  if (!in_array($estado, ['BORRADOR', 'EN_PROCESO', 'FINALIZADO', 'APROBADO', 'RECHAZADO'])) {
    throw new InvalidArgumentException('Estado inválido');
  }

  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  if ($id > 0) {
    // Actualizar
    $sql = "UPDATE inv_conteos SET 
              codigo = :codigo,
              fecha_conteo = :fecha_conteo,
              deposito_id = :deposito_id,
              cliente_id = :cliente_id,
              tipo = :tipo,
              estado = :estado,
              observaciones = :observaciones,
              updated_at = NOW()
            WHERE id = :id AND deleted_at IS NULL";
    
    $st = $pdo->prepare($sql);
    $st->execute([
      ':id' => $id,
      ':codigo' => $codigo,
      ':fecha_conteo' => $fecha_conteo,
      ':deposito_id' => $deposito_id,
      ':cliente_id' => $cliente_id,
      ':tipo' => $tipo,
      ':estado' => $estado,
      ':observaciones' => $observaciones
    ]);

    echo json_encode(['ok' => true, 'id' => $id, 'message' => 'Actualizado correctamente']);
    
  } else {
    // Insertar nuevo
    $sql = "INSERT INTO inv_conteos 
              (codigo, fecha_conteo, deposito_id, cliente_id, tipo, estado, observaciones, user_id, created_at)
            VALUES 
              (:codigo, :fecha_conteo, :deposito_id, :cliente_id, :tipo, :estado, :observaciones, :user_id, NOW())";
    
    $st = $pdo->prepare($sql);
    $st->execute([
      ':codigo' => $codigo,
      ':fecha_conteo' => $fecha_conteo,
      ':deposito_id' => $deposito_id,
      ':cliente_id' => $cliente_id,
      ':tipo' => $tipo,
      ':estado' => $estado,
      ':observaciones' => $observaciones,
      ':user_id' => $user_id
    ]);

    $newId = (int)$pdo->lastInsertId();
    echo json_encode(['ok' => true, 'id' => $newId, 'message' => 'Creado correctamente']);
  }

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'ok' => false,
    'error' => $e->getMessage()
  ]);
}
