<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';

if (empty($_SESSION['usuario_id'])) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'No autorizado']);
  exit;
}

try {
  $input = json_decode(file_get_contents('php://input'), true);
  
  $id = (int)($input['id'] ?? 0);
  $conteo_id = (int)($input['conteo_id'] ?? 0);
  $producto_id = (int)($input['producto_id'] ?? 0);
  $lote = trim((string)($input['lote'] ?? ''));
  $posicion = trim((string)($input['posicion'] ?? ''));
  $cantidad_pallets = (int)($input['cantidad_pallets'] ?? 0);
  $cantidad_cajas = (int)($input['cantidad_cajas'] ?? 0);
  $cantidad_unidades = (int)($input['cantidad_unidades'] ?? 0);
  $total_unidades = (int)($input['total_unidades'] ?? 0);
  $observaciones = trim((string)($input['observaciones'] ?? ''));

  // Validaciones
  if ($conteo_id <= 0 || $producto_id <= 0) {
    throw new InvalidArgumentException('Datos incompletos');
  }

  if ($cantidad_pallets < 0 || $cantidad_cajas < 0 || $cantidad_unidades < 0) {
    throw new InvalidArgumentException('Las cantidades no pueden ser negativas');
  }

  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  if ($id > 0) {
    // Actualizar item existente
    $sql = "UPDATE inv_conteo_items SET 
              producto_id = :producto_id,
              lote = :lote,
              posicion = :posicion,
              cantidad_pallets = :cantidad_pallets,
              cantidad_cajas = :cantidad_cajas,
              cantidad_unidades = :cantidad_unidades,
              total_unidades = :total_unidades,
              observaciones = :observaciones
            WHERE id = :id AND conteo_id = :conteo_id";
    
    $st = $pdo->prepare($sql);
    $st->execute([
      ':id' => $id,
      ':conteo_id' => $conteo_id,
      ':producto_id' => $producto_id,
      ':lote' => $lote ?: null,
      ':posicion' => $posicion ?: null,
      ':cantidad_pallets' => $cantidad_pallets,
      ':cantidad_cajas' => $cantidad_cajas,
      ':cantidad_unidades' => $cantidad_unidades,
      ':total_unidades' => $total_unidades,
      ':observaciones' => $observaciones
    ]);

    echo json_encode(['ok' => true, 'id' => $id, 'message' => 'Item actualizado']);
    
  } else {
    // Verificar duplicados según lote y posición
    $loteCheck = $lote ?: null;
    $posicionCheck = $posicion ?: null;
    
    // Si lote y posición están vacíos, se asume conteo total del producto
    // Solo permitir una entrada "sin lote ni posición" por producto
    if ($loteCheck === null && $posicionCheck === null) {
      // Verificar si ya existe este producto sin lote ni posición (conteo total)
      $sql = "SELECT id FROM inv_conteo_items 
              WHERE conteo_id = :conteo_id 
              AND producto_id = :producto_id 
              AND lote IS NULL 
              AND posicion IS NULL";
      $st = $pdo->prepare($sql);
      $st->execute([
        ':conteo_id' => $conteo_id, 
        ':producto_id' => $producto_id
      ]);
      
      if ($st->fetch()) {
        throw new InvalidArgumentException('Este producto ya fue contado (stock total). Use lote/posición para conteos específicos.');
      }
    } else {
      // Verificar duplicado exacto de producto + lote + posición
      $sql = "SELECT id FROM inv_conteo_items 
              WHERE conteo_id = :conteo_id 
              AND producto_id = :producto_id 
              AND (lote <=> :lote)
              AND (posicion <=> :posicion)";
      $st = $pdo->prepare($sql);
      $st->execute([
        ':conteo_id' => $conteo_id, 
        ':producto_id' => $producto_id,
        ':lote' => $loteCheck,
        ':posicion' => $posicionCheck
      ]);
      
      if ($st->fetch()) {
        throw new InvalidArgumentException('Este producto con el mismo lote y posición ya fue agregado');
      }
    }

    // Insertar nuevo item
    $sql = "INSERT INTO inv_conteo_items 
              (conteo_id, producto_id, lote, posicion, cantidad_pallets, cantidad_cajas, cantidad_unidades, total_unidades, observaciones)
            VALUES 
              (:conteo_id, :producto_id, :lote, :posicion, :cantidad_pallets, :cantidad_cajas, :cantidad_unidades, :total_unidades, :observaciones)";
    
    $st = $pdo->prepare($sql);
    $st->execute([
      ':conteo_id' => $conteo_id,
      ':producto_id' => $producto_id,
      ':lote' => $lote ?: null,
      ':posicion' => $posicion ?: null,
      ':cantidad_pallets' => $cantidad_pallets,
      ':cantidad_cajas' => $cantidad_cajas,
      ':cantidad_unidades' => $cantidad_unidades,
      ':total_unidades' => $total_unidades,
      ':observaciones' => $observaciones
    ]);

    $newId = (int)$pdo->lastInsertId();
    echo json_encode(['ok' => true, 'id' => $newId, 'message' => 'Item agregado']);
  }

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'ok' => false,
    'error' => $e->getMessage()
  ]);
}
