<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $depositoId = (int)($_GET['deposito_id'] ?? 0);
    if ($depositoId <= 0) {
        http_response_code(400);
        echo json_encode(['ok'=>false, 'error'=>'deposito_id requerido']);
        exit;
    }

    // Obtener información del depósito
    $stmtDep = $pdo->prepare("SELECT * FROM wh_deposito WHERE id = ?");
    $stmtDep->execute([$depositoId]);
    $deposito = $stmtDep->fetch(PDO::FETCH_ASSOC);
    
    if (!$deposito) {
        http_response_code(404);
        echo json_encode(['ok'=>false, 'error'=>'Depósito no encontrado']);
        exit;
    }

    // Obtener todas las posiciones del depósito con información de stock
    $sql = "
        SELECT 
            p.id,
            p.rack,
            p.columna,
            p.nivel,
            p.fondo,
            p.lado,
            p.capacidad_pallets,
            p.ocupado,
            p.picked,
            p.code,
            p.pos_code,
            p.title,
            p.code_full,
            p.pos_code_full,
            COALESCE(SUM(s.qty_uv), 0) AS stock_uv,
            COALESCE(SUM(s.qty_uc), 0) AS stock_uc,
            COUNT(DISTINCT s.pallet_id) AS pallets_count,
            CASE 
                WHEN p.ocupado = 1 THEN 'occupied'
                WHEN p.picked = 1 THEN 'picked'
                WHEN p.capacidad_pallets = 0 THEN 'blocked'
                ELSE 'available'
            END AS status
        FROM wh_posicion p
        LEFT JOIN wh_stock s ON s.posicion_id = p.id AND s.qty_uv > 0
        WHERE p.deposito_id = ?
            AND p.activo = 1
        GROUP BY p.id, p.rack, p.columna, p.nivel, p.fondo, p.lado, p.capacidad_pallets, p.ocupado, p.picked
        ORDER BY p.rack, p.columna, p.nivel, p.fondo
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([$depositoId]);
    $posiciones = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Organizar por racks
    $layout = [];
    foreach ($posiciones as $pos) {
        $rack = $pos['rack'];
        if (!isset($layout[$rack])) {
            $layout[$rack] = [
                'rack' => $rack,
                'positions' => []
            ];
        }
        $layout[$rack]['positions'][] = $pos;
    }

    // Convertir a array indexado
    $layout = array_values($layout);

    echo json_encode([
        'ok' => true,
        'data' => [
            'deposito' => $deposito,
            'layout' => $layout
        ]
    ]);

} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode([
        'ok' => false,
        'error' => 'Error cargando layout',
        'msg' => (env('APP_ENV') === 'local') ? $e->getMessage() : ''
    ]);
}