<?php

declare(strict_types=1);

/**
 * SOL - Sistema de Operaciones Logísticas
 * Endpoint: api/control/mapeo_layout.php
 *
 * Provee:
 *   - ?meta=depositos
 *   - ?meta=racks&deposito_id=#
 *   - ?layout=1&deposito_id=#&rack=R##|##
 *
 * Reglas de pintado (lado API):
 *   - occupied: pallet_count > 0
 *   - picked:   pallet_count > 1  → usa color de pickeado
 *   - color producto (fallback hash determinístico si no hay color en DB)
 *
 * Requisitos:
 *   - config/config.php (Dotenv, helpers env(), url(), project_path())
 *   - config/db.php     (PDO en $pdo o función db())
 *   - Tablas: wh_positions, wh_position_occupancy (hasta NULL = vigente)
 *   - Opcional: wh_pallets, wh_pallet_items, para_productos (para color)
 */

header('Content-Type: application/json; charset=utf-8');

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';

function jexit(array $payload, int $code = 200): void
{
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}

try {
  $pdo = get_pdo();
  // ---------------------------
  // Helpers
  // ---------------------------
  // Normaliza "R05" → 5 ó "5" → 5
  $rackParam = isset($_GET['rack']) ? (string)$_GET['rack'] : '';
  $rackNum = 0;
  if ($rackParam !== '') {
    if (preg_match('/^\s*[Rr]\s*0*([0-9]+)\s*$/', $rackParam, $m)) {
      $rackNum = (int)$m[1];
    } else {
      $rackNum = (int)$rackParam;
    }
  }

  $depositoId = isset($_GET['deposito_id']) ? (int)$_GET['deposito_id'] : 0;

  // Color de “pickeado” (si hay más de 1 pallet en la misma posición).
  // Si tenés un token CSS, podés cambiar por "var(--sol-pick-color)" y
  // en el front interpretarlo. Para mantener la compatibilidad,
  // devolvemos un HEX por defecto (Bootstrap purple-ish).
  $PICK_COLOR = '#6f42c1';

  // Hash → color estable para productos (fallback)
  $hashColor = function (int $id): string {
    // hue estable (id * 57 mod 360), s/l fijos para buen contraste
    $h = ($id * 57) % 360;
    $s = 70;
    $l = 45;
    // Conversión simple HSL→RGB→HEX
    $c = (1 - abs(2 * $l / 100 - 1)) * ($s / 100);
    $x = $c * (1 - abs(fmod(($h / 60), 2) - 1));
    $m = $l / 100 - $c / 2;
    $r = $g = $b = 0;
    if ($h < 60) {
      $r = $c;
      $g = $x;
      $b = 0;
    } elseif ($h < 120) {
      $r = $x;
      $g = $c;
      $b = 0;
    } elseif ($h < 180) {
      $r = 0;
      $g = $c;
      $b = $x;
    } elseif ($h < 240) {
      $r = 0;
      $g = $x;
      $b = $c;
    } elseif ($h < 300) {
      $r = $x;
      $g = 0;
      $b = $c;
    } else {
      $r = $c;
      $g = 0;
      $b = $x;
    }
    $R = (int)round(($r + $m) * 255);
    $G = (int)round(($g + $m) * 255);
    $B = (int)round(($b + $m) * 255);
    return sprintf("#%02X%02X%02X", max(0, min(255, $R)), max(0, min(255, $G)), max(0, min(255, $B)));
  };

  // Intenta leer color del producto desde DB (si existe la columna)
  // Soporta nombres posibles: color_hex, color, color_code
  $productoColors = [];
  $getProductoColor = function (PDO $pdo, int $productoId) use (&$productoColors) {
    if ($productoId <= 0) return null;
    if (array_key_exists($productoId, $productoColors)) {
      return $productoColors[$productoId];
    }
    // Intento dinámico de columnas
    $cols = ['color_hex', 'color', 'color_code'];
    foreach ($cols as $c) {
      try {
        $stmt = $pdo->prepare("SELECT `$c` AS c FROM para_productos WHERE id = ? LIMIT 1");
        $stmt->execute([$productoId]);
        $val = $stmt->fetchColumn();
        if (is_string($val) && strlen(trim($val)) >= 4) {
          $productoColors[$productoId] = trim($val);
          return $productoColors[$productoId];
        }
      } catch (Throwable $e) {
        // columna no existe: probamos la siguiente
      }
    }
    $productoColors[$productoId] = null;
    return null;
  };

  // ---------------------------
  // META: depósitos
  // ---------------------------
  if (isset($_GET['meta']) && $_GET['meta'] === 'depositos') {
    $sql = "SELECT DISTINCT deposito_id AS id
            FROM wh_positions
            WHERE activo = 1
            ORDER BY deposito_id ASC";
    $rows = $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC);
    $out = [];
    foreach ($rows as $r) {
      $id = (int)$r['id'];
      $out[] = [
        'id'   => $id,
        'code' => 'DEP' . $id,
        'name' => null,
      ];
    }
    jexit(['ok' => true, 'depositos' => $out]);
  }

  // ---------------------------
  // META: racks por depósito
  // ---------------------------
  if (isset($_GET['meta']) && $_GET['meta'] === 'racks') {
    if ($depositoId <= 0) {
      jexit(['ok' => false, 'error' => 'Parámetro deposito_id requerido'], 400);
    }
    $stmt = $pdo->prepare("SELECT DISTINCT rack
                           FROM wh_positions
                           WHERE activo = 1 AND deposito_id = ?
                           ORDER BY rack ASC");
    $stmt->execute([$depositoId]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $out = [];
    foreach ($rows as $r) {
      $rk = (int)$r['rack'];
      $out[] = [
        'id'   => $rk,
        'code' => 'R' . str_pad((string)$rk, 2, '0', STR_PAD_LEFT),
      ];
    }
    jexit(['ok' => true, 'racks' => $out]);
  }

  // ---------------------------
  // LAYOUT del rack
  // ---------------------------
  $isLayoutCall = (
    (isset($_GET['layout']) && (int)$_GET['layout'] === 1)
    || (isset($_GET['deposito_id']) && isset($_GET['rack']) && !isset($_GET['meta']))
  );

  if ($isLayoutCall) {
    if ($depositoId <= 0 || $rackNum <= 0) {
      jexit(['ok' => false, 'error' => 'Parámetros deposito_id y rack requeridos'], 400);
    }

    // 1) Traer todas las posiciones del rack
    $stmt = $pdo->prepare("
      SELECT
        p.id            AS position_id,
        p.deposito_id   AS deposito_id,
        p.rack          AS rack_num,
        p.columna       AS col,
        p.nivel         AS niv,
        p.fondo         AS fondo,
        p.tipo_ambiente AS ambiente,
        p.capacidad_pallets AS capacidad
      FROM wh_positions p
      WHERE p.activo = 1
        AND p.deposito_id = :dep
        AND p.rack = :rack
      ORDER BY p.columna ASC, p.nivel ASC, p.fondo ASC
    ");
    $stmt->execute([':dep' => $depositoId, ':rack' => $rackNum]);
    $positions = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (!$positions) {
      jexit(['ok' => true, 'header' => [
        'deposito_id' => $depositoId,
        'deposito_code' => 'DEP' . $depositoId,
        'rack_code' => 'R' . str_pad((string)$rackNum, 2, '0', STR_PAD_LEFT)
      ], 'positions' => []]);
    }

    // 2) Ocupación vigente (hasta IS NULL) y productos en esos pallets
    $ids = array_column($positions, 'position_id');
    // split IN en chunks por seguridad
    $chunks = array_chunk($ids, 900);
    $occByPos = [];       // position_id => ['pallet_count'=>N, 'productos'=>[...]]
    foreach ($chunks as $chunk) {
      $in  = implode(',', array_fill(0, count($chunk), '?'));
      $sql = "
        SELECT
          o.position_id,
          COUNT(*) AS pallet_count,
          GROUP_CONCAT(DISTINCT it.producto_id) AS productos
        FROM wh_position_occupancy o
        JOIN wh_pallets       pa ON pa.id = o.pallet_id
        LEFT JOIN wh_pallet_items it ON it.pallet_id = pa.id
        WHERE o.hasta IS NULL
          AND o.position_id IN ($in)
        GROUP BY o.position_id
      ";
      $st = $pdo->prepare($sql);
      $st->execute($chunk);
      while ($r = $st->fetch(PDO::FETCH_ASSOC)) {
        $productos = [];
        if (!empty($r['productos'])) {
          foreach (explode(',', (string)$r['productos']) as $pid) {
            $pid = (int)$pid;
            if ($pid > 0) $productos[$pid] = true;
          }
        }
        $occByPos[(int)$r['position_id']] = [
          'pallet_count' => (int)$r['pallet_count'],
          'productos'    => array_map('intval', array_keys($productos)),
        ];
      }
    }

    // 3) Construir payload para layout-core/renderers
    $outPositions = [];
    $onlyActive   = isset($_GET['only_active']) ? (int)$_GET['only_active'] : 0;

    $maxCol = 0;
    $maxNiv = 0;
    $maxFondo = 0;

    foreach ($positions as $p) {
      $pid        = (int)$p['position_id'];
      $deposito   = (int)$p['deposito_id'];
      $rackN      = (int)$p['rack_num'];
      $col        = (int)$p['col'];    // alias de columna
      $niv        = (int)$p['niv'];    // alias de nivel
      $fondo      = (int)$p['fondo'];
      $ambiente   = $p['ambiente'];
      $capacidad  = (int)$p['capacidad'];

      $occ        = $occByPos[$pid] ?? ['pallet_count' => 0, 'productos' => []];
      $palletCount = max(0, (int)$occ['pallet_count']);
      $occupied   = $palletCount > 0;
      $picked     = $palletCount > 1;

      if ($onlyActive === 1 && !$occupied) {
        continue; // si piden solo ocupadas, salteamos libres
      }

      // Color final
      $color = null;
      if ($picked) {
        $color = $PICK_COLOR;
      } elseif (!empty($occ['productos'])) {
        sort($occ['productos'], SORT_NUMERIC);
        $firstProd = (int)$occ['productos'][0];
        $dbColor   = $getProductoColor($pdo, $firstProd);
        $color     = $dbColor ?: $hashColor($firstProd);
      }

      $code     = sprintf('C%02d-N%02d-F%d', $col, $niv, $fondo);
      $codeFull = sprintf('DEP%d-R%02d-C%02d-N%02d-F%d', $deposito, $rackN, $col, $niv, $fondo);

      $maxCol   = max($maxCol, $col);
      $maxNiv   = max($maxNiv, $niv);
      $maxFondo = max($maxFondo, $fondo);

      $outPositions[] = [
        // IDs / claves
        'id'            => $pid,
        'fondo_id'      => $pid,           // varios renderers usan data-fondo-id
        'position_id'   => $pid,

        // Ubicación (nombres oficiales + alias)
        'deposito_id'   => $deposito,
        'rack'          => $rackN,
        'columna'       => $col,
        'nivel'         => $niv,
        'fondo'         => $fondo,
        'col'           => $col,           // alias
        'niv'           => $niv,           // alias
        'x'             => $col,           // alias
        'y'             => $niv,           // alias
        'z'             => $fondo,         // alias

        // Códigos
        'code'          => $code,
        'code_full'     => $codeFull,
        'pos_code'      => $code,          // alias
        'pos_code_full' => $codeFull,      // alias

        // Atributos físicos
        'ambiente'      => $ambiente,
        'capacidad'     => $capacidad,

        // Ocupación
        'occupied'      => $occupied,
        'picked'        => $picked,
        'pallet_count'  => $palletCount,
        'productos'     => $occ['productos'],   // ids

        // Pintado
        'color'         => $color,

        // Tooltip
        'title'         => $occupied
          ? ($picked ? 'Pickeado (≥2 pallets)' : 'Ocupado (1 pallet)')
          : 'Libre',
      ];
    }

    // Leyenda sugerida
    $legend = [
      ['label' => 'Libre',                 'color' => '#dee2e6'],
      ['label' => 'Pickeado (≥2 pallets)', 'color' => $PICK_COLOR],
    ];

    $maxCol = 0;
    $maxNiv = 0;
    $maxFondo = 0;
    foreach ($outPositions as $c) {
      $maxCol   = max($maxCol,   (int)$c['columna']);
      $maxNiv   = max($maxNiv,   (int)$c['nivel']);
      $maxFondo = max($maxFondo, (int)$c['fondo']);
    }

    // algunos cores miran estos campos
    $summary = [
      'total_positions' => count($outPositions),
      'max_columna'     => $maxCol,
      'max_nivel'       => $maxNiv,
      'max_fondo'       => $maxFondo,
    ];

    $bounds = ['cols' => $maxCol, 'niveles' => $maxNiv, 'fondos' => $maxFondo];

    $payload = [
      'ok'        => true,
      'status'    => 'ok',
      'header'    => [
        'deposito_id'   => $depositoId,
        'deposito_code' => 'DEP' . $depositoId,
        'rack_code'     => 'R' . str_pad((string)$rackNum, 2, '0', STR_PAD_LEFT),
      ],
      'positions' => array_values($outPositions),
      'cells'     => array_values($outPositions),
      'list'      => array_values($outPositions),
      'data'      => ['positions' => array_values($outPositions)],
      'legend'    => $legend,
      'summary'   => $summary,
      'bounds'    => $bounds,
      'view'      => (string)($_GET['view'] ?? 'microtiles'),
    ];

    jexit($payload);
  }
  // Si no coincide con nada:
  jexit(['ok' => false, 'error' => 'Parámetros inválidos'], 400);
} catch (Throwable $e) {
  jexit(['ok' => false, 'error' => $e->getMessage()], 500);
}
