<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';

try {
  if (session_status() === PHP_SESSION_NONE) session_start();
  if (empty($_SESSION['usuario_id'])) {
    http_response_code(401);
    echo json_encode(['ok'=>false,'error'=>'No autorizado']); exit;
  }

  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $hasTbl = function(PDO $pdo, string $t): bool {
    static $cache = [];
    if (array_key_exists($t,$cache)) return $cache[$t];
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME=?");
    $st->execute([$t]);
    return $cache[$t] = ((int)$st->fetchColumn() > 0);
  };
  $usePlural = $hasTbl($pdo,'wh_moves');
  $useSing   = !$usePlural && $hasTbl($pdo,'wh_move');

  // Parámetros
  $dias = max(1, (int)($_GET['dias'] ?? 60)); // ventana de análisis
  $top  = max(1, (int)($_GET['top']  ?? 8));  // cantidad de alertas
  $cliente_id   = trim((string)($_GET['cliente_id'] ?? ''));
  $operativa_id = trim((string)($_GET['operativa_id'] ?? ''));

  $tz = new DateTimeZone((string)env('TIMEZONE','UTC'));
  $today = (new DateTime('now', $tz))->format('Y-m-d');
  $desde = (new DateTime("-{$dias} days", $tz))->format('Y-m-d');

  // Filtros dinámicos por producto
  $where = [];
  $params = [
    ':desde' => $desde,
    ':hoy'   => $today,
  ];
  if ($cliente_id !== '')   { $where[] = "p.cliente_id=:cliente";   $params[':cliente']=$cliente_id; }
  if ($operativa_id !== '') { $where[] = "p.operativa_id=:operativa"; $params[':operativa']=$operativa_id; }
  $whereSql = $where ? ('WHERE '.implode(' AND ', $where)) : '';

  // CTEs: OUT en la ventana, último OUT global y stock actual
  if ($usePlural) {
    $sql = "
      WITH out_win AS (
        SELECT p.id producto_id,
               SUM(COALESCE(mi.uc_unidades, mi.uv_cajas, 0)) AS out_uc
        FROM wh_moves m
        JOIN wh_move_items mi   ON mi.move_id = m.id
        JOIN wh_pallet_items pi ON pi.id = mi.pallet_item_id
        JOIN para_productos p   ON p.id = pi.producto_id
        " . ($whereSql ? $whereSql . " AND " : "WHERE ") . "
        m.move_type='OUTBOUND'
        AND DATE(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at)) BETWEEN :desde AND :hoy
        GROUP BY p.id
      ),
      last_out AS (
        SELECT p.id producto_id,
               MAX(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at)) AS last_out_at
        FROM wh_moves m
        JOIN wh_move_items mi   ON mi.move_id = m.id
        JOIN wh_pallet_items pi ON pi.id = mi.pallet_item_id
        JOIN para_productos p   ON p.id = pi.producto_id
        " . ($whereSql ? $whereSql . " AND " : "WHERE ") . " m.move_type='OUTBOUND'
        GROUP BY p.id
      ),
      stock_now AS (
        SELECT p.id producto_id,
               SUM(
                 COALESCE(
                   pi.uc_total_cache,
                   (COALESCE(pi.uc_por_caja,0)*COALESCE(pi.uv_cajas,0)+COALESCE(pi.uc_sueltas,0))
                 )
               ) AS stock_uc
        FROM wh_pallet_items pi
        JOIN para_productos p ON p.id = pi.producto_id
        " . ($whereSql ? $whereSql : "") . "
        GROUP BY p.id
      )
      SELECT
        p.id AS producto_id, p.sku, p.denominacion,
        COALESCE(sn.stock_uc,0) AS stock_uc,
        COALESCE(ow.out_uc,0)   AS out_en_ventana,
        DATE_FORMAT(lo.last_out_at, '%Y-%m-%d') AS ultimo_out,
        CASE WHEN lo.last_out_at IS NULL THEN NULL
             ELSE DATEDIFF(:hoy, DATE(lo.last_out_at)) END AS dias_sin_out
      FROM para_productos p
      LEFT JOIN out_win  ow ON ow.producto_id = p.id
      LEFT JOIN last_out lo ON lo.producto_id = p.id
      LEFT JOIN stock_now sn ON sn.producto_id = p.id
      $whereSql
      WHERE COALESCE(sn.stock_uc,0) > 0
        AND COALESCE(ow.out_uc,0) <= 5  
      ORDER BY COALESCE(ow.out_uc,0) ASC,
               CASE WHEN lo.last_out_at IS NULL THEN 1 ELSE 0 END DESC,
               dias_sin_out DESC,
               p.sku ASC
      LIMIT $top
    ";
  } elseif ($useSing) {
    $sql = "
      WITH out_win AS (
        SELECT p.id producto_id,
               SUM(CASE WHEN m.tipo='OUT' THEN COALESCE(m.delta_uc, m.delta_uv, 0) ELSE 0 END) AS out_uc
        FROM wh_move m
        JOIN para_productos p ON p.id = m.producto_id
        " . ($whereSql ? $whereSql . " AND " : "WHERE ") . "
        m.tipo='OUT'
        AND DATE(m.created_at) BETWEEN :desde AND :hoy
        GROUP BY p.id
      ),
      last_out AS (
        SELECT p.id producto_id,
               MAX(m.created_at) AS last_out_at
        FROM wh_move m
        JOIN para_productos p ON p.id = m.producto_id
        " . ($whereSql ? $whereSql . " AND " : "WHERE ") . " m.tipo='OUT'
        GROUP BY p.id
      ),
      stock_now AS (
        SELECT p.id producto_id,
               SUM(
                 COALESCE(
                   pi.uc_total_cache,
                   (COALESCE(pi.uc_por_caja,0)*COALESCE(pi.uv_cajas,0)+COALESCE(pi.uc_sueltas,0))
                 )
               ) AS stock_uc
        FROM wh_pallet_items pi
        JOIN para_productos p ON p.id = pi.producto_id
        " . ($whereSql ? $whereSql : "") . "
        GROUP BY p.id
      )
      SELECT
        p.id AS producto_id, p.sku, p.denominacion,
        COALESCE(sn.stock_uc,0) AS stock_uc,
        COALESCE(ow.out_uc,0)   AS out_en_ventana,
        DATE_FORMAT(lo.last_out_at, '%Y-%m-%d') AS ultimo_out,
        CASE WHEN lo.last_out_at IS NULL THEN NULL
             ELSE DATEDIFF(:hoy, DATE(lo.last_out_at)) END AS dias_sin_out
      FROM para_productos p
      LEFT JOIN out_win  ow ON ow.producto_id = p.id
      LEFT JOIN last_out lo ON lo.producto_id = p.id
      LEFT JOIN stock_now sn ON sn.producto_id = p.id
      $whereSql
      WHERE COALESCE(sn.stock_uc,0) > 0
        AND COALESCE(ow.out_uc,0) <= 5  
      ORDER BY COALESCE(ow.out_uc,0) ASC,
               CASE WHEN lo.last_out_at IS NULL THEN 1 ELSE 0 END DESC,
               dias_sin_out DESC,
               p.sku ASC
      LIMIT $top
    ";
  } else {
    throw new RuntimeException('No existe tabla de movimientos');
  }

  $st = $pdo->prepare($sql);
  foreach ($params as $k=>$v) $st->bindValue($k, $v);
  $st->execute();
  $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

  echo json_encode(['ok'=>true,'data'=>$rows,'meta'=>['dias'=>$dias,'top'=>$top]], JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['ok'=>false,'error'=>'Error del servidor','message'=>$e->getMessage()]);
}
