<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';

if (empty($_SESSION['usuario_id'])) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'No autorizado']); exit;
}

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  // -------- Filtros --------
  $dias          = max(1, (int)($_GET['dias'] ?? 90)); // N de días sin OUT
  $cliente_id    = trim((string)($_GET['cliente_id'] ?? ''));
  $operativa_id  = trim((string)($_GET['operativa_id'] ?? ''));
  $q             = trim((string)($_GET['q'] ?? ''));
  $incluir_sin_stock = (($_GET['incluir_sin_stock'] ?? 'no') === 'si');

  // Fechas de referencia
  $tz = new DateTimeZone((string)env('TIMEZONE','UTC'));
  $today = (new DateTime('now', $tz))->format('Y-m-d');
  $desde_ref = (new DateTime("-{$dias} days", $tz))->format('Y-m-d');

  // WHERE dinámico por producto
  $where = [];
  $params = [
    ':today' => $today,
    ':desde_ref' => $desde_ref,
    ':dias'  => $dias,
  ];
  if ($cliente_id !== '')   { $where[] = "p.cliente_id = :cliente";   $params[':cliente'] = $cliente_id; }
  if ($operativa_id !== '') { $where[] = "p.operativa_id = :operativa"; $params[':operativa'] = $operativa_id; }
  if ($q !== '')            { $where[] = "(p.sku LIKE :q OR p.denominacion LIKE :q)"; $params[':q'] = "%{$q}%"; }
  $whereSql = $where ? ("WHERE " . implode(' AND ', $where)) : "";

  // CTEs:
  // - out_all: último OUT global por producto
  // - out_period: OUT totales en los últimos N días
  // - stock_now: stock actual en unidades por producto
  $sql = "
    WITH out_all AS (
      SELECT
        p.id AS producto_id,
        MAX(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at)) AS last_out_at
      FROM wh_moves m
      JOIN wh_move_items mi   ON mi.move_id = m.id
      JOIN wh_pallet_items pi ON pi.id = mi.pallet_item_id
      JOIN para_productos p   ON p.id = pi.producto_id
      WHERE m.move_type = 'OUTBOUND'
      GROUP BY p.id
    ),
    out_period AS (
      SELECT
        p.id AS producto_id,
        SUM(COALESCE(mi.uc_unidades, mi.uv_cajas, 0)) AS out_uc
      FROM wh_moves m
      JOIN wh_move_items mi   ON mi.move_id = m.id
      JOIN wh_pallet_items pi ON pi.id = mi.pallet_item_id
      JOIN para_productos p   ON p.id = pi.producto_id
      WHERE m.move_type = 'OUTBOUND'
        AND DATE(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at)) >= :desde_ref
      GROUP BY p.id
    ),
    stock_now AS (
      SELECT
        p.id AS producto_id,
        SUM(
          COALESCE(
            pi.uc_total_cache,
            (COALESCE(pi.uc_por_caja,0)*COALESCE(pi.uv_cajas,0)+COALESCE(pi.uc_sueltas,0))
          )
        ) AS stock_uc
      FROM wh_pallet_items pi
      JOIN para_productos p ON p.id = pi.producto_id
      GROUP BY p.id
    )
    SELECT
      cli.razon_social    AS cliente,
      opv.nombre          AS operativa,
      p.id                AS producto_id,
      p.sku,
      p.denominacion,
      sn.stock_uc,
      DATE_FORMAT(oa.last_out_at, '%Y-%m-%d %H:%i') AS ultimo_out,
      CASE
        WHEN oa.last_out_at IS NULL THEN NULL
        ELSE DATEDIFF(:today, DATE(oa.last_out_at))
      END AS dias_sin_out,
      COALESCE(op.out_uc, 0) AS out_en_periodo
    FROM para_productos p
    LEFT JOIN para_clientes cli ON cli.id = p.cliente_id
    LEFT JOIN sys_operativas opv ON opv.id = p.operativa_id
    LEFT JOIN out_all    oa ON oa.producto_id = p.id
    LEFT JOIN out_period op ON op.producto_id = p.id
    LEFT JOIN stock_now  sn ON sn.producto_id = p.id
    $whereSql
    -- Condición de INMOVILIZADO:
    -- 1) Nunca tuvo OUT (oa.last_out_at IS NULL)  OR
    -- 2) Su último OUT fue hace más de :dias días (dias_sin_out > :dias)
    WHERE (
      oa.last_out_at IS NULL
      OR DATEDIFF(:today, DATE(oa.last_out_at)) > :dias
    )
    " . ($incluir_sin_stock ? "" : " AND COALESCE(sn.stock_uc,0) > 0 ") . "
    ORDER BY
      CASE WHEN oa.last_out_at IS NULL THEN 1 ELSE 0 END DESC,
      dias_sin_out DESC,
      p.sku ASC
  ";

  $st = $pdo->prepare($sql);
  foreach ($params as $k=>$v) $st->bindValue($k, $v);
  $st->execute();
  $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

  echo json_encode(['ok' => true, 'data' => $rows], JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['ok' => false, 'error' => 'Error cargando reporte', 'message' => $e->getMessage()]);
}
