<?php
declare(strict_types=1);
// api/deposito/layout.php
// Returns JSON array of positions for a given deposito_id and optional ambiente_id

// api is one level under project root
$ROOT = dirname(__DIR__); // .../api
$BASE = dirname($ROOT); // project root
require_once $BASE . '/config/db.php';
header('Content-Type: application/json; charset=utf-8');

$deposito_id = isset($_GET['deposito_id']) ? (int)$_GET['deposito_id'] : null;
$ambiente_id = isset($_GET['ambiente_id']) ? (int)$_GET['ambiente_id'] : null;

try {
  $pdo = getPDO();
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['error' => 'db_connection', 'msg' => $e->getMessage()]);
  exit;
}

// gather safe columns
$colsToTry = ['id','deposito_id','rack','columna','nivel','fondo','code','code_full','pos_code','pos_code_full','capacidad_pallets','ambiente_id','activo','created_at','title'];
$existing = [];
foreach ($colsToTry as $c) {
  $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'wh_posicion' AND COLUMN_NAME = ?");
  $st->execute([$c]);
  if ((int)$st->fetchColumn() > 0) $existing[] = $c;
}
if (empty($existing)) {
  http_response_code(500);
  echo json_encode(['error' => 'no_columns', 'msg' => 'wh_posicion tiene columnas incompatibles o no existe']);
  exit;
}
$colSqlBase = implode(', ', array_map(function($c){ return "`$c`"; }, $existing));

// Add a derived column with current pallets count in position so UI can mark occupied
// Prefer COUNT(DISTINCT pallet_id) if column exists in wh_stock, else COUNT(*)
$stockHasPalletId = false;
try {
  $stChk = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'wh_stock' AND COLUMN_NAME = 'pallet_id'");
  $stChk->execute();
  $stockHasPalletId = ((int)$stChk->fetchColumn() > 0);
} catch (Throwable $e) {
  $stockHasPalletId = false;
}
$countExpr = $stockHasPalletId ? 'COUNT(DISTINCT s.pallet_id)' : 'COUNT(*)';

// Try to expose producto color_web for multi-pallet styling
$productCol = null; // column name in wh_stock
try {
  foreach (['producto_id','product_id'] as $cand) {
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'wh_stock' AND COLUMN_NAME = ?");
    $st->execute([$cand]);
    if ((int)$st->fetchColumn() > 0) { $productCol = $cand; break; }
  }
} catch (Throwable $e) { $productCol = null; }

// Detect product reference table that contains color_web
$productTable = null;
try {
  // Prefer 'ara_productos'
  $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'ara_productos' AND COLUMN_NAME = 'id'");
  $st->execute();
  $araHasId = ((int)$st->fetchColumn() > 0);
  $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'ara_productos' AND COLUMN_NAME = 'color_web'");
  $st->execute();
  $araHasColor = ((int)$st->fetchColumn() > 0);
  if ($araHasId && $araHasColor) { $productTable = 'ara_productos'; }
  else {
    // Fallback 'para_productos'
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'para_productos' AND COLUMN_NAME = 'id'");
    $st->execute();
    $ppHasId = ((int)$st->fetchColumn() > 0);
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'para_productos' AND COLUMN_NAME = 'color_web'");
    $st->execute();
    $ppHasColor = ((int)$st->fetchColumn() > 0);
    if ($ppHasId && $ppHasColor) { $productTable = 'para_productos'; }
  }
} catch (Throwable $e) { $productTable = null; }

$colorExpr = null;
if ($productCol && $productTable) {
  // dominant color by product count for the position
  $colorExpr = "(SELECT pp.color_web FROM wh_stock s JOIN `$productTable` pp ON pp.id = s.`$productCol` WHERE s.posicion_id = wh_posicion.id AND s.qty_uv > 0 AND pp.color_web IS NOT NULL AND pp.color_web <> '' GROUP BY pp.color_web ORDER BY COUNT(*) DESC LIMIT 1) AS producto_color_web";
}

$colSql = $colSqlBase
  . ", (SELECT $countExpr FROM wh_stock s WHERE s.posicion_id = wh_posicion.id AND s.qty_uv > 0) AS pallets_actuales"
  . ($colorExpr ? ", $colorExpr" : '');

// If deposito_id provided, get deposito orientation
$depositoOrient = null;
if ($deposito_id) {
  try {
    $qDep = $pdo->prepare("SELECT orientacion FROM wh_deposito WHERE id = ? LIMIT 1");
    $qDep->execute([$deposito_id]);
    $depositoOrient = $qDep->fetchColumn();
    if (is_string($depositoOrient)) {
      $depositoOrient = strtoupper(trim($depositoOrient));
    } else {
      $depositoOrient = null;
    }
  } catch (Throwable $e) {
    // ignore orientation on error; API should still return positions
    $depositoOrient = null;
  }
}

try {
  if ($deposito_id) {
    if ($ambiente_id) {
      $q = $pdo->prepare("SELECT $colSql FROM wh_posicion WHERE deposito_id = ? AND ambiente_id = ? ORDER BY id");
      $q->execute([$deposito_id, $ambiente_id]);
    } else {
      $q = $pdo->prepare("SELECT $colSql FROM wh_posicion WHERE deposito_id = ? ORDER BY id");
      $q->execute([$deposito_id]);
    }
  } else {
    $q = $pdo->prepare("SELECT $colSql FROM wh_posicion ORDER BY id LIMIT 1000");
    $q->execute();
  }
  $rows = $q->fetchAll(PDO::FETCH_ASSOC);
  if ($depositoOrient) {
    // Attach deposito orientation to each row (without changing response shape)
    foreach ($rows as &$r) {
      $r['deposito_orientacion'] = $depositoOrient;
    }
    unset($r);
  }
  echo json_encode($rows, JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['error' => 'query', 'msg' => $e->getMessage()]);
}
