<?php
declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/db.php';
if (is_file($ROOT . '/app/Support/Auth.php')) require_once $ROOT . '/app/Support/Auth.php';
session_start();

try { $pdo = getPDO(); $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION); }
catch (Throwable $e) { http_response_code(500); echo json_encode(['ok'=>false,'error'=>'db']); exit; }

$payload = json_decode(file_get_contents('php://input'), true);
if (!$payload) { http_response_code(400); echo json_encode(['ok'=>false,'error'=>'JSON inválido']); exit; }

// Esperamos { items: [ { deposito_from, deposito_to, from_pos_id, to_pos_id, pallets:[{id,producto_id,lote_id,uv,uc}] } ] }
$items = is_array($payload['items'] ?? null) ? $payload['items'] : [];
if (!$items) { http_response_code(422); echo json_encode(['ok'=>false,'error'=>'items vacío']); exit; }

// Intentar detectar columnas disponibles de wh_move_buffer para un insert flexible
function tableColumns(PDO $pdo, string $table): array {
  try {
    $cols = [];
    $st = $pdo->query("SHOW COLUMNS FROM `" . str_replace('`','',$table) . "`");
    foreach ($st->fetchAll(PDO::FETCH_ASSOC) as $r) { $cols[] = $r['Field']; }
    return $cols;
  } catch (Throwable $e) { return []; }
}

$cols = tableColumns($pdo, 'wh_move_buffer');
if (!$cols) { http_response_code(500); echo json_encode(['ok'=>false,'error'=>'Tabla wh_move_buffer no disponible']); exit; }

// Campos deseados en orden de prioridad
$desired = [
  'deposito_id_from','deposito_id_to',
  'from_pos_id','to_pos_id','pallet_id','producto_id','lote_id',
  'delta_uv','delta_uc',
  'estado','referencia','user_id','created_at'
];

// Usaremos los que existan realmente
$use = array_values(array_intersect($desired, $cols));
if (!in_array('deposito_id_from', $use, true) || !in_array('from_pos_id', $use, true) || !in_array('pallet_id', $use, true)) {
  // mínimos razonables para un buffer
  http_response_code(500);
  echo json_encode(['ok'=>false,'error'=>'wh_move_buffer carece de columnas mínimas']); exit;
}

// Armar SQL dinámico
$fieldList = '`' . implode('`,`', $use) . '`';
$paramList = ':' . implode(',:', $use);
$sql = "INSERT INTO wh_move_buffer ($fieldList) VALUES ($paramList)";
$st = $pdo->prepare($sql);

// user id si está disponible
$uid = null;
if (function_exists('current_user_id')) { $uid = current_user_id(); }
if (!$uid && isset($_SESSION['user_id'])) { $uid = (int)$_SESSION['user_id']; }
if (!$uid && isset($_SESSION['usuario_id'])) { $uid = (int)$_SESSION['usuario_id']; }

try {
  $pdo->beginTransaction();
  $inserted = 0;

  foreach ($items as $it) {
    $depFrom = (int)($it['deposito_from'] ?? $it['deposito_id_from'] ?? 0);
    $depTo   = (int)($it['deposito_to']   ?? $it['deposito_id_to']   ?? 0);
    $fromPos = (int)($it['from_pos_id'] ?? 0);
    $toPos   = (int)($it['to_pos_id']   ?? 0);
    $pallets = is_array($it['pallets'] ?? null) ? $it['pallets'] : [];

    foreach ($pallets as $p) {
      $row = [
        'deposito_id_from' => $depFrom,
        'deposito_id_to'   => $depTo,
        'from_pos_id'      => $fromPos,
        'to_pos_id'        => $toPos,
        'pallet_id'        => (int)($p['id'] ?? 0),
        'producto_id'      => !empty($p['producto_id']) ? (int)$p['producto_id'] : null,
        'lote_id'          => !empty($p['lote_id']) ? (int)$p['lote_id'] : null,
        'delta_uv'         => isset($p['uv']) ? (int)$p['uv'] : 0,
        'delta_uc'         => isset($p['uc']) ? (int)$p['uc'] : 0,
        'estado'           => 'PENDIENTE',
        'referencia'       => 'plan-rotacion',
        'user_id'          => $uid ?: null,
        'created_at'       => date('Y-m-d H:i:s'),
      ];

      // Filtrar sólo columnas realmente existentes
      $params = [];
      foreach ($use as $c) { $params[":$c"] = $row[$c] ?? null; }
      $st->execute($params);
      $inserted++;
    }
  }

  $pdo->commit();
  echo json_encode(['ok'=>true,'inserted'=>$inserted]);
} catch (Throwable $e) {
  $pdo->rollBack();
  http_response_code(500);
  echo json_encode(['ok'=>false,'error'=>'tx','debug'=>$e->getMessage()]);
}
