<?php
declare(strict_types=1);
// api/deposito/posicion_pallets.php
// Returns detailed pallet info and stock items for a given posicion_id

$ROOT = dirname(__DIR__); // .../api
$BASE = dirname($ROOT); // project root
require_once $BASE . '/config/db.php';
header('Content-Type: application/json; charset=utf-8');

$posicion_id = isset($_GET['posicion_id']) ? (int) $_GET['posicion_id'] : 0;
if ($posicion_id <= 0) {
  http_response_code(400);
  echo json_encode(['error' => 'bad_request', 'msg' => 'posicion_id requerido']);
  exit;
}

try {
  $pdo = getPDO();
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['error' => 'db_connection', 'msg' => $e->getMessage()]);
  exit;
}

$warnings = [];

// Check stock columns
$stockCols = [];
$stockColsTry = [
  'id',
  'pallet_id',
  'posicion_id',
  'producto_id',
  'product_id',
  'lote_id',
  'lote',
  'lote_uc',
  'fecha_venc',
  'fecha_vto',
  'fecha_prod',
  'fecha_produccion',
  'fec_produccion',
  'fecha_elab',
  'fecha_elaboracion',
  'fecha_fab',
  'fecha_fabricacion',
  'qty_uv',
  'qty_uc',
  'created_at',
  'updated_at'
];
foreach ($stockColsTry as $c) {
  $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'wh_stock' AND COLUMN_NAME = ?");
  $st->execute([$c]);
  if ((int) $st->fetchColumn() > 0)
    $stockCols[] = $c;
}
if (empty($stockCols)) {
  echo json_encode(['error' => 'no_stock_cols', 'msg' => 'wh_stock sin columnas compatibles']);
  exit;
}

$hasPalletId = in_array('pallet_id', $stockCols, true);
$productCol = null;
foreach (['producto_id', 'product_id'] as $cand) {
  if (in_array($cand, $stockCols, true)) {
    $productCol = $cand;
    break;
  }
}

// Detect product table: prefer ara_productos, fallback para_productos
$productTable = null;
$prodIdCol = 'id';
try {
  $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'ara_productos' AND COLUMN_NAME IN ('id','color_web')");
  $st->execute();
  $cnt = (int) $st->fetchColumn();
  if ($cnt >= 1) { // single COUNT(*) for IN is not exact; validate columns individually
    $q1 = $pdo->query("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'ara_productos' AND COLUMN_NAME = 'id'");
    $q2 = $pdo->query("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'ara_productos' AND COLUMN_NAME = 'color_web'");
    if ((int) $q1->fetchColumn() > 0 && (int) $q2->fetchColumn() > 0)
      $productTable = 'ara_productos';
  }
  if (!$productTable) {
    $q1 = $pdo->query("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'para_productos' AND COLUMN_NAME = 'id'");
    $q2 = $pdo->query("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'para_productos' AND COLUMN_NAME = 'color_web'");
    if ((int) $q1->fetchColumn() > 0 && (int) $q2->fetchColumn() > 0)
      $productTable = 'para_productos';
  }
} catch (Throwable $e) { /* ignore */
}

// Detect wh_pallet
$palletCols = [];
$palletColsTry = [
  'id',
  'codigo',
  'code',
  'estado_id',
  'estado',
  'estado_code',
  'created_at',
  'updated_at',
  'peso_kg',
  'volumen_m3',
  'cliente_id',
  'ingreso_id',
  'posicion_id',
  'deposito_id',
  'lote',
  'propietario_id'
];
$hasWhPallet = false;
try {
  $st = $pdo->query("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'wh_pallet'");
  $hasWhPallet = ((int) $st->fetchColumn() > 0);
  if ($hasWhPallet) {
    foreach ($palletColsTry as $c) {
      $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'wh_pallet' AND COLUMN_NAME = ?");
      $st->execute([$c]);
      if ((int) $st->fetchColumn() > 0)
        $palletCols[] = $c;
    }
  }
} catch (Throwable $e) {
  $hasWhPallet = false;
}

// Build base list of pallet_ids present in this position
$palletIds = [];
if ($hasPalletId) {
  $q = $pdo->prepare("SELECT DISTINCT pallet_id FROM wh_stock WHERE posicion_id = ? AND qty_uv > 0 AND pallet_id IS NOT NULL");
  $q->execute([$posicion_id]);
  $palletIds = array_map('intval', $q->fetchAll(PDO::FETCH_COLUMN));
}

$result = ['posicion_id' => $posicion_id, 'pallets' => [], 'warnings' => $warnings];

if ($hasPalletId && $hasWhPallet && !empty($palletIds)) {
  // Fetch pallets data
  $palColsSql = '`' . implode('`,`', $palletCols) . '`';
  $in = implode(',', array_fill(0, count($palletIds), '?'));
  $qp = $pdo->prepare("SELECT $palColsSql FROM wh_pallet WHERE id IN ($in)");
  $qp->execute($palletIds);
  $palData = [];
  while ($row = $qp->fetch(PDO::FETCH_ASSOC)) {
    $palData[(int) $row['id']] = $row;
  }

  // Enrich pallet: estado and cliente
  // estado
  $estadoMap = [];
  try {
    $hasTbl = (int) $pdo->query("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'wh_pallet_estado'")->fetchColumn() > 0;
    if ($hasTbl) {
      $cols = [];
      foreach (['id', 'code', 'nombre', 'name'] as $c) {
        $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'wh_pallet_estado' AND COLUMN_NAME = ?");
        $st->execute([$c]);
        if ((int) $st->fetchColumn() > 0)
          $cols[] = $c;
      }
      if (in_array('id', $cols, true)) {
        $sel = '`' . implode('`,`', $cols) . '`';
        $rowsE = $pdo->query("SELECT $sel FROM wh_pallet_estado")->fetchAll(PDO::FETCH_ASSOC);
        foreach ($rowsE as $er) {
          $estadoMap[(int) $er['id']] = $er;
        }
      }
    }
  } catch (Throwable $e) { /* ignore */
  }
  // cliente
  $clienteMap = [];
  try {
    $hasTbl = (int) $pdo->query("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'para_clientes'")->fetchColumn() > 0;
    if ($hasTbl) {
      $cols = [];
      foreach (['id', 'razon_social', 'nombre', 'name'] as $c) {
        $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'para_clientes' AND COLUMN_NAME = ?");
        $st->execute([$c]);
        if ((int) $st->fetchColumn() > 0)
          $cols[] = $c;
      }
      if (in_array('id', $cols, true)) {
        $sel = '`' . implode('`,`', $cols) . '`';
        // optimize by selecting only used clients
        $cliIds = [];
        foreach ($palData as $pd) {
          if (!empty($pd['cliente_id']))
            $cliIds[(int) $pd['cliente_id']] = true;
        }
        if (!empty($cliIds)) {
          $inCli = implode(',', array_fill(0, count($cliIds), '?'));
          $stc = $pdo->prepare("SELECT $sel FROM para_clientes WHERE id IN ($inCli)");
          $stc->execute(array_keys($cliIds));
          while ($cr = $stc->fetch(PDO::FETCH_ASSOC)) {
            $clienteMap[(int) $cr['id']] = $cr;
          }
        }
      }
    }
  } catch (Throwable $e) { /* ignore */
  }
  // attach to pallet rows
  foreach ($palData as &$pd) {
    if (!empty($pd['estado_id']) && isset($estadoMap[(int) $pd['estado_id']]))
      $pd['estado'] = $estadoMap[(int) $pd['estado_id']];
    if (!empty($pd['cliente_id']) && isset($clienteMap[(int) $pd['cliente_id']]))
      $pd['cliente'] = $clienteMap[(int) $pd['cliente_id']];
  }
  unset($pd);

  // Fetch stock items grouped by pallet
  $selStockCols = array_diff($stockCols, ['posicion_id']);
  $stkColsSql = '`' . implode('`,`', $selStockCols) . '`';
  $qs = $pdo->prepare("SELECT $stkColsSql FROM wh_stock WHERE posicion_id = ? AND qty_uv > 0 ORDER BY pallet_id");
  $qs->execute([$posicion_id]);
  $itemsByPallet = [];
  while ($r = $qs->fetch(PDO::FETCH_ASSOC)) {
    $itemsByPallet[(int) $r['pallet_id']][] = $r;
  }

  // Enrich with lote info if lote_id exists and wh_lote table is present
  $hasLoteId = in_array('lote_id', $stockCols, true);
  if ($hasLoteId) {
    try {
      $hasTbl = (int) $pdo->query("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'wh_lote'")->fetchColumn() > 0;
      if ($hasTbl) {
        $loteColsTry = ['id', 'code', 'codigo', 'numero', 'fecha_venc', 'fecha_vto', 'fecha_vencimiento', 'fecha_prod', 'fecha_produccion', 'fec_produccion', 'fecha_elab', 'fecha_elaboracion', 'fecha_fab', 'fecha_fabricacion'];
        $loteCols = [];
        foreach ($loteColsTry as $c) {
          $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'wh_lote' AND COLUMN_NAME = ?");
          $st->execute([$c]);
          if ((int) $st->fetchColumn() > 0)
            $loteCols[] = $c;
        }
        if (in_array('id', $loteCols, true)) {
          $selL = '`' . implode('`,`', $loteCols) . '`';
          $ids = [];
          foreach ($itemsByPallet as $plist) {
            foreach ($plist as $it) {
              if (!empty($it['lote_id']))
                $ids[(int) $it['lote_id']] = true;
            }
          }
          if (!empty($ids)) {
            $inL = implode(',', array_fill(0, count($ids), '?'));
            $ql = $pdo->prepare("SELECT $selL FROM wh_lote WHERE id IN ($inL)");
            $ql->execute(array_keys($ids));
            $loteMap = [];
            while ($lr = $ql->fetch(PDO::FETCH_ASSOC)) {
              $loteMap[(int) $lr['id']] = $lr;
            }
            foreach ($itemsByPallet as &$plist) {
              foreach ($plist as &$it) {
                if (!empty($it['lote_id']) && isset($loteMap[(int) $it['lote_id']]))
                  $it['lote_info'] = $loteMap[(int) $it['lote_id']];
              }
              unset($it);
            }
            unset($plist);
          }
        }
      }
    } catch (Throwable $e) { /* ignore */
    }
  }

  // Enrich items with product info
  if ($productTable && $productCol) {
    // collect product ids
    $prodIds = [];
    foreach ($itemsByPallet as $plist) {
      foreach ($plist as $it) {
        if (!empty($it[$productCol]))
          $prodIds[(int) $it[$productCol]] = true;
      }
    }
    if (!empty($prodIds)) {
      // detect existing columns in product table
      $prodColsTry = ['id', 'code', 'codigo', 'nombre', 'nombre_corto', 'sku', 'denominacion', 'color_web'];
      $prodCols = [];
      foreach ($prodColsTry as $c) {
        try {
          $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?");
          $st->execute([$productTable, $c]);
          if ((int) $st->fetchColumn() > 0)
            $prodCols[] = $c;
        } catch (Throwable $e) { /* ignore */
        }
      }
      if (!in_array('id', $prodCols, true))
        $prodCols[] = 'id';
      $prodSel = '`' . implode('`,`', $prodCols) . '`';
      try {
        $in2 = implode(',', array_fill(0, count($prodIds), '?'));
        $qp2 = $pdo->prepare("SELECT $prodSel FROM `$productTable` WHERE id IN ($in2)");
        $qp2->execute(array_keys($prodIds));
        $prodMap = [];
        while ($p = $qp2->fetch(PDO::FETCH_ASSOC)) {
          $prodMap[(int) $p['id']] = $p;
        }
        // attach
        foreach ($itemsByPallet as $pid => &$plist) {
          foreach ($plist as &$it) {
            $pidVal = (int) ($it[$productCol] ?? 0);
            if ($pidVal && isset($prodMap[$pidVal]))
              $it['producto'] = $prodMap[$pidVal];
          }
          unset($it);
        }
        unset($plist);
      } catch (Throwable $e) {
        $warnings[] = 'product_lookup_error';
      }
    }
  }

  // assemble
  foreach ($palletIds as $pid) {
    $itms = $itemsByPallet[$pid] ?? [];
    $totUv = 0;
    $totUc = 0;
    foreach ($itms as $it) {
      $totUv += (float) ($it['qty_uv'] ?? 0);
      $totUc += (float) ($it['qty_uc'] ?? 0);
    }
    $result['pallets'][] = [
      'pallet' => $palData[$pid] ?? ['id' => $pid],
      'items' => $itms,
      'totals' => ['qty_uv' => $totUv, 'qty_uc' => $totUc, 'lines' => count($itms)],
    ];
  }
} else {
  // Fallback: no pallet ids or table — return stock rows grouped by an inferred key
  $warnings[] = 'fallback_stock_rows';
  $selStockCols = array_diff($stockCols, ['posicion_id']);
  $stkColsSql = '`' . implode('`,`', $selStockCols) . '`';
  $qs = $pdo->prepare("SELECT $stkColsSql FROM wh_stock WHERE posicion_id = ? AND qty_uv > 0");
  $qs->execute([$posicion_id]);
  $rows = $qs->fetchAll(PDO::FETCH_ASSOC);
  // attach product info if available
  if ($productTable && $productCol) {
    $prodIds = [];
    foreach ($rows as $r) {
      if (!empty($r[$productCol]))
        $prodIds[(int) $r[$productCol]] = true;
    }
    if (!empty($prodIds)) {
      // detect existing columns
      $prodColsTry = ['id', 'code', 'codigo', 'nombre', 'nombre_corto', 'sku', 'denominacion', 'color_web'];
      $prodCols = [];
      foreach ($prodColsTry as $c) {
        try {
          $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?");
          $st->execute([$productTable, $c]);
          if ((int) $st->fetchColumn() > 0)
            $prodCols[] = $c;
        } catch (Throwable $e) { /* ignore */
        }
      }
      if (!in_array('id', $prodCols, true))
        $prodCols[] = 'id';
      $prodSel = '`' . implode('`,`', $prodCols) . '`';
      try {
        $in2 = implode(',', array_fill(0, count($prodIds), '?'));
        $qp2 = $pdo->prepare("SELECT $prodSel FROM `$productTable` WHERE id IN ($in2)");
        $qp2->execute(array_keys($prodIds));
        $prodMap = [];
        while ($p = $qp2->fetch(PDO::FETCH_ASSOC)) {
          $prodMap[(int) $p['id']] = $p;
        }
        foreach ($rows as &$r) {
          $pidVal = (int) ($r[$productCol] ?? 0);
          if ($pidVal && isset($prodMap[$pidVal]))
            $r['producto'] = $prodMap[$pidVal];
        }
        unset($r);
      } catch (Throwable $e) {
        $warnings[] = 'product_lookup_error';
      }
    }
  }
  $sumUv = 0.0;
  $sumUc = 0.0;
  foreach ($rows as $rr) {
    $sumUv += (float) ($rr['qty_uv'] ?? 0);
    $sumUc += (float) ($rr['qty_uc'] ?? 0);
  }
  $result['pallets'][] = ['pallet' => null, 'items' => $rows, 'totals' => ['qty_uv' => $sumUv, 'qty_uc' => $sumUc, 'lines' => count($rows)]];
  $result['warnings'] = $warnings;
}

echo json_encode($result, JSON_UNESCAPED_UNICODE);
