<?php
declare(strict_types=1);

// api/deposito/rack_layout.php
// Devuelve HTML o JSON del layout de un rack según ?format=json|html (default html)

header('Content-Type: text/html; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2); // /api/deposito -> project root
require_once $ROOT . '/config/db.php';

function h($s): string
{
    return htmlspecialchars((string) $s, ENT_QUOTES, 'UTF-8');
}

$depositoId = isset($_GET['deposito_id']) ? (int) $_GET['deposito_id'] : 0;
$rackId = isset($_GET['rack_id']) ? (int) $_GET['rack_id'] : 0;
$mode = $_GET['mode'] ?? 'microtiles';
$format = strtolower($_GET['format'] ?? 'html'); // html|json

if ($depositoId <= 0 || $rackId < 0) {
    http_response_code(422);
    if ($format === 'json') {
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['ok' => false, 'error' => 'Parámetros insuficientes (deposito_id y rack_id requeridos)']);
    } else {
        echo '<div class="alert alert-warning">Faltan parámetros: depósito y rack son requeridos.</div>';
    }
    exit;
}

try {
    $pdo = getPDO();
} catch (Throwable $e) {
    http_response_code(500);
    if ($format === 'json') {
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['ok' => false, 'error' => 'DB connection failed', 'debug' => $e->getMessage()]);
    } else {
        echo '<div class="alert alert-danger">Error de conexión a la base de datos.</div>';
    }
    exit;
}

// =============================
// 1) Depósito
// =============================
$deposito = null;
try {
    $st = $pdo->prepare("SELECT id, code, nombre FROM wh_deposito WHERE id = :id AND deleted_at IS NULL");
    $st->execute([':id' => $depositoId]);
    $depRow = $st->fetch(PDO::FETCH_ASSOC);
    if ($depRow) {
        $deposito = [
            'id' => (int) $depRow['id'],
            'code' => (string) $depRow['code'],
            'name' => (string) ($depRow['nombre'] ?? $depRow['code'])
        ];
    }
} catch (Throwable $e) {
}

if (!$deposito) {
    http_response_code(404);
    if ($format === 'json') {
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['ok' => false, 'error' => 'Depósito no encontrado']);
    } else {
        echo '<div class="alert alert-warning">Depósito no encontrado.</div>';
    }
    exit;
}

// =============================
// 2) Posiciones del rack
// =============================
$rows = [];
try {
    $sql = "
        SELECT
            wp.id,
            wp.columna,
            wp.nivel,
            wp.fondo,
            wp.code_full,
            wp.pos_code_full,
            wp.ocupado,
            wp.activo,
            wp.picked
        FROM wh_posicion wp
        WHERE wp.deposito_id = :dep
            AND wp.rack = :rack
            AND wp.activo = 1
            AND wp.deleted_at IS NULL
        ORDER BY wp.columna ASC, wp.nivel ASC, wp.fondo ASC
        ";
    $st = $pdo->prepare($sql);
    $st->execute([':dep' => $depositoId, ':rack' => $rackId]);
    $rows = $st->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
}

if (!$rows) {
    if ($format === 'json') {
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['ok' => true, 'positions' => [], 'deposito' => $deposito, 'rack' => null]);
    } else {
        echo '<div class="alert alert-info">No hay posiciones activas para este rack.</div>';
    }
    exit;
}
// Paletas por posición (para colorear ocupadas vs vacías)
$posIds = array_map(fn($r) => (int)$r['id'], $rows);
$colorByPos = [];

// Detectar columna de referencia en wh_pallet (posicion_id preferida)
function detectPalletPosColumn(PDO $pdo): ?string {
        try {
                $st = $pdo->query("SHOW COLUMNS FROM wh_pallet LIKE 'posicion_id'");
                if ($st && $st->fetch(PDO::FETCH_ASSOC)) return 'posicion_id';
        } catch (Throwable $e) {}
        try {
                $st = $pdo->query("SHOW COLUMNS FROM wh_pallet LIKE 'pos_id'");
                if ($st && $st->fetch(PDO::FETCH_ASSOC)) return 'pos_id';
        } catch (Throwable $e) {}
        return null;
}

$palletPosCol = detectPalletPosColumn($pdo);

if ($palletPosCol && $posIds) {
        $in = implode(',', array_fill(0, count($posIds), '?'));
        $sqlColors = "
            SELECT
                p.$palletPosCol   AS pos_id,
                COUNT(*)          AS pallets,
                MAX(p.id)         AS last_pallet_id,
                MAX(p.producto_id)                AS producto_id,
                MAX(pr.nombre)                    AS producto_nombre,
                MAX(pr.color_web)                 AS color_web
            FROM wh_pallet p
            LEFT JOIN para_productos pr ON pr.id = p.producto_id
            WHERE p.deleted_at IS NULL
                AND p.$palletPosCol IN ($in)
            GROUP BY p.$palletPosCol
        ";
        try {
                $st2 = $pdo->prepare($sqlColors);
                $st2->execute($posIds);
                while ($r2 = $st2->fetch(PDO::FETCH_ASSOC)) {
                        $colorByPos[(int)$r2['pos_id']] = [
                                'pallets' => (int) $r2['pallets'],
                                'producto_id' => $r2['producto_id'] ? (int)$r2['producto_id'] : null,
                                'producto_nombre' => (string) ($r2['producto_nombre'] ?? ''),
                                'color_web' => (string) ($r2['color_web'] ?? '')
                        ];
                }
        } catch (Throwable $e) {
                // silencioso: si falla, seguimos sin colores
        }
}

// =============================
// 3) Normalización
// =============================
$positions = [];
foreach ($rows as $r) {
    $colCode = 'C' . str_pad((string) $r['columna'], 2, '0', STR_PAD_LEFT);
    $nivCode = 'N' . str_pad((string) $r['nivel'], 2, '0', STR_PAD_LEFT);
    $depthIdx = (int) $r['fondo'];
    $posId = (int) $r['id'];

    $status = ((int) $r['activo'] === 1) ? 'ACTIVE' : 'INACTIVE';
    if ((int) $r['ocupado'] === 1) {
        $status = 'BLOCKED';
    }

    // color del producto (si hay pallets en esa posición)
    $meta = $colorByPos[$posId] ?? null;
    $color = $meta && !empty($meta['color_web']) ? (string) $meta['color_web'] : null;
    // Sanitizar color (#RRGGBB)
    if ($color && !preg_match('/^#?[0-9A-Fa-f]{6}$/', $color))
        $color = null;
    if ($color && $color[0] !== '#')
        $color = '#' . $color;

    // pallets count: prefer computed meta; fallback to wh_posicion.ocupado flag
    $palletsCount = isset($meta['pallets']) ? (int)$meta['pallets'] : (((int)$r['ocupado'] === 1) ? 1 : 0);

    $positions[] = [
        'id' => $posId,
        'col_code' => $colCode,
        'niv_code' => $nivCode,
        'depth_index' => $depthIdx,
        'code_full' => (string) ($r['code_full'] ?? $r['pos_code_full'] ?? ''),
        'status' => $status,
        'is_pick_face' => (int) ($r['picked'] ?? 0),
        'fondo_id' => $posId,
        'color_hex' => $color,
        'producto_id' => $meta['producto_id'] ?? null,
        'producto_nombre' => $meta['producto_nombre'] ?? '',
        'pallets' => $palletsCount
    ];
}

// =============================
// 4) Rack
// =============================
$rackCode = 'R' . str_pad((string) $rackId, 2, '0', STR_PAD_LEFT);
$rack = [
    'id' => (int) $rackId,
    'code' => $rackCode,
    'name' => 'Rack ' . str_pad((string) $rackId, 2, '0', STR_PAD_LEFT),
];

// =============================
// 5) Respuesta según formato
// =============================
if ($format === 'json') {
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode([
        'ok' => true,
        'deposito' => $deposito,
        'rack' => $rack,
        'positions' => $positions,
        'options' => [
            'mode' => $mode,
            'count' => count($positions)
        ]
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// ---- Render HTML ----
$options = [
    'mode' => $mode ?: 'microtiles',
    'show_toolbar' => true,
    'title' => "Depósito {$deposito['code']} · {$rack['name']}"
];
$view = $ROOT . '/views/layout/_partials/warehouse/rack_layout.php';

if (!is_file($view)) {
    http_response_code(500);
    echo '<div class="alert alert-danger">No se encontró el parcial rack_layout.php.</div>';
    exit;
}

ob_start();
include $view; // recibe $deposito, $rack, $positions, $options
echo ob_get_clean();
