<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';

if (empty($_SESSION['usuario_id'])) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'No autorizado']); exit;
}

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  // Helper para chequear existencia de tabla
  $hasTbl = function(PDO $pdo, string $t): bool {
    static $cache = [];
    if (array_key_exists($t, $cache)) return $cache[$t];
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
    $st->execute([$t]);
    return $cache[$t] = ((int)$st->fetchColumn() > 0);
  };

  $desde      = trim((string)($_GET['desde'] ?? ''));
  $hasta      = trim((string)($_GET['hasta'] ?? ''));
  $cliente_id = trim((string)($_GET['cliente_id'] ?? ''));
  $operativa_id = trim((string)($_GET['operativa_id'] ?? ''));
  $agrupar    = ($_GET['agrupar'] ?? 'grupo'); // grupo|clase

  $where = [];
  $params = [];

  if ($desde !== '') { $where[] = "DATE(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at)) >= :desde"; $params[':desde'] = $desde; }
  if ($hasta !== '') { $where[] = "DATE(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at)) <= :hasta"; $params[':hasta'] = $hasta; }
  if ($cliente_id !== '') { $where[] = "p.cliente_id = :cliente"; $params[':cliente'] = $cliente_id; }
  if ($operativa_id !== '') { $where[] = "p.operativa_id = :operativa"; $params[':operativa'] = $operativa_id; }

  $whereSql = $where ? ("WHERE " . implode(' AND ', $where)) : "";

  $campoAgrupar = ($agrupar === 'clase')
    ? "clase.nombre"
    : "grupo.nombre";

  // Elegir origen: nuevo core (wh_moves) o esquema base (wh_move)
  $usePlural = $hasTbl($pdo, 'wh_moves');
  $useSing   = !$usePlural && $hasTbl($pdo, 'wh_move');

  if ($usePlural) {
    $sql = "
      SELECT
        $campoAgrupar AS grupo_clase,
        p.sku,
        p.denominacion,
        SUM(CASE WHEN m.move_type = 'INBOUND' THEN COALESCE(mi.uc_unidades, mi.uv_cajas, 0) ELSE 0 END) AS mov_in,
        SUM(CASE WHEN m.move_type = 'OUTBOUND' THEN COALESCE(mi.uc_unidades, mi.uv_cajas, 0) ELSE 0 END) AS mov_out,
        (SUM(CASE WHEN m.move_type = 'INBOUND' THEN COALESCE(mi.uc_unidades, mi.uv_cajas, 0) ELSE 0 END) -
         SUM(CASE WHEN m.move_type = 'OUTBOUND' THEN COALESCE(mi.uc_unidades, mi.uv_cajas, 0) ELSE 0 END)) AS mov_net,
        MAX(DATE_FORMAT(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at),'%Y-%m-%d %H:%i')) AS ultimo_mov
      FROM wh_moves m
      JOIN wh_move_items mi   ON mi.move_id = m.id
      JOIN wh_pallet_items pi ON pi.id = mi.pallet_item_id
      JOIN para_productos p   ON p.id = pi.producto_id
      LEFT JOIN para_grupos grupo ON grupo.id = p.grupo_id
      LEFT JOIN para_clases clase ON clase.id = p.clase_id
      $whereSql
      GROUP BY $campoAgrupar, p.id
      ORDER BY mov_out DESC
    ";
  } elseif ($useSing) {
    // Esquema base: wh_move con columnas: tipo ('IN','MOVE','OUT'), producto_id, delta_uc/delta_uv, created_at
    $sql = "
      SELECT
        $campoAgrupar AS grupo_clase,
        p.sku,
        p.denominacion,
        SUM(CASE WHEN m.tipo = 'IN'  THEN COALESCE(m.delta_uc, m.delta_uv, 0) ELSE 0 END) AS mov_in,
        SUM(CASE WHEN m.tipo = 'OUT' THEN COALESCE(m.delta_uc, m.delta_uv, 0) ELSE 0 END) AS mov_out,
        (
          SUM(CASE WHEN m.tipo = 'IN'  THEN COALESCE(m.delta_uc, m.delta_uv, 0) ELSE 0 END) -
          SUM(CASE WHEN m.tipo = 'OUT' THEN COALESCE(m.delta_uc, m.delta_uv, 0) ELSE 0 END)
        ) AS mov_net,
        MAX(DATE_FORMAT(m.created_at, '%Y-%m-%d %H:%i')) AS ultimo_mov
      FROM wh_move m
      JOIN para_productos p   ON p.id = m.producto_id
      LEFT JOIN para_grupos grupo ON grupo.id = p.grupo_id
      LEFT JOIN para_clases clase ON clase.id = p.clase_id
      $whereSql
      GROUP BY $campoAgrupar, p.id
      ORDER BY mov_out DESC
    ";
    // Ajustar placeholders de fecha a m.created_at en $whereSql
    // (El where ya usa COALESCE(...) con alias m, compatible con m.created_at por fechas)
    $sql = str_replace(
      ["COALESCE(m.finalizado_at, m.iniciado_at, m.created_at)"],
      ["m.created_at"],
      $sql
    );
  } else {
    throw new RuntimeException("No existe la tabla de movimientos (wh_moves ni wh_move)");
  }

  $st = $pdo->prepare($sql);
  foreach ($params as $k => $v) $st->bindValue($k, $v);
  $st->execute();

  $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

  // Generamos sugerencias básicas
  foreach ($rows as &$r) {
    $out = (int)($r['mov_out'] ?? 0);
    if ($out >= 1000)       $r['sugerencia'] = "Ubicar en zona frontal / baja altura";
    elseif ($out >= 500)    $r['sugerencia'] = "Revisar proximidad al área de despacho";
    elseif ($out > 0)       $r['sugerencia'] = "Ubicación actual aceptable";
    else                    $r['sugerencia'] = "Sin rotación";
  }

  echo json_encode(['ok' => true, 'data' => $rows], JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['ok' => false, 'error' => 'Error cargando reporte', 'message' => $e->getMessage()]);
}
