<?php

declare(strict_types=1);

header('Content-Type: application/pdf');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

/* ======================
 * Dependencias (mPDF)
 * ====================== */
$aut = $ROOT . '/vendor/autoload.php';
if (!is_file($aut)) {
  http_response_code(500);
  echo "Falta vendor/autoload.php. Ejecuta: composer install";
  exit;
}
require_once $aut;
if (!class_exists(\Mpdf\Mpdf::class)) {
  http_response_code(500);
  echo "No está instalado mpdf/mpdf. Ejecuta: composer require mpdf/mpdf";
  exit;
}

/* ======================
 * Helpers
 * ====================== */
function nf($n, int $dec = 0): string
{
  if ($n === null || $n === '') return '0';
  return number_format((float)$n, $dec, ',', '.');
}
function pstr($v): string
{
  return htmlspecialchars((string)$v ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * Construye cláusulas WHERE por rango físico.
 * Si un límite viene vacío, no filtra por ese atributo.
 */
function wherePosRange(array $a, string $alias = 'pos'): array
{
  $w = [];
  $p = [];
  if (!empty($a['deposito_id'])) {
    $w[] = "$alias.deposito_id = :dep";
    $p[':dep'] = (int)$a['deposito_id'];
  }
  if (!empty($a['rack'])) {
    $w[] = "$alias.rack = :rack";
    $p[':rack'] = (int)$a['rack'];
  }
  if (!empty($a['col_ini'])) {
    $w[] = "$alias.columna >= :cini";
    $p[':cini'] = (int)$a['col_ini'];
  }
  if (!empty($a['col_fin'])) {
    $w[] = "$alias.columna <= :cfin";
    $p[':cfin'] = (int)$a['col_fin'];
  }
  if (!empty($a['niv_ini'])) {
    $w[] = "$alias.nivel   >= :nini";
    $p[':nini'] = (int)$a['niv_ini'];
  }
  if (!empty($a['niv_fin'])) {
    $w[] = "$alias.nivel   <= :nfin";
    $p[':nfin'] = (int)$a['niv_fin'];
  }
  if (!empty($a['fondo'])) {
    $w[] = "$alias.fondo = :fondo";
    $p[':fondo'] = (int)$a['fondo'];
  }
  return [$w ? ('WHERE ' . implode(' AND ', $w)) : '', $p];
}

/* ======================
 * Main
 * ====================== */
try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  // -------- Parámetros --------
  // ORIGEN: rango físico desde donde se retirarán pallets
  $origen = [
    'deposito_id' => $_GET['dep_origen'] ?? '',
    'rack'        => $_GET['rack_origen'] ?? '',
    'col_ini'     => $_GET['col_origen_ini'] ?? '',
    'col_fin'     => $_GET['col_origen_fin'] ?? '',
    'niv_ini'     => $_GET['niv_origen_ini'] ?? '',
    'niv_fin'     => $_GET['niv_origen_fin'] ?? '',
    'fondo'       => $_GET['fondo_origen'] ?? '',
  ];

  // DESTINO: rango físico donde se ubicarán (se asigna secuencialmente)
  $destino = [
    'deposito_id' => $_GET['dep_destino'] ?? '',
    'rack'        => $_GET['rack_destino'] ?? '',
    'col_ini'     => $_GET['col_dest_ini'] ?? '',
    'col_fin'     => $_GET['col_dest_fin'] ?? '',
    'niv_ini'     => $_GET['niv_dest_ini'] ?? '',
    'niv_fin'     => $_GET['niv_dest_fin'] ?? '',
    'fondo'       => $_GET['fondo_dest'] ?? '',
  ];

  // Límite de movimientos a planificar
  $max_rows = max(1, (int)($_GET['limite'] ?? 50));

  // Labels opcionales
  $responsable = trim((string)($_GET['responsable'] ?? ''));
  $observacion = trim((string)($_GET['nota'] ?? ''));

  // -------- Selección de ORIGEN: pallets actualmente ocupando posiciones del rango --------
  [$wOri, $pOri] = wherePosRange($origen, 'po');

  $sqlOrigen = "
    SELECT
      po.id              AS pos_id,
      po.deposito_id, po.rack, po.columna, po.nivel, po.fondo,
      CONCAT('R',LPAD(po.rack,2,'0'),'-C',LPAD(po.columna,2,'0'),'-N',po.nivel,'-F',po.fondo) AS pos_code,
      pa.id              AS pallet_id,
      pa.codigo          AS pallet_codigo,
      COALESCE(p.sku,'') AS sku,
      COALESCE(p.denominacion,'') AS producto,
      pr.prod_count
    FROM wh_positions po
    JOIN wh_position_occupancy occ ON occ.position_id = po.id AND occ.hasta IS NULL
    JOIN wh_pallets pa            ON pa.id = occ.pallet_id

    /* Elegimos UN (1) item representativo por pallet (el de menor pi.id) */
    LEFT JOIN (
      SELECT
        pi.pallet_id,
        MIN(pi.id) AS any_pi_id,
        COUNT(DISTINCT pi.producto_id) AS prod_count
      FROM wh_pallet_items pi
      GROUP BY pi.pallet_id
    ) pr ON pr.pallet_id = pa.id
    LEFT JOIN wh_pallet_items pi ON pi.id = pr.any_pi_id
    LEFT JOIN para_productos p    ON p.id = pi.producto_id

    $wOri
    GROUP BY po.id, pa.id, pr.any_pi_id
    ORDER BY po.rack ASC, po.columna ASC, po.nivel ASC, po.fondo ASC, pa.id ASC
    LIMIT :lim
";

  $stO = $pdo->prepare($sqlOrigen);
  foreach ($pOri as $k => $v) $stO->bindValue($k, $v);
  $stO->bindValue(':lim', $max_rows, PDO::PARAM_INT);
  $stO->execute();
  $rowsOri = $stO->fetchAll(PDO::FETCH_ASSOC) ?: [];

  // -------- Selección de DESTINO: posiciones libres en el rango destino --------
  [$wDes, $pDes] = wherePosRange($destino, 'pd');

  $sqlDestino = "
    SELECT
      pd.id AS pos_id,
      pd.deposito_id, pd.rack, pd.columna, pd.nivel, pd.fondo,
      CONCAT('R',LPAD(pd.rack,2,'0'),'-C',LPAD(pd.columna,2,'0'),'-N',pd.nivel,'-F',pd.fondo) AS pos_code
    FROM wh_positions pd
    LEFT JOIN wh_position_occupancy occ ON occ.position_id = pd.id AND occ.hasta IS NULL
    $wDes
      " . ($wDes ? " AND " : " WHERE ") . " occ.pallet_id IS NULL
    ORDER BY pd.rack ASC, pd.columna ASC, pd.nivel ASC, pd.fondo ASC
    LIMIT :lim
  ";

  $stD = $pdo->prepare($sqlDestino);
  foreach ($pDes as $k => $v) $stD->bindValue($k, $v);
  $stD->bindValue(':lim', $max_rows, PDO::PARAM_INT);
  $stD->execute();
  $rowsDes = $stD->fetchAll(PDO::FETCH_ASSOC) ?: [];

  // Emparejamiento 1:1 (origen i → destino i)
  $pairs = [];
  $N = min(count($rowsOri), count($rowsDes));
  for ($i = 0; $i < $N; $i++) {
    $pairs[] = [
      'idx'          => $i + 1,
      'pallet_codigo' => $rowsOri[$i]['pallet_codigo'] ?? '',
      'sku'          => $rowsOri[$i]['sku'] ?? '',
      'producto'     => $rowsOri[$i]['producto'] ?? '',
      'from_code'    => $rowsOri[$i]['pos_code'] ?? '',
      'to_code'      => $rowsDes[$i]['pos_code'] ?? '',
    ];
  }

  // -------- PDF ----------
  $mpdf = new \Mpdf\Mpdf([
    'format' => 'A4',
    'margin_left' => 10,
    'margin_right' => 10,
    'margin_top' => 14,
    'margin_bottom' => 12
  ]);

  $title = "Orden de movimientos para montacarguista";
  $styles = '<style>
    body{font-family:DejaVu Sans,Arial,sans-serif;font-size:11px;}
    h1{font-size:18px;margin:0 0 6px 0;}
    .muted{color:#666;}
    .kv{display:flex;flex-wrap:wrap;gap:12px;margin:6px 0 10px 0;}
    .kv div{min-width:180px;}
    .grid{width:100%;border-collapse:collapse;margin-top:6px;}
    .grid th,.grid td{border:0.6px solid #999;padding:6px;}
    .grid th{background:#f1f1f1;}
    .right{text-align:right;}
    .center{text-align:center;}
    .big{font-size:13px;font-weight:600;}
    .from{background:#e3f2fd;}   /* celeste claro */
    .to{background:#e8f5e9;}     /* verde claro */
    .note{height:24px}
    .sign{height:24px}
    .legend{margin-top:6px;font-size:10px;color:#555;}
    .badge{display:inline-block;padding:1px 6px;border-radius:999px;background:#212529;color:#fff;font-size:10px;}
  </style>';

  ob_start(); ?>
  <?= $styles ?>
  <h1><?= pstr($title) ?></h1>
  <div class="muted small">
    Generado: <?= date('Y-m-d H:i') ?> · Sistema: <?= pstr((string)env('APP_NAME', 'SOL')) ?>
  </div>

  <div class="kv">
    <div><b>Resp.:</b> <?= pstr($responsable ?: '—') ?></div>
    <div><b>Límite:</b> <?= (int)$max_rows ?></div>
    <div><b>Obs.:</b> <?= pstr($observacion ?: '—') ?></div>
  </div>

  <div class="kv">
    <div class="badge">ORIGEN</div>
    <div>Depósito: <?= pstr($origen['deposito_id'] ?: '—') ?></div>
    <div>Rack: <?= pstr($origen['rack'] ?: '—') ?></div>
    <div>Col: <?= pstr(($origen['col_ini'] ?: '—') . ' → ' . ($origen['col_fin'] ?: '—')) ?></div>
    <div>Niv: <?= pstr(($origen['niv_ini'] ?: '—') . ' → ' . ($origen['niv_fin'] ?: '—')) ?></div>
    <div>Fondo: <?= pstr($origen['fondo'] ?: '—') ?></div>
  </div>
  <div class="kv">
    <div class="badge">DESTINO</div>
    <div>Depósito: <?= pstr($destino['deposito_id'] ?: '—') ?></div>
    <div>Rack: <?= pstr($destino['rack'] ?: '—') ?></div>
    <div>Col: <?= pstr(($destino['col_ini'] ?: '—') . ' → ' . ($destino['col_fin'] ?: '—')) ?></div>
    <div>Niv: <?= pstr(($destino['niv_ini'] ?: '—') . ' → ' . ($destino['niv_fin'] ?: '—')) ?></div>
    <div>Fondo: <?= pstr($destino['fondo'] ?: '—') ?></div>
  </div>

  <table class="grid">
    <thead>
      <tr>
        <th class="center">#</th>
        <th>SKU</th>
        <th>Producto</th>
        <th class="from">Desde</th>
        <th class="to">Hacia</th>
        <th>Pallet</th>
        <th class="center">Pick</th>
        <th class="center">Drop</th>
        <th>Firma</th>
        <th>Obs.</th>
      </tr>
    </thead>
    <tbody>
      <?php foreach ($pairs as $r): ?>
        <tr>
          <td class="center big"><?= (int)$r['idx'] ?></td>
          <td><?= pstr($r['sku']) ?></td>
          <td><?= pstr($r['producto']) ?></td>
          <td class="from big"><?= pstr($r['from_code']) ?></td>
          <td class="to big"><?= pstr($r['to_code']) ?></td>
          <td class="big"><?= pstr($r['pallet_codigo']) ?></td>
          <td class="center"><input type="checkbox" /></td>
          <td class="center"><input type="checkbox" /></td>
          <td class="sign"></td>
          <td class="note"></td>
        </tr>
      <?php endforeach; ?>
      <?php if (empty($pairs)): ?>
        <tr>
          <td colspan="10" class="center">No hay movimientos planificados (verifique rangos y disponibilidad).</td>
        </tr>
      <?php endif; ?>
    </tbody>
    <tfoot>
      <tr>
        <th colspan="10" class="right">Total movimientos: <?= nf(count($pairs)) ?></th>
      </tr>
    </tfoot>
  </table>

  <div class="legend">
    Orden sugerido: procesar filas de arriba hacia abajo. La asignación es 1:1 en orden físico.
  </div>
<?php
  $html = ob_get_clean();

  $mpdf->SetTitle($title);
  $mpdf->SetAuthor((string)env('APP_NAME', 'SOL'));
  $mpdf->WriteHTML($html);
  $mpdf->Output('orden_movimientos.pdf', \Mpdf\Output\Destination::INLINE);
} catch (Throwable $e) {
  http_response_code(500);
  header('Content-Type: text/plain; charset=utf-8');
  echo (env('APP_ENV') === 'local') ? ("Error generando PDF: " . $e->getMessage()) : "No se pudo generar el PDF";
}
