<?php
declare(strict_types=1);
header('Content-Type: application/json');

require_once dirname(__DIR__, 2) . '/config/db.php';

$pdo = null;
try { $pdo = getPDO(); $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION); }
catch (Throwable $e) { http_response_code(500); echo json_encode(['ok'=>false,'error'=>'DB error']); exit; }

$in = json_decode(file_get_contents('php://input') ?: '[]', true) ?: [];
$ingresoId = isset($in['ingreso_id']) ? (int)$in['ingreso_id'] : 0;
$capacity = max(1, (int)($in['capacity'] ?? 20));
$limit    = max(0, (int)($in['limit'] ?? 0));
$simulate = (int)($in['simulate'] ?? 0) === 1;
$includeWithPosition = (int)($in['include_with_position'] ?? 0) === 1;
$changeState = (int)($in['change_state'] ?? 0) === 1;
$targetStateCode = isset($in['target_state_code']) ? strtoupper(trim((string)$in['target_state_code'])) : '';
date_default_timezone_set('America/Asuncion');

// Helpers
$hasTbl = function(PDO $pdo, string $t): bool {
  $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
  $st->execute([$t]);
  return (int)$st->fetchColumn() > 0;
};
$hasCol = function(PDO $pdo, string $tbl, string $col): bool {
  $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?");
  $st->execute([$tbl, $col]);
  return (int)$st->fetchColumn() > 0;
};

$palTbl = $hasTbl($pdo, 'wh_pallet') ? 'wh_pallet' : ($hasTbl($pdo, 'wh_pallets') ? 'wh_pallets' : null);
if (!$palTbl) { echo json_encode(['ok'=>false,'error'=>'Tabla de pallets no encontrada']); exit; }
$posCol = $hasCol($pdo, $palTbl, 'posicion_id') ? 'posicion_id' : ($hasCol($pdo, $palTbl, 'pos_id') ? 'pos_id' : null);
if (!$posCol) { echo json_encode(['ok'=>false,'error'=>'Columna de posición no encontrada']); exit; }
$hasDeleted = $hasCol($pdo, $palTbl, 'deleted_at');
$hasDepoPal = $hasCol($pdo, $palTbl, 'deposito_id');

// Identificar pallets asociados al ingreso (si corresponde)
$ingresoPalletIds = null;
$ingresoPalletLookupAttempted = false;
if ($ingresoId > 0) {
  $collected = [];
  try {
    if ($hasTbl($pdo, 'pl_ingreso') && $hasTbl($pdo, 'pl_rcv_link')) {
      $ingresoPalletLookupAttempted = true;
      $sqlIng = "SELECT DISTINCT link.pallet_id\n                   FROM pl_ingreso ing\n                   JOIN pl_rcv_link link ON link.packinglist_id = ing.packinglist_id\n                  WHERE ing.id = ?";
      $stIng = $pdo->prepare($sqlIng);
      $stIng->execute([$ingresoId]);
      $collected = array_merge($collected, array_map('intval', $stIng->fetchAll(PDO::FETCH_COLUMN, 0)));
    }
    if ($hasCol($pdo, $palTbl, 'pl_ingreso_id')) {
      $ingresoPalletLookupAttempted = true;
      $stIng = $pdo->prepare("SELECT id FROM {$palTbl} WHERE pl_ingreso_id = ?" . ($hasDeleted ? " AND deleted_at IS NULL" : ""));
      $stIng->execute([$ingresoId]);
      $collected = array_merge($collected, array_map('intval', $stIng->fetchAll(PDO::FETCH_COLUMN, 0)));
    }
    if ($hasCol($pdo, $palTbl, 'ingreso_id')) {
      $ingresoPalletLookupAttempted = true;
      $stIng = $pdo->prepare("SELECT id FROM {$palTbl} WHERE ingreso_id = ?" . ($hasDeleted ? " AND deleted_at IS NULL" : ""));
      $stIng->execute([$ingresoId]);
      $collected = array_merge($collected, array_map('intval', $stIng->fetchAll(PDO::FETCH_COLUMN, 0)));
    }
  } catch (Throwable $e) {
    $collected = [];
  }
  if ($ingresoPalletLookupAttempted) {
    $ingresoPalletIds = array_values(array_unique(array_filter($collected, static function($v){ return (int)$v > 0; })));
  }
}

$cuaId = 0; try { $cuaId = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='CUARENTENA' LIMIT 1")->fetchColumn() ?: 0); } catch (Throwable $e) { $cuaId = 0; }
$depFilter = isset($in['deposito_id']) && strlen((string)$in['deposito_id']) ? (int)$in['deposito_id'] : null;

// PICKING positions
$hasDepoPos = $hasCol($pdo, 'wh_posicion', 'deposito_id');
$params = [];
$sqlPos = "SELECT pos.id, pos.capacidad_pallets, pos.code, pos.code_full, pos.rack, pos.nivel, pos.columna" . ($hasDepoPos ? ", pos.deposito_id" : ", NULL AS deposito_id") . " AS deposito_id
            FROM wh_posicion pos
            JOIN wh_ambiente a ON a.id = pos.ambiente_id AND a.code='PICKING'
           WHERE (pos.activo = 1 OR pos.activo IS NULL)";
if ($depFilter && $hasDepoPos) { $sqlPos .= " AND pos.deposito_id = ?"; $params[] = $depFilter; }
$sqlPos .= " ORDER BY COALESCE(pos.code_full,pos.code,LPAD(pos.id,6,'0'))";
$pickPositions = $pdo->prepare($sqlPos); $pickPositions->execute($params);
$pickPositions = $pickPositions->fetchAll(PDO::FETCH_ASSOC);
if (!$pickPositions) { echo json_encode(['ok'=>false,'error'=>'No hay posiciones PICKING activas']); exit; }

// Candidate pallets: quarantine (if available) and no position; filter by deposito if requested
$where = [];
if (!$includeWithPosition) { $where[] = "(p.{$posCol} IS NULL OR p.{$posCol} = 0)"; }
$pp = [];
if ($cuaId) { $where[] = 'p.estado_id = ?'; $pp[] = $cuaId; }
if ($hasDeleted) { $where[] = 'p.deleted_at IS NULL'; }
if ($depFilter && $hasDepoPal && !$ingresoPalletLookupAttempted) { $where[] = 'p.deposito_id = ?'; $pp[] = $depFilter; }
$sqlPal = "SELECT p.id" . ($hasDepoPal ? ", p.deposito_id" : ", NULL AS deposito_id") . " AS deposito_id
            FROM {$palTbl} p
           WHERE " . implode(' AND ', $where) . "
           ORDER BY p.id ASC";
$palStmt = $pdo->prepare($sqlPal); $palStmt->execute($pp);
$pallets = $palStmt->fetchAll(PDO::FETCH_ASSOC);
if ($depFilter && $hasDepoPal && $ingresoPalletLookupAttempted) {
  $pallets = array_values(array_filter($pallets, static function(array $row) use ($depFilter): bool {
    if (!array_key_exists('deposito_id', $row)) { return true; }
    $dep = $row['deposito_id'];
    if ($dep === null || $dep === '' || (int)$dep === 0) { return true; }
    return (int)$dep === $depFilter;
  }));
}
if ($ingresoId > 0) {
  if (is_array($ingresoPalletIds) && !empty($ingresoPalletIds)) {
    $allowed = array_flip($ingresoPalletIds);
    $pallets = array_values(array_filter($pallets, static function(array $row) use ($allowed): bool {
      $pid = isset($row['id']) ? (int)$row['id'] : 0;
      return $pid > 0 && isset($allowed[$pid]);
    }));
  } elseif (is_array($ingresoPalletIds) && empty($ingresoPalletIds)) {
    echo json_encode(['ok'=>true,'message'=>'No se encontraron pallets asociados al ingreso #'.$ingresoId,'assigned'=>0,'positions'=>[], 'ingreso_id'=>$ingresoId]);
    exit;
  }
}
if (!$pallets) { echo json_encode(['ok'=>true,'message'=>($ingresoId>0 ? 'Sin pallets candidatos para el ingreso #'.$ingresoId : 'Sin pallets candidatos'),'assigned'=>0,'positions'=>[], 'ingreso_id'=>$ingresoId]); exit; }

// Current occupancy per position
$stCount = $pdo->prepare("SELECT COUNT(*) FROM {$palTbl} WHERE {$posCol} = ?" . ($hasDeleted ? " AND deleted_at IS NULL" : ""));
$posInfo = [];
foreach ($pickPositions as $pos) {
  $stCount->execute([(int)$pos['id']]);
  $cur = (int)$stCount->fetchColumn();
  $cap = (int)($pos['capacidad_pallets'] ?? 0);
  $posInfo[(int)$pos['id']] = [
    'depo' => $hasDepoPos ? (int)$pos['deposito_id'] : null,
    'current' => $cur,
    'remaining' => max(0, (($cap>0?$cap:$capacity) - $cur)),
  ];
}

// Group pallets by deposito (if applicable)
$palByDepo = [];
foreach ($pallets as $r) {
  $d = $hasDepoPal ? (int)$r['deposito_id'] : -1;
  $palByDepo[$d][] = (int)$r['id'];
}

$totalAssigned = 0; $assignments = [];
foreach ($pickPositions as $pos) {
  $posId = (int)$pos['id'];
  $remain = $posInfo[$posId]['remaining'];
  if ($remain <= 0) continue;
  $bucketKey = ($depFilter && $hasDepoPos && $hasDepoPal) ? (int)$pos['deposito_id'] : ($hasDepoPos && $hasDepoPal ? (int)$pos['deposito_id'] : -1);
  if (!isset($palByDepo[$bucketKey]) || empty($palByDepo[$bucketKey])) continue;
  $take = min($remain, ($limit>0?($limit-$totalAssigned):PHP_INT_MAX)); if ($take<=0) break;
  $takeIds = array_splice($palByDepo[$bucketKey], 0, $take);
  if (!$takeIds) continue;
  $assignments[$posId] = ($assignments[$posId] ?? []);
  foreach ($takeIds as $pid) { $assignments[$posId][] = $pid; }
  $totalAssigned += count($takeIds);
  if ($limit>0 && $totalAssigned >= $limit) break;
}

if ($totalAssigned === 0) { echo json_encode(['ok'=>true,'message'=>'Sin capacidad disponible','assigned'=>0,'positions'=>[], 'ingreso_id'=>$ingresoId]); exit; }

if ($simulate) {
  $byPos = [];
  foreach ($assignments as $posId => $ids) { $byPos[] = ['posicion_id'=>$posId,'count'=>count($ids)]; }
  $simPayload = ['ok'=>true,'simulate'=>1,'assigned'=>$totalAssigned,'positions'=>$byPos, 'ingreso_id'=>$ingresoId];
  if ($changeState) { $simPayload['changed_state'] = ['count'=>$totalAssigned, 'code'=>$targetStateCode ?: '(auto)']; }
  echo json_encode($simPayload);
  exit;
}

$pdo->beginTransaction();
try {
  $fromPosSet = [];
  // Map origin positions per pallet if rehoming, to recompute and to build stock moves later
  $originByPallet = [];
  foreach ($assignments as $posId => $ids) {
    $chunks = array_chunk($ids, 500);
    foreach ($chunks as $chunk) {
      if ($includeWithPosition) {
        // collect origin positions to recompute later
        $qs0 = implode(',', array_fill(0, count($chunk), '?'));
        $st0 = $pdo->prepare("SELECT id, {$posCol} FROM {$palTbl} WHERE id IN ({$qs0})" . ($hasDeleted ? " AND deleted_at IS NULL" : ""));
        $st0->execute($chunk);
        while ($row = $st0->fetch(PDO::FETCH_ASSOC)) {
          $pid = (int)($row['id'] ?? 0);
          $oid = (int)($row[$posCol] ?? 0);
          if ($oid>0) { $fromPosSet[$oid] = true; $originByPallet[$pid] = $oid; }
        }
      }
      $qs = implode(',', array_fill(0, count($chunk), '?'));
      $sqlUpd = "UPDATE {$palTbl} SET {$posCol} = ? WHERE id IN ({$qs})" . ($hasDeleted ? " AND deleted_at IS NULL" : "");
      $st = $pdo->prepare($sqlUpd);
      $st->execute(array_merge([$posId], $chunk));
    }
  }
  // Optionally change pallet state (from CUARENTENA -> target)
  $changedCount = 0; $resolvedCode = '';
  if ($changeState && $cuaId) {
    $targetId = 0; $resolvedCode = $targetStateCode;
    try {
      if ($targetStateCode) {
        $st = $pdo->prepare("SELECT id FROM wh_pallet_estado WHERE UPPER(code)=? LIMIT 1");
        $st->execute([$targetStateCode]); $targetId = (int)($st->fetchColumn() ?: 0);
      }
      if ($targetId<=0) {
        $candidates = ['POS_PICKEADO','UBICADO','DISPONIBLE','ALMACENADO','PICKING'];
        foreach ($candidates as $cc) {
          $st = $pdo->prepare("SELECT id FROM wh_pallet_estado WHERE code=? LIMIT 1"); $st->execute([$cc]);
          $targetId = (int)($st->fetchColumn() ?: 0);
          if ($targetId>0) { $resolvedCode = $cc; break; }
        }
      }
    } catch (Throwable $e) { $targetId = 0; }
    if ($targetId>0) {
      // Flatten ids to update
      $allIds = [];
      foreach ($assignments as $ids) { foreach ($ids as $id) $allIds[] = (int)$id; }
      if ($allIds) {
        $chunks = array_chunk($allIds, 500);
        foreach ($chunks as $chunk) {
          $qs = implode(',', array_fill(0, count($chunk), '?'));
          $sql = "UPDATE {$palTbl} SET estado_id = ? WHERE estado_id = ? AND id IN ({$qs})" . ($hasDeleted ? " AND deleted_at IS NULL" : "");
          $st = $pdo->prepare($sql);
          $st->execute(array_merge([$targetId, $cuaId], $chunk));
          $changedCount += $st->rowCount();
        }
      }
    }
  }
  // Recompute flags
  $stCnt = $pdo->prepare("SELECT COUNT(*) FROM {$palTbl} WHERE {$posCol} = ?" . ($hasDeleted ? " AND deleted_at IS NULL" : ""));
  $stUpd = $pdo->prepare("UPDATE wh_posicion SET ocupado = ?, picked = ? WHERE id = ?");
  foreach (array_keys($assignments) as $posId) {
    $stCnt->execute([$posId]); $cnt=(int)$stCnt->fetchColumn();
    $stUpd->execute([$cnt>0?1:0, $cnt>1?1:0, $posId]);
  }
  // Recompute flags for origin positions (rehomed)
  $rehomedFrom = [];
  if (!empty($fromPosSet)) {
    foreach (array_keys($fromPosSet) as $op) {
      // avoid double recompute if already updated above
      if (isset($assignments[$op])) continue;
      $stCnt->execute([$op]); $cnt=(int)$stCnt->fetchColumn();
      $stUpd->execute([$cnt>0?1:0, $cnt>1?1:0, $op]);
      $rehomedFrom[] = (int)$op;
    }
  }
  // If rehoming, create wh_move entries to sync wh_stock (move all stock rows per pallet to target position)
  $movesCreated = 0; $stockUpdated = 0;
  if ($includeWithPosition && !empty($originByPallet)) {
    try {
      $stStock = $pdo->prepare("SELECT producto_id, lote_id, posicion_id, qty_uv, qty_uc FROM wh_stock WHERE deposito_id = ? AND pallet_id = ? AND (qty_uv>0 OR qty_uc>0)");
      $insMove = $pdo->prepare("INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia) VALUES (?,?,?,?,?,?,?,?,?,?,?)");
      // Prepare helpers to directly update wh_stock when triggers are not present
      $stFindTgt = $pdo->prepare("SELECT id FROM wh_stock WHERE deposito_id=? AND producto_id=? AND (lote_id <=> ?) AND (pallet_id <=> ?) AND posicion_id=? LIMIT 1");
      $stUpdMerge = $pdo->prepare("UPDATE wh_stock SET qty_uv = qty_uv + ?, qty_uc = qty_uc + ? WHERE id=?");
      $stGetSrc   = $pdo->prepare("SELECT id FROM wh_stock WHERE deposito_id=? AND producto_id=? AND (lote_id <=> ?) AND (pallet_id <=> ?) AND posicion_id=? LIMIT 1");
      $stUpdPos   = $pdo->prepare("UPDATE wh_stock SET posicion_id = ? WHERE id = ?");
      $stInsRow   = $pdo->prepare("INSERT INTO wh_stock (deposito_id, posicion_id, producto_id, lote_id, pallet_id, qty_uv, qty_uc) VALUES (?,?,?,?,?,?,?)");
      foreach ($assignments as $toPosId => $pids) {
        $depTgt = $posInfo[$toPosId]['depo'] ?? ($depFilter ?: null);
        if ($depTgt === null) continue;
        foreach ($pids as $pid) {
          $stStock->execute([(int)$depTgt, (int)$pid]);
          while ($s = $stStock->fetch(PDO::FETCH_ASSOC)) {
            $fromPos = (int)($s['posicion_id'] ?? 0);
            $duv = (int)($s['qty_uv'] ?? 0); $duc = (int)($s['qty_uc'] ?? 0);
            if ($fromPos>0 && ($duv>0 || $duc>0)) {
              $insMove->execute([(int)$depTgt, 'MOVE', 'REUBICACION_PICKING', (int)$pid, (int)($s['producto_id'] ?? 0), ($s['lote_id']!==null ? (int)$s['lote_id'] : null), $fromPos, (int)$toPosId, $duv, $duc, 'AUTO-PICKING']);
              $movesCreated += 1;
              // Directly reflect the move in wh_stock to ensure position changes even without DB triggers
              $prodId = (int)($s['producto_id'] ?? 0);
              $loteId = ($s['lote_id'] !== null ? (int)$s['lote_id'] : null);
              // Try to merge into an existing target row
              $stFindTgt->execute([(int)$depTgt, $prodId, $loteId, (int)$pid, (int)$toPosId]);
              $tgtId = (int)($stFindTgt->fetchColumn() ?: 0);
              if ($tgtId > 0) {
                $stUpdMerge->execute([$duv, $duc, $tgtId]);
                $stockUpdated += $stUpdMerge->rowCount();
              } else {
                // Move the source row to the new position if found; otherwise insert a new one
                $stGetSrc->execute([(int)$depTgt, $prodId, $loteId, (int)$pid, $fromPos]);
                $srcId = (int)($stGetSrc->fetchColumn() ?: 0);
                if ($srcId > 0) {
                  $stUpdPos->execute([(int)$toPosId, $srcId]);
                  $stockUpdated += $stUpdPos->rowCount();
                } else {
                  $stInsRow->execute([(int)$depTgt, (int)$toPosId, $prodId, $loteId, (int)$pid, $duv, $duc]);
                  $stockUpdated += $stInsRow->rowCount();
                }
              }
            }
          }
        }
      }
    } catch (Throwable $e) {
      // If wh_move/wh_stock schema isn't available, continue without failing the whole op
    }
  }

  $pdo->commit();
  $byPos = [];
  foreach ($assignments as $posId => $ids) { $byPos[] = ['posicion_id'=>$posId,'count'=>count($ids)]; }
  $out = ['ok'=>true,'assigned'=>$totalAssigned,'positions'=>$byPos, 'ingreso_id'=>$ingresoId];
  if ($changeState) { $out['changed_state'] = ['count'=>$changedCount, 'code'=>$resolvedCode]; }
  if (!empty($rehomedFrom)) { $out['rehomed_from'] = $rehomedFrom; }
  if ($movesCreated>0) { $out['moves_created'] = $movesCreated; }
  if ($stockUpdated>0) { $out['stock_updated'] = $stockUpdated; }
  echo json_encode($out);
} catch (Throwable $e) {
  $pdo->rollBack();
  http_response_code(500);
  echo json_encode(['ok'=>false,'error'=>'Error al aplicar asignación','detail'=>$e->getMessage()]);
}
