<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  // ---------------------------
  // Helpers
  // ---------------------------
  $hasTable = function (PDO $pdo, string $t): bool {
    static $cache = [];
    if (array_key_exists($t, $cache)) return $cache[$t];
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
    $st->execute([$t]);
    return $cache[$t] = ((int)$st->fetchColumn() > 0);
  };

  // ---------------------------
  // Meta endpoints (combos)
  // ---------------------------
  if (isset($_GET['meta'])) {
    $meta = (string)$_GET['meta'];
    if ($meta === 'clientes') {
      $rows = $pdo->query("
        SELECT id, razon_social
        FROM para_clientes
        WHERE deleted_at IS NULL OR deleted_at IS NULL
        ORDER BY razon_social
      ")->fetchAll(PDO::FETCH_ASSOC);
      echo json_encode(['clientes' => $rows], JSON_UNESCAPED_UNICODE);
      exit;
    }
    if ($meta === 'operativas') {
      $rows = $pdo->query("
        SELECT id, nombre
        FROM sys_operativas
        WHERE deleted_at IS NULL OR deleted_at IS NULL
        ORDER BY nombre
      ")->fetchAll(PDO::FETCH_ASSOC);
      echo json_encode(['operativas' => $rows], JSON_UNESCAPED_UNICODE);
      exit;
    }
  }

  // ---------------------------
  // DataTables params
  // ---------------------------
  $draw   = (int)($_GET['draw']   ?? 1);
  $start  = (int)($_GET['start']  ?? 0);
  $length = (int)($_GET['length'] ?? 25);

  // ---------------------------
  // Filtros
  // ---------------------------
  $cliente_id   = trim((string)($_GET['cliente_id']   ?? ''));
  $operativa_id = trim((string)($_GET['operativa_id'] ?? ''));
  $q_producto   = trim((string)($_GET['q_producto']   ?? ''));
  $lote         = trim((string)($_GET['lote']         ?? ''));
  $dentro_dias  = (int)($_GET['dentro_dias']          ?? 30);
  $venc_desde   = trim((string)($_GET['venc_desde']   ?? ''));
  $venc_hasta   = trim((string)($_GET['venc_hasta']   ?? ''));
  $q_ubicacion  = trim((string)($_GET['q_ubicacion']  ?? ''));
  $mostrar      = (string)($_GET['mostrar']           ?? 'solo_proximos'); // solo_proximos|vencidos|proximos_y_vencidos|todos

  // Fecha "hoy" con la TZ del .env
  $tz = new DateTimeZone((string)env('TIMEZONE','UTC'));
  $today = (new DateTime('now', $tz))->format('Y-m-d');

  // ---------------------------
  // SQL base (schema-aware): por producto + lote + posición + vencimiento
  // ---------------------------
  $useNew = $hasTable($pdo,'wh_pallets') && $hasTable($pdo,'wh_pallet_items')
         && $hasTable($pdo,'wh_positions') && $hasTable($pdo,'wh_position_occupancy');

  if ($useNew) {
    $sql = "
      SELECT
        c.razon_social                                 AS cliente,
        opv.nombre                                     AS operativa,
        pr.sku,
        pr.denominacion,
        it.lote,
        it.fecha_vencimiento                           AS fecha_venc_raw,
        DATE_FORMAT(it.fecha_vencimiento, '%Y-%m-%d')  AS vencimiento,
        CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) AS ubicacion,

        COUNT(DISTINCT pa.id) AS pallets,
        SUM(COALESCE(it.uv_cajas, 0)) AS cajas,
        SUM(COALESCE(it.uc_total_cache,
              (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
            )) AS unidades,
        SUM(COALESCE(it.uc_total_cache,
              (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
            )) AS stock,
        SUM(CASE WHEN it.estado = 'RESERVADO'
                 THEN COALESCE(it.uc_total_cache,
                      (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
                      )
                 ELSE 0 END
            ) AS reservados,
        ( SUM(COALESCE(it.uc_total_cache,
              (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
            ))
          - SUM(CASE WHEN it.estado = 'RESERVADO'
                     THEN COALESCE(it.uc_total_cache,
                          (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
                          )
                     ELSE 0 END)
        ) AS disponibles,

        -- Días restantes (>=0: falta X días; <0: vencido hace |X| días)
        DATEDIFF(it.fecha_vencimiento, :today) AS dias_restantes
      FROM wh_pallets pa
      JOIN wh_pallet_items it ON it.pallet_id = pa.id
      JOIN para_productos pr ON pr.id = it.producto_id
      LEFT JOIN para_clientes c ON c.id = pr.cliente_id
      LEFT JOIN sys_operativas opv ON opv.id = pr.operativa_id

      LEFT JOIN wh_position_occupancy occ ON occ.pallet_id = pa.id AND occ.hasta IS NULL
      LEFT JOIN wh_positions pos ON pos.id = occ.position_id

      WHERE 1=1
    ";
  } else {
    // Legacy: wh_stock + wh_lote + wh_posicion; reservados = 0
    $sql = "
      SELECT
        c.razon_social                                 AS cliente,
        opv.nombre                                     AS operativa,
        pr.sku,
        pr.denominacion,
        COALESCE(l.codigo,'')                           AS lote,
        l.fecha_vencimiento                             AS fecha_venc_raw,
        DATE_FORMAT(l.fecha_vencimiento, '%Y-%m-%d')    AS vencimiento,
        CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) AS ubicacion,

        COUNT(DISTINCT s.pallet_id) AS pallets,
        SUM(COALESCE(s.qty_uv, 0))  AS cajas,
        SUM(COALESCE(s.qty_uc, 0))  AS unidades,
        SUM(COALESCE(s.qty_uc, 0))  AS stock,
        0                           AS reservados,
        SUM(COALESCE(s.qty_uc, 0))  AS disponibles,

        DATEDIFF(l.fecha_vencimiento, :today) AS dias_restantes
      FROM wh_stock s
      JOIN para_productos pr ON pr.id = s.producto_id
      LEFT JOIN para_clientes c ON c.id = pr.cliente_id
      LEFT JOIN sys_operativas opv ON opv.id = pr.operativa_id
      LEFT JOIN wh_lote l ON l.id = s.lote_id
      LEFT JOIN wh_posicion pos ON pos.id = s.posicion_id
      WHERE 1=1
    ";
  }

  $params = [':today' => $today];

  // Filtros directos
  if ($cliente_id !== '') {
    $sql .= " AND pr.cliente_id = :cliente_id";
    $params[':cliente_id'] = $cliente_id;
  }
  if ($operativa_id !== '') {
    $sql .= " AND pr.operativa_id = :operativa_id";
    $params[':operativa_id'] = $operativa_id;
  }
  if ($q_producto !== '') {
    $sql .= " AND (pr.sku LIKE :qp OR pr.denominacion LIKE :qp)";
    $params[':qp'] = "%{$q_producto}%";
  }
  if ($useNew) {
    if ($lote !== '') {
      $sql .= " AND it.lote LIKE :lote";
      $params[':lote'] = "%{$lote}%";
    }
    if ($venc_desde !== '') {
      $sql .= " AND it.fecha_vencimiento >= :vdesde";
      $params[':vdesde'] = $venc_desde;
    }
    if ($venc_hasta !== '') {
      $sql .= " AND it.fecha_vencimiento <= :vhasta";
      $params[':vhasta'] = $venc_hasta;
    }
  } else {
    if ($lote !== '') {
      $sql .= " AND l.codigo LIKE :lote";
      $params[':lote'] = "%{$lote}%";
    }
    if ($venc_desde !== '') {
      $sql .= " AND l.fecha_vencimiento >= :vdesde";
      $params[':vdesde'] = $venc_desde;
    }
    if ($venc_hasta !== '') {
      $sql .= " AND l.fecha_vencimiento <= :vhasta";
      $params[':vhasta'] = $venc_hasta;
    }
  }
  if ($q_ubicacion !== '') {
    $sql .= " AND CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) LIKE :qub";
    $params[':qub'] = "%{$q_ubicacion}%";
  }

  if ($useNew) {
    $sql .= "
      GROUP BY
        c.razon_social, opv.nombre, pr.sku, pr.denominacion,
        it.lote, it.fecha_vencimiento, pos.rack, pos.columna, pos.nivel, pos.fondo
    ";
  } else {
    $sql .= "
      GROUP BY
        c.razon_social, opv.nombre, pr.sku, pr.denominacion,
        l.codigo, l.fecha_vencimiento, pos.rack, pos.columna, pos.nivel, pos.fondo
    ";
  }

  // ---------------------------
  // Envoltura: filtrar por "mostrar" en base a dias_restantes
  // ---------------------------
  // - VENCIDO: dias_restantes < 0
  // - PRÓXIMO: 0 <= dias_restantes <= dentro_dias
  // - OK: dias_restantes > dentro_dias
  $outer = "
    SELECT
      t.*,
      CASE
        WHEN t.dias_restantes < 0 THEN 'VENCIDO'
        WHEN t.dias_restantes <= :dentro_case THEN 'PROXIMO'
        ELSE 'OK'
      END AS estado
    FROM (
      $sql
    ) t
    WHERE 1=1
  ";
  $params[':dentro_case'] = $dentro_dias;

  if ($mostrar === 'solo_proximos') {
    $outer .= " AND t.dias_restantes BETWEEN 0 AND :dentro_filter";
    $params[':dentro_filter'] = $dentro_dias;
  } elseif ($mostrar === 'vencidos') {
    $outer .= " AND t.dias_restantes < 0";
  } elseif ($mostrar === 'proximos_y_vencidos') {
    $outer .= " AND (t.dias_restantes < 0 OR t.dias_restantes <= :dentro_filter2)";
    $params[':dentro_filter2'] = $dentro_dias;
  } else {
    // 'todos' -> sin filtro extra
  }

  // Orden por defecto: primero los que vencen antes (menor dias_restantes), luego fecha
  $outer .= " ORDER BY t.dias_restantes ASC, t.fecha_venc_raw ASC";

  // ---------------------------
  // Conteo total
  // ---------------------------
  $stmtCount = $pdo->prepare("SELECT COUNT(*) FROM ($outer) z");
  foreach ($params as $k=>$v) $stmtCount->bindValue($k, $v);
  $stmtCount->execute();
  $recordsTotal = (int)$stmtCount->fetchColumn();

  // ---------------------------
  // Paginación
  // ---------------------------
  $sqlPaged = $outer . " LIMIT :start, :length";
  $stmt = $pdo->prepare($sqlPaged);
  foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
  $stmt->bindValue(':start', $start, PDO::PARAM_INT);
  $stmt->bindValue(':length', $length, PDO::PARAM_INT);
  $stmt->execute();
  $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

  echo json_encode([
    'draw'            => $draw,
    'recordsTotal'    => $recordsTotal,
    'recordsFiltered' => $recordsTotal,
    'data'            => $rows,
  ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'ok'    => false,
    'error' => 'Error cargando próximos a caducar',
    'msg'   => (env('APP_ENV')==='local') ? $e->getMessage() : ''
  ], JSON_UNESCAPED_UNICODE);
}
