<?php
declare(strict_types=1);

header('Content-Type: application/pdf');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

/* ======================
 * Dependencias (mPDF)
 * ====================== */
$aut = $ROOT . '/vendor/autoload.php';
if (!is_file($aut)) {
  http_response_code(500);
  echo "Falta vendor/autoload.php. Ejecuta: composer install";
  exit;
}
require_once $aut;
if (!class_exists(\Mpdf\Mpdf::class)) {
  http_response_code(500);
  echo "No está instalado mpdf/mpdf. Ejecuta: composer require mpdf/mpdf";
  exit;
}

/* ======================
 * Helpers
 * ====================== */
function nf($n, int $dec = 0): string {
  if ($n === null || $n === '') return '0';
  return number_format((float)$n, $dec, ',', '.'); // 1.234.567,89
}

/* ======================
 * Main
 * ====================== */
try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  // Helpers
  $hasTable = function (PDO $pdo, string $t): bool {
    static $cache = [];
    if (array_key_exists($t, $cache)) return $cache[$t];
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
    $st->execute([$t]);
    return $cache[$t] = ((int)$st->fetchColumn() > 0);
  };

  // -------- Filtros (igual que API listado) --------
  $cliente_id   = trim((string)($_GET['cliente_id']   ?? ''));
  $operativa_id = trim((string)($_GET['operativa_id'] ?? ''));
  $q_producto   = trim((string)($_GET['q_producto']   ?? ''));
  $lote         = trim((string)($_GET['lote']         ?? ''));
  $dentro_dias  = (int)($_GET['dentro_dias']          ?? 30);
  $venc_desde   = trim((string)($_GET['venc_desde']   ?? ''));
  $venc_hasta   = trim((string)($_GET['venc_hasta']   ?? ''));
  $q_ubicacion  = trim((string)($_GET['q_ubicacion']  ?? ''));
  $mostrar      = (string)($_GET['mostrar']           ?? 'solo_proximos');

  $tz = new DateTimeZone((string)env('TIMEZONE','UTC'));
  $today = (new DateTime('now', $tz))->format('Y-m-d');

  // Labels para encabezado
  $clienteLabel = $cliente_id ? ($pdo->query("SELECT razon_social FROM para_clientes WHERE id=".(int)$cliente_id)->fetchColumn() ?: '#'.$cliente_id) : '(Todos)';
  $operativaLabel = $operativa_id ? ($pdo->query("SELECT nombre FROM sys_operativas WHERE id=".(int)$operativa_id)->fetchColumn() ?: '#'.$operativa_id) : '(Todas)';

  // --- SQL (idéntico al endpoint listado, pero sin LIMIT) con detección de esquema ---
  $useNew = $hasTable($pdo,'wh_pallets') && $hasTable($pdo,'wh_pallet_items')
         && $hasTable($pdo,'wh_positions') && $hasTable($pdo,'wh_position_occupancy');

  if ($useNew) {
    $sql = "
      SELECT
        c.razon_social AS cliente,
        opv.nombre AS operativa,
        pr.sku,
        pr.denominacion,
        it.lote,
        it.fecha_vencimiento AS fecha_venc_raw,
        DATE_FORMAT(it.fecha_vencimiento, '%Y-%m-%d') AS vencimiento,
        CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) AS ubicacion,
        COUNT(DISTINCT pa.id) AS pallets,
        SUM(COALESCE(it.uv_cajas,0)) AS cajas,
        SUM(COALESCE(it.uc_total_cache,
          (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
        )) AS unidades,
        SUM(COALESCE(it.uc_total_cache,
          (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
        )) AS stock,
        SUM(CASE WHEN it.estado='RESERVADO'
          THEN COALESCE(it.uc_total_cache,
            (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
          )
          ELSE 0 END) AS reservados,
        (SUM(COALESCE(it.uc_total_cache,
          (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
        )) -
        SUM(CASE WHEN it.estado='RESERVADO'
          THEN COALESCE(it.uc_total_cache,
            (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
          )
          ELSE 0 END)) AS disponibles,
        DATEDIFF(it.fecha_vencimiento, :today) AS dias_restantes
      FROM wh_pallets pa
      JOIN wh_pallet_items it ON it.pallet_id = pa.id
      JOIN para_productos pr ON pr.id = it.producto_id
      LEFT JOIN para_clientes c ON c.id = pr.cliente_id
      LEFT JOIN sys_operativas opv ON opv.id = pr.operativa_id
      LEFT JOIN wh_position_occupancy occ ON occ.pallet_id=pa.id AND occ.hasta IS NULL
      LEFT JOIN wh_positions pos ON pos.id=occ.position_id
      WHERE 1=1
    ";
  } else {
    $sql = "
      SELECT
        c.razon_social AS cliente,
        opv.nombre AS operativa,
        pr.sku,
        pr.denominacion,
        COALESCE(l.codigo,'') AS lote,
        l.fecha_vencimiento AS fecha_venc_raw,
        DATE_FORMAT(l.fecha_vencimiento, '%Y-%m-%d') AS vencimiento,
        CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) AS ubicacion,
        COUNT(DISTINCT s.pallet_id) AS pallets,
        SUM(COALESCE(s.qty_uv,0)) AS cajas,
        SUM(COALESCE(s.qty_uc,0)) AS unidades,
        SUM(COALESCE(s.qty_uc,0)) AS stock,
        0 AS reservados,
        SUM(COALESCE(s.qty_uc,0)) AS disponibles,
        DATEDIFF(l.fecha_vencimiento, :today) AS dias_restantes
      FROM wh_stock s
      JOIN para_productos pr ON pr.id = s.producto_id
      LEFT JOIN para_clientes c ON c.id = pr.cliente_id
      LEFT JOIN sys_operativas opv ON opv.id = pr.operativa_id
      LEFT JOIN wh_lote l ON l.id = s.lote_id
      LEFT JOIN wh_posicion pos ON pos.id = s.posicion_id
      WHERE 1=1
    ";
  }
  $params = [':today'=>$today];

  if ($cliente_id)   { $sql.=" AND pr.cliente_id=:cid"; $params[':cid']=$cliente_id; }
  if ($operativa_id) { $sql.=" AND pr.operativa_id=:oid"; $params[':oid']=$operativa_id; }
  if ($q_producto)   { $sql.=" AND (pr.sku LIKE :qp OR pr.denominacion LIKE :qp)"; $params[':qp']="%$q_producto%"; }
  if ($useNew) {
    if ($lote)         { $sql.=" AND it.lote LIKE :lote"; $params[':lote']="%$lote%"; }
    if ($venc_desde)   { $sql.=" AND it.fecha_vencimiento >= :vd"; $params[':vd']=$venc_desde; }
    if ($venc_hasta)   { $sql.=" AND it.fecha_vencimiento <= :vh"; $params[':vh']=$venc_hasta; }
  } else {
    if ($lote)         { $sql.=" AND l.codigo LIKE :lote"; $params[':lote']="%$lote%"; }
    if ($venc_desde)   { $sql.=" AND l.fecha_vencimiento >= :vd"; $params[':vd']=$venc_desde; }
    if ($venc_hasta)   { $sql.=" AND l.fecha_vencimiento <= :vh"; $params[':vh']=$venc_hasta; }
  }
  if ($q_ubicacion)  { $sql.=" AND CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) LIKE :qub"; $params[':qub']="%$q_ubicacion%"; }

  if ($useNew) {
    $sql.="
      GROUP BY c.razon_social, opv.nombre, pr.sku, pr.denominacion,
               it.lote, it.fecha_vencimiento, pos.rack, pos.columna, pos.nivel, pos.fondo
    ";
  } else {
    $sql.="
      GROUP BY c.razon_social, opv.nombre, pr.sku, pr.denominacion,
               l.codigo, l.fecha_vencimiento, pos.rack, pos.columna, pos.nivel, pos.fondo
    ";
  }

  $outer = "SELECT t.*,
      CASE
        WHEN t.dias_restantes < 0 THEN 'VENCIDO'
        WHEN t.dias_restantes <= :dentro THEN 'PROXIMO'
        ELSE 'OK'
      END AS estado
    FROM ($sql) t
    WHERE 1=1";
  $params[':dentro']=$dentro_dias;

  if ($mostrar==='solo_proximos')         $outer.=" AND t.dias_restantes BETWEEN 0 AND :dentro";
  elseif ($mostrar==='vencidos')          $outer.=" AND t.dias_restantes<0";
  elseif ($mostrar==='proximos_y_vencidos') $outer.=" AND (t.dias_restantes<0 OR t.dias_restantes<=:dentro)";

  // Orden
  $orderSql=" ORDER BY t.dias_restantes ASC, t.fecha_venc_raw ASC";
  if (!empty($_GET['dt_order'])) {
    $colMap=[0=>'cliente',1=>'operativa',2=>'sku',3=>'denominacion',4=>'lote',5=>'vencimiento',6=>'dias_restantes',7=>'ubicacion',8=>'pallets',9=>'cajas',10=>'unidades',11=>'stock',12=>'reservados',13=>'disponibles',14=>'estado'];
    $raw=json_decode((string)$_GET['dt_order'],true);
    if (is_array($raw)&&!empty($raw)) {
      $parts=[];
      foreach($raw as $o){
        if(!is_array($o)||count($o)<2)continue;
        [$idx,$dir]=$o;
        $dir=strtolower((string)$dir)==='desc'?'DESC':'ASC';
        if(isset($colMap[(int)$idx]))$parts[]=$colMap[(int)$idx].' '.$dir;
        if(count($parts)>=4)break;
      }
      if($parts)$orderSql=' ORDER BY '.implode(', ',$parts);
    }
  }
  $outer.=$orderSql;

  $st=$pdo->prepare($outer);
  foreach($params as $k=>$v)$st->bindValue($k,$v);
  $st->execute();
  $rows=$st->fetchAll(PDO::FETCH_ASSOC)?:[];

  // Totales
  $tot=['pallets'=>0,'cajas'=>0,'unidades'=>0,'stock'=>0,'reservados'=>0,'disponibles'=>0];
  foreach($rows as $r){
    foreach($tot as $k=>$_)$tot[$k]+=(int)($r[$k]??0);
  }

  // ---------- PDF ----------
  $mpdf=new \Mpdf\Mpdf(['format'=>'A4','margin_left'=>10,'margin_right'=>10,'margin_top'=>14,'margin_bottom'=>12]);

  $title="Próximos a caducar";
  $styles='<style>
    body{font-family:DejaVu Sans,Arial,sans-serif;font-size:11px;}
    h1{font-size:18px;margin:0 0 6px 0;}
    .muted{color:#666;}
    .kv{display:flex;flex-wrap:wrap;gap:12px;margin:6px 0 8px 0;}
    .kv div{min-width:180px;}
    .grid{width:100%;border-collapse:collapse;margin-top:6px;}
    .grid th,.grid td{border:0.6px solid #999;padding:4px 6px;}
    .grid th{background:#f1f1f1;}
    .right{text-align:right;}
    .badge{display:inline-block;padding:1px 6px;border:1px solid #999;border-radius:4px;}
    .b-ok{background:#dff0d8;border-color:#adcea1;}
    .b-low{background:#f2dede;border-color:#e0b1b8;}
    .b-hi{background:#fcf8e3;border-color:#f0e1a0;}
  </style>';

  ob_start(); ?>
  <?= $styles ?>
  <h1><?= htmlspecialchars($title) ?></h1>
  <div class="muted small">Generado: <?= date('Y-m-d H:i') ?> · Sistema: <?= htmlspecialchars((string)env('APP_NAME','SOL')) ?></div>

  <div class="kv">
    <div><b>Cliente:</b> <?= htmlspecialchars($clienteLabel) ?></div>
    <div><b>Operativa:</b> <?= htmlspecialchars($operativaLabel) ?></div>
    <div><b>Producto:</b> <?= htmlspecialchars($q_producto ?: '—') ?></div>
    <div><b>Lote:</b> <?= htmlspecialchars($lote ?: '—') ?></div>
    <div><b>Rango vencimiento:</b> <?= htmlspecialchars($venc_desde ?: '—') ?> → <?= htmlspecialchars($venc_hasta ?: '—') ?></div>
    <div><b>Días próximos:</b> <?= $dentro_dias ?></div>
    <div><b>Mostrar:</b> <?= htmlspecialchars($mostrar) ?></div>
    <div><b>Filas:</b> <?= count($rows) ?></div>
  </div>

  <table class="grid">
    <thead>
      <tr>
        <th>Cliente</th><th>Operativa</th><th>SKU</th><th>Producto</th><th>Lote</th>
        <th>Vencimiento</th><th class="right">Días rest.</th><th>Ubicación</th>
        <th class="right">Pallets</th><th class="right">Cajas</th><th class="right">Unidades</th>
        <th class="right">Stock</th><th class="right">Reservados</th><th class="right">Disponibles</th>
        <th>Estado</th>
      </tr>
    </thead>
    <tbody>
    <?php foreach($rows as $r):
      $estado=strtoupper((string)($r['estado']??''));
      $cls='badge b-ok'; if($estado==='VENCIDO')$cls='badge b-low'; elseif($estado==='PROXIMO')$cls='badge b-hi'; ?>
      <tr>
        <td><?= htmlspecialchars((string)$r['cliente']) ?></td>
        <td><?= htmlspecialchars((string)$r['operativa']) ?></td>
        <td><?= htmlspecialchars((string)$r['sku']) ?></td>
        <td><?= htmlspecialchars((string)$r['denominacion']) ?></td>
        <td><?= htmlspecialchars((string)$r['lote']) ?></td>
        <td><?= htmlspecialchars((string)$r['vencimiento']) ?></td>
        <td class="right"><?= (int)$r['dias_restantes'] ?></td>
        <td><?= htmlspecialchars((string)$r['ubicacion']) ?></td>
        <td class="right"><?= nf($r['pallets']) ?></td>
        <td class="right"><?= nf($r['cajas']) ?></td>
        <td class="right"><?= nf($r['unidades']) ?></td>
        <td class="right"><?= nf($r['stock']) ?></td>
        <td class="right"><?= nf($r['reservados']) ?></td>
        <td class="right"><?= nf($r['disponibles']) ?></td>
        <td><span class="<?= $cls ?>"><?= $estado ?></span></td>
      </tr>
    <?php endforeach; ?>
    </tbody>
    <tfoot>
      <tr>
        <th colspan="8" class="right">Totales</th>
        <th class="right"><?= nf($tot['pallets']) ?></th>
        <th class="right"><?= nf($tot['cajas']) ?></th>
        <th class="right"><?= nf($tot['unidades']) ?></th>
        <th class="right"><?= nf($tot['stock']) ?></th>
        <th class="right"><?= nf($tot['reservados']) ?></th>
        <th class="right"><?= nf($tot['disponibles']) ?></th>
        <th></th>
      </tr>
    </tfoot>
  </table>
  <?php
  $html=ob_get_clean();

  $mpdf->SetTitle($title);
  $mpdf->SetAuthor((string)env('APP_NAME','SOL'));
  $mpdf->WriteHTML($html);
  $mpdf->Output('a_caducar.pdf', \Mpdf\Output\Destination::INLINE);

} catch (Throwable $e) {
  http_response_code(500);
  header('Content-Type: text/plain; charset=utf-8');
  echo (env('APP_ENV')==='local')?("Error generando PDF: ".$e->getMessage()):"No se pudo generar el PDF";
}
