<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);

require_once $ROOT . '/vendor/Editor/lib/DataTables.php';

$inventarioHelperPath = $ROOT . '/app/Support/InventarioReport.php';
if (!is_file($inventarioHelperPath)) {
    $altCandidates = [
        $ROOT . '/app/support/InventarioReport.php',
        $ROOT . '/app/Support/inventarioreport.php',
        $ROOT . '/app/support/inventarioreport.php',
    ];
    foreach ($altCandidates as $candidate) {
        if (is_file($candidate)) {
            $inventarioHelperPath = $candidate;
            break;
        }
    }

    if (!is_file($inventarioHelperPath) && is_dir($ROOT . '/app')) {
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($ROOT . '/app', FilesystemIterator::SKIP_DOTS)
        );
        foreach ($iterator as $fileInfo) {
            if (!$fileInfo->isFile()) {
                continue;
            }
            if (strcasecmp($fileInfo->getFilename(), 'InventarioReport.php') === 0) {
                $inventarioHelperPath = $fileInfo->getPathname();
                break;
            }
        }
    }
}

if (!is_file($inventarioHelperPath)) {
    throw new RuntimeException('No se encontró el helper InventarioReport.php.');
}

require_once $inventarioHelperPath;

use DataTables\Database;

try {
    global $db;

    if (!isset($db) || !$db instanceof Database) {
        throw new RuntimeException('No se pudo inicializar la conexión a la base de datos.');
    }

    $pdo = $db->resource();

    $dataset = inventario_existencias_dataset($pdo);
    if (($dataset['ok'] ?? false) !== true) {
        throw new RuntimeException($dataset['error'] ?? 'No se pudo obtener existencias.');
    }

    $allRows = $dataset['rows'] ?? [];
    $request = $_POST;
    if (!$request) {
        $request = $_GET;
    }

    $draw   = isset($request['draw']) ? (int) $request['draw'] : 0;
    $start  = isset($request['start']) ? max(0, (int) $request['start']) : 0;
    $length = isset($request['length']) ? (int) $request['length'] : -1;

    $searchValue = '';
    if (isset($request['search']) && is_array($request['search'])) {
        $searchValue = (string) ($request['search']['value'] ?? '');
    }

    $orders = [];
    if (isset($request['order']) && is_array($request['order'])) {
        $orders = $request['order'];
    }

    $columnMap = [
        0 => null,
        1 => 'cliente',
        2 => 'operativa',
        3 => 'sku',
        4 => 'denominacion',
        5 => 'pallets',
    6 => 'cajas',
    7 => 'unidades_sueltas',
    8 => 'stock',
    9 => 'reservados',
    10 => 'disponibles',
    ];

    $searchableFields = ['cliente', 'operativa', 'sku', 'denominacion'];

    $filteredRows = inventario_filter_rows($allRows, $searchValue, $searchableFields);
    $recordsTotal = count($allRows);
    $recordsFiltered = count($filteredRows);

    $orderedRows = inventario_order_rows($filteredRows, $orders, $columnMap);
    $pageRows = inventario_slice_rows($orderedRows, $start, $length);

    $response = [
        'ok'              => true,
        'draw'            => $draw,
        'recordsTotal'    => $recordsTotal,
        'recordsFiltered' => $recordsFiltered,
        'data'            => $pageRows,
        'rows'            => $pageRows,
        'schema'          => $dataset['schema'] ?? null,
        'totals'          => inventario_compute_totals($filteredRows),
        'grandTotals'     => inventario_compute_totals($allRows),
    ];

    echo json_encode($response, JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode([
        'ok'     => false,
        'error'  => 'Error cargando existencias',
        'detail' => inventario_is_local_env() ? $e->getMessage() : null,
    ], JSON_UNESCAPED_UNICODE);
}
