<?php

declare(strict_types=1);

header('Content-Type: application/pdf');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

// ---- mPDF
$aut = $ROOT . '/vendor/autoload.php';
if (!is_file($aut)) {
    http_response_code(500);
    echo "Falta vendor/autoload.php. Ejecuta: composer install";
    exit;
}
require_once $aut;
if (!class_exists(\Mpdf\Mpdf::class)) {
    http_response_code(500);
    echo "No está instalado mpdf/mpdf. Ejecuta: composer require mpdf/mpdf";
    exit;
}

// ---- Helper de inventario (mismo que el listado web)
$inventarioHelperPath = $ROOT . '/app/Support/InventarioReport.php';
if (!is_file($inventarioHelperPath)) {
    $fallbacks = [
        $ROOT . '/app/support/InventarioReport.php',
        $ROOT . '/app/Support/inventarioreport.php',
        $ROOT . '/app/support/inventarioreport.php',
    ];
    foreach ($fallbacks as $candidate) {
        if (is_file($candidate)) {
            $inventarioHelperPath = $candidate;
            break;
        }
    }
    if (!is_file($inventarioHelperPath)) {
        $iterator = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($ROOT . '/app', FilesystemIterator::SKIP_DOTS));
        foreach ($iterator as $fileInfo) {
            if ($fileInfo->isFile() && strcasecmp($fileInfo->getFilename(), 'InventarioReport.php') === 0) {
                $inventarioHelperPath = $fileInfo->getPathname();
                break;
            }
        }
    }
}

if (!is_file($inventarioHelperPath)) {
    http_response_code(500);
    echo 'No se encontró el helper InventarioReport.';
    exit;
}

require_once $inventarioHelperPath;

// ---- Helpers locales
function nf($n, int $dec = 0): string
{
    if ($n === null || $n === '') {
        return '0';
    }

    return number_format((float) $n, $dec, ',', '.');
}

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $dataset = inventario_existencias_dataset($pdo);
    if (($dataset['ok'] ?? false) !== true) {
        throw new RuntimeException($dataset['error'] ?? 'No se pudo obtener existencias.');
    }

    $rows = $dataset['rows'] ?? [];

    // Filtros básicos (string match) para mantener paridad con el listado
    $search = trim((string) ($_GET['search'] ?? ''));
    if ($search !== '') {
        $rows = inventario_filter_rows($rows, $search, ['cliente', 'operativa', 'sku', 'denominacion']);
    }

    $clienteName    = trim((string) ($_GET['cliente'] ?? ''));
    $operativaName  = trim((string) ($_GET['operativa'] ?? ''));
    $skuFilter      = trim((string) ($_GET['sku'] ?? ''));
    $productoFilter = trim((string) ($_GET['producto'] ?? ''));

    if ($clienteName !== '') {
        $rows = array_values(array_filter($rows, static function (array $row) use ($clienteName): bool {
            return strcasecmp(trim((string) ($row['cliente'] ?? '')), $clienteName) === 0;
        }));
    }

    if ($operativaName !== '') {
        $rows = array_values(array_filter($rows, static function (array $row) use ($operativaName): bool {
            return strcasecmp(trim((string) ($row['operativa'] ?? '')), $operativaName) === 0;
        }));
    }

    if ($skuFilter !== '') {
        $rows = array_values(array_filter($rows, static function (array $row) use ($skuFilter): bool {
            return stripos((string) ($row['sku'] ?? ''), $skuFilter) !== false;
        }));
    }

    if ($productoFilter !== '') {
        $rows = array_values(array_filter($rows, static function (array $row) use ($productoFilter): bool {
            return stripos((string) ($row['denominacion'] ?? ''), $productoFilter) !== false;
        }));
    }

    // Ordenamiento consistente con la vista de existencias
    usort($rows, static function (array $a, array $b): int {
        $fields = ['cliente', 'operativa', 'sku'];
        foreach ($fields as $field) {
            $va = mb_strtolower((string) ($a[$field] ?? ''));
            $vb = mb_strtolower((string) ($b[$field] ?? ''));
            if ($va === $vb) {
                continue;
            }
            return $va <=> $vb;
        }
        return 0;
    });

    $totals = inventario_compute_totals($rows);

    $summaryMetrics = [
        'Productos'         => nf(count($rows)),
        'Pallets'           => nf($totals['pallets']),
        'Cajas (UV)'        => nf($totals['cajas']),
        'Unidades sueltas'  => nf($totals['unidades_sueltas']),
        'Stock total (UC)'  => nf($totals['stock']),
        'Reservados'        => nf($totals['reservados']),
        'Disponibles'       => nf($totals['disponibles']),
    ];

    // ---------- Render PDF ----------
    $mpdf = new \Mpdf\Mpdf([
        'format' => 'A4',
        'margin_left'   => 10,
        'margin_right'  => 10,
        'margin_top'    => 14,
        'margin_bottom' => 12,
    ]);

        $title = 'Listado de existencias';
        $styles = '
        <style>
            body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 11px; }
            h1 { font-size: 18px; margin: 0 0 6px 0; }
            .muted { color:#666; }
            .kv { display:flex; flex-wrap:wrap; gap:12px; margin:6px 0 8px 0; }
            .kv div { min-width: 180px; }
            .grid { width:100%; border-collapse: collapse; margin-top:6px; }
            .grid th, .grid td { border: 0.6px solid #999; padding: 4px 6px; }
            .grid th { background: #f1f1f1; }
            .right { text-align:right; }
            .small { font-size: 10px; }
        </style>
    ';

        ob_start();
?>
        <?= $styles ?>
    <h1><?= htmlspecialchars($title) ?></h1>
    <div class="muted small">Generado: <?= date('Y-m-d H:i') ?> · Sistema: <?= htmlspecialchars((string) env('APP_NAME', 'SOL')) ?></div>

    <div class="kv">
        <div><b>Filtro búsqueda:</b> <?= htmlspecialchars($search !== '' ? $search : '—') ?></div>
        <div><b>Cliente:</b> <?= htmlspecialchars($clienteName !== '' ? $clienteName : '(Todos)') ?></div>
        <div><b>Operativa:</b> <?= htmlspecialchars($operativaName !== '' ? $operativaName : '(Todas)') ?></div>
        <div><b>SKU:</b> <?= htmlspecialchars($skuFilter !== '' ? $skuFilter : '—') ?></div>
        <div><b>Producto:</b> <?= htmlspecialchars($productoFilter !== '' ? $productoFilter : '—') ?></div>
        <div><b>Filas listadas:</b> <?= nf(count($rows)) ?></div>
    </div>

    <table class="grid">
        <thead>
            <tr>
                <?php foreach ($summaryMetrics as $label => $value): ?>
                    <th><?= htmlspecialchars($label) ?></th>
                <?php endforeach; ?>
            </tr>
        </thead>
        <tbody>
            <tr>
                <?php foreach ($summaryMetrics as $value): ?>
                    <td class="right">&nbsp;<?= $value ?></td>
                <?php endforeach; ?>
            </tr>
        </tbody>
    </table>

    <br>

    <table class="grid">
        <thead>
            <tr>
                <th>Cliente</th>
                <th>Operativa</th>
                <th>SKU</th>
                <th>Producto</th>
                <th class="right">Pallets</th>
                <th class="right">Cajas (UV)</th>
                <th class="right">Unidades sueltas (UC)</th>
                <th class="right">Stock total (UC)</th>
                <th class="right">Reservados</th>
                <th class="right">Disponibles</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($rows as $r): ?>
                <tr>
                    <td><?= htmlspecialchars((string) ($r['cliente'] ?? '')) ?></td>
                    <td><?= htmlspecialchars((string) ($r['operativa'] ?? '')) ?></td>
                    <td><?= htmlspecialchars((string) ($r['sku'] ?? '')) ?></td>
                    <td><?= htmlspecialchars((string) ($r['denominacion'] ?? '')) ?></td>
                    <td class="right">&nbsp;<?= nf($r['pallets'] ?? 0) ?></td>
                    <td class="right">&nbsp;<?= nf($r['cajas'] ?? 0) ?></td>
                    <td class="right">&nbsp;<?= nf($r['unidades_sueltas'] ?? ($r['uc'] ?? 0)) ?></td>
                    <td class="right">&nbsp;<?= nf($r['stock'] ?? 0) ?></td>
                    <td class="right">&nbsp;<?= nf($r['reservados'] ?? 0) ?></td>
                    <td class="right">&nbsp;<?= nf($r['disponibles'] ?? 0) ?></td>
                </tr>
            <?php endforeach; ?>
        </tbody>
        <tfoot>
            <tr>
                <th colspan="4" class="right">Totales</th>
                <th class="right"><?= nf($totals['pallets']) ?></th>
                <th class="right"><?= nf($totals['cajas']) ?></th>
                <th class="right"><?= nf($totals['unidades_sueltas']) ?></th>
                <th class="right"><?= nf($totals['stock']) ?></th>
                <th class="right"><?= nf($totals['reservados']) ?></th>
                <th class="right"><?= nf($totals['disponibles']) ?></th>
            </tr>
        </tfoot>
    </table>
<?php
    $html = ob_get_clean();

    $mpdf->SetTitle($title);
    $mpdf->SetAuthor((string) env('APP_NAME', 'SOL'));
    $mpdf->WriteHTML($html);
    $mpdf->Output('existencias.pdf', \Mpdf\Output\Destination::INLINE);
} catch (Throwable $e) {
    http_response_code(500);
    header('Content-Type: text/plain; charset=utf-8');
    echo (env('APP_ENV') === 'local') ? ('Error generando PDF: ' . $e->getMessage()) : 'No se pudo generar el PDF';
}
