<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  // Helpers
  $hasTable = function (PDO $pdo, string $t): bool {
    static $cache = [];
    if (array_key_exists($t, $cache)) return $cache[$t];
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
    $st->execute([$t]);
    return $cache[$t] = ((int)$st->fetchColumn() > 0);
  };

  // ---------------------------
  // Meta endpoints (combos)
  // ---------------------------
  if (isset($_GET['meta'])) {
    $meta = (string)$_GET['meta'];
    if ($meta === 'clientes') {
      $rows = $pdo->query("
        SELECT id, razon_social
        FROM para_clientes
        WHERE deleted_at IS NULL
        ORDER BY razon_social
      ")->fetchAll(PDO::FETCH_ASSOC);
      echo json_encode(['clientes' => $rows], JSON_UNESCAPED_UNICODE);
      exit;
    }
    if ($meta === 'operativas') {
      $rows = $pdo->query("
        SELECT id, nombre
        FROM sys_operativas
        WHERE deleted_at IS NULL
        ORDER BY nombre
      ")->fetchAll(PDO::FETCH_ASSOC);
      echo json_encode(['operativas' => $rows], JSON_UNESCAPED_UNICODE);
      exit;
    }
  }

  // ---------------------------
  // DataTables params
  // ---------------------------
  $draw   = (int)($_GET['draw']   ?? 1);
  $start  = (int)($_GET['start']  ?? 0);
  $length = (int)($_GET['length'] ?? 25);

  // ---------------------------
  // Filtros
  // ---------------------------
  $cliente_id   = trim((string)($_GET['cliente_id']   ?? ''));
  $operativa_id = trim((string)($_GET['operativa_id'] ?? ''));
  $q_producto   = trim((string)($_GET['q_producto']   ?? ''));
  $q_pallet     = trim((string)($_GET['q_pallet']     ?? ''));
  $q_ubicacion  = trim((string)($_GET['q_ubicacion']  ?? ''));
  $tipo         = strtoupper(trim((string)($_GET['tipo'] ?? '')));

  // Fecha "hoy"
  $today = (new DateTime('now', new DateTimeZone((string)env('TIMEZONE','UTC'))))->format('Y-m-d H:i:s');

  // ---------------------------
  // Detección de esquema
  // ---------------------------
  $useNew = $hasTable($pdo,'wh_pallets') && $hasTable($pdo,'wh_pallet_items')
         && $hasTable($pdo,'wh_positions') && $hasTable($pdo,'wh_position_occupancy');

  if (!$useNew) {
    // En esquema legacy no existe separación "físico" (ocupación) vs "registrado" (ítems),
    // por lo que no podemos detectar diferencias. Devolvemos dataset vacío de forma segura.
    echo json_encode([
      'draw' => $draw,
      'recordsTotal' => 0,
      'recordsFiltered' => 0,
      'data' => [],
      'note' => 'Reporte de diferencias no disponible en esquema legacy.'
    ], JSON_UNESCAPED_UNICODE);
    return;
  }

  // ---------------------------
  // Query base por pallet (warehouse-core)
  // ---------------------------
  $sql = "
    SELECT
      pa.id AS pallet_id,
      pa.codigo AS pallet_codigo,
      c.razon_social AS cliente,
      opv.nombre AS operativa,
      CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) AS ubicacion,
      COUNT(DISTINCT it.id) AS items_count,
      SUM(COALESCE(it.uc_total_cache,
            (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
          )) AS unidades,
      COUNT(DISTINCT it.producto_id) AS productos_distintos,
      -- tipo de diferencia
      CASE
        WHEN COUNT(DISTINCT it.id) > 0 AND occ.id IS NULL THEN 'REGISTRADO_SIN_FISICO'
        WHEN COUNT(DISTINCT it.id) = 0 AND occ.id IS NOT NULL THEN 'FISICO_SIN_REGISTRO'
        ELSE NULL
      END AS tipo,
      CASE
        WHEN COUNT(DISTINCT it.id) > 0 AND occ.id IS NULL THEN 'Tiene registros pero no está ubicado'
        WHEN COUNT(DISTINCT it.id) = 0 AND occ.id IS NOT NULL THEN 'Está en posición física pero no tiene ítems registrados'
        ELSE NULL
      END AS observacion
    FROM wh_pallets pa
    LEFT JOIN wh_pallet_items it ON it.pallet_id = pa.id
    LEFT JOIN para_productos pr ON pr.id = it.producto_id
    LEFT JOIN para_clientes c ON c.id = pr.cliente_id
    LEFT JOIN sys_operativas opv ON opv.id = pr.operativa_id
    LEFT JOIN wh_position_occupancy occ ON occ.pallet_id = pa.id AND occ.hasta IS NULL
    LEFT JOIN wh_positions pos ON pos.id = occ.position_id
    WHERE 1=1
  ";

  $params = [];

  if ($cliente_id !== '') {
    $sql .= " AND pr.cliente_id = :cliente_id";
    $params[':cliente_id'] = $cliente_id;
  }
  if ($operativa_id !== '') {
    $sql .= " AND pr.operativa_id = :operativa_id";
    $params[':operativa_id'] = $operativa_id;
  }
  if ($q_producto !== '') {
    $sql .= " AND (pr.sku LIKE :qp OR pr.denominacion LIKE :qp)";
    $params[':qp'] = "%{$q_producto}%";
  }
  if ($q_pallet !== '') {
    $sql .= " AND pa.codigo LIKE :qpallet";
    $params[':qpallet'] = "%{$q_pallet}%";
  }
  if ($q_ubicacion !== '') {
    $sql .= " AND CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) LIKE :qub";
    $params[':qub'] = "%{$q_ubicacion}%";
  }

  $sql .= "
    GROUP BY pa.id, pa.codigo, c.razon_social, opv.nombre,
             pos.rack, pos.columna, pos.nivel, pos.fondo, occ.id
    HAVING tipo IS NOT NULL
  ";

  if ($tipo === 'REGISTRADO_SIN_FISICO') {
    $sql .= " AND tipo='REGISTRADO_SIN_FISICO'";
  } elseif ($tipo === 'FISICO_SIN_REGISTRO') {
    $sql .= " AND tipo='FISICO_SIN_REGISTRO'";
  }

  $sql .= " ORDER BY tipo ASC, pa.codigo ASC";

  // ---------------------------
  // Conteo total
  // ---------------------------
  $stmtCount = $pdo->prepare("SELECT COUNT(*) FROM ($sql) x");
  foreach ($params as $k=>$v) $stmtCount->bindValue($k, $v);
  $stmtCount->execute();
  $recordsTotal = (int)$stmtCount->fetchColumn();

  // ---------------------------
  // Paginación
  // ---------------------------
  $sqlPaged = $sql . " LIMIT :start, :length";
  $stmt = $pdo->prepare($sqlPaged);
  foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
  $stmt->bindValue(':start', $start, PDO::PARAM_INT);
  $stmt->bindValue(':length', $length, PDO::PARAM_INT);
  $stmt->execute();
  $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

  echo json_encode([
    'draw'            => $draw,
    'recordsTotal'    => $recordsTotal,
    'recordsFiltered' => $recordsTotal,
    'data'            => $rows,
  ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'ok'    => false,
    'error' => 'Error cargando físico vs registrado',
    'msg'   => (env('APP_ENV')==='local') ? $e->getMessage() : ''
  ], JSON_UNESCAPED_UNICODE);
}
