<?php
declare(strict_types=1);

header('Content-Type: application/pdf');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

/* ======================
 * Dependencias (mPDF)
 * ====================== */
$aut = $ROOT . '/vendor/autoload.php';
if (!is_file($aut)) {
  http_response_code(500);
  echo "Falta vendor/autoload.php. Ejecuta: composer install";
  exit;
}
require_once $aut;
if (!class_exists(\Mpdf\Mpdf::class)) {
  http_response_code(500);
  echo "No está instalado mpdf/mpdf. Ejecuta: composer require mpdf/mpdf";
  exit;
}

/* ======================
 * Helpers
 * ====================== */
function nf($n, int $dec = 0): string {
  if ($n === null || $n === '') return '0';
  return number_format((float)$n, $dec, ',', '.'); // 1.234.567,89
}

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  // Helpers
  $hasTable = function (PDO $pdo, string $t): bool {
    static $cache = [];
    if (array_key_exists($t, $cache)) return $cache[$t];
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
    $st->execute([$t]);
    return $cache[$t] = ((int)$st->fetchColumn() > 0);
  };

  // -------- Filtros --------
  $cliente_id   = trim((string)($_GET['cliente_id']   ?? ''));
  $operativa_id = trim((string)($_GET['operativa_id'] ?? ''));
  $q_producto   = trim((string)($_GET['q_producto']   ?? ''));
  $q_pallet     = trim((string)($_GET['q_pallet']     ?? ''));
  $q_ubicacion  = trim((string)($_GET['q_ubicacion']  ?? ''));
  $tipo         = strtoupper(trim((string)($_GET['tipo'] ?? '')));

  // Labels
  $clienteLabel = $cliente_id ? ($pdo->query("SELECT razon_social FROM para_clientes WHERE id=".(int)$cliente_id)->fetchColumn() ?: '#'.$cliente_id) : '(Todos)';
  $operativaLabel = $operativa_id ? ($pdo->query("SELECT nombre FROM sys_operativas WHERE id=".(int)$operativa_id)->fetchColumn() ?: '#'.$operativa_id) : '(Todas)';

  // -------- SQL base (igual al API, sin LIMIT) --------
  $useNew = $hasTable($pdo,'wh_pallets') && $hasTable($pdo,'wh_pallet_items')
         && $hasTable($pdo,'wh_positions') && $hasTable($pdo,'wh_position_occupancy');

  if (!$useNew) {
    // Mostrar un PDF mínimo con mensaje de no disponible (legacy)
    $mpdf=new \Mpdf\Mpdf(['format'=>'A4']);
    $mpdf->WriteHTML('<h2>Físico vs. Registrado</h2><p>Este reporte no está disponible en el esquema legacy.</p>');
    $mpdf->Output('fisico_vs_registrado.pdf', \Mpdf\Output\Destination::INLINE);
    return;
  }

  $sql = "
    SELECT
      pa.codigo AS pallet_codigo,
      c.razon_social AS cliente,
      opv.nombre AS operativa,
      CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) AS ubicacion,
      COUNT(DISTINCT it.id) AS items_count,
      SUM(COALESCE(it.uc_total_cache,
            (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
          )) AS unidades,
      COUNT(DISTINCT it.producto_id) AS productos_distintos,
      CASE
        WHEN COUNT(DISTINCT it.id) > 0 AND occ.id IS NULL THEN 'REGISTRADO_SIN_FISICO'
        WHEN COUNT(DISTINCT it.id) = 0 AND occ.id IS NOT NULL THEN 'FISICO_SIN_REGISTRO'
        ELSE NULL
      END AS tipo,
      CASE
        WHEN COUNT(DISTINCT it.id) > 0 AND occ.id IS NULL THEN 'Tiene registros pero no está ubicado'
        WHEN COUNT(DISTINCT it.id) = 0 AND occ.id IS NOT NULL THEN 'Está en posición física pero no tiene ítems registrados'
        ELSE NULL
      END AS observacion
    FROM wh_pallets pa
    LEFT JOIN wh_pallet_items it ON it.pallet_id = pa.id
    LEFT JOIN para_productos pr ON pr.id = it.producto_id
    LEFT JOIN para_clientes c ON c.id = pr.cliente_id
    LEFT JOIN sys_operativas opv ON opv.id = pr.operativa_id
    LEFT JOIN wh_position_occupancy occ ON occ.pallet_id = pa.id AND occ.hasta IS NULL
    LEFT JOIN wh_positions pos ON pos.id = occ.position_id
    WHERE 1=1
  ";
  $params = [];

  if ($cliente_id)   { $sql.=" AND pr.cliente_id=:cid"; $params[':cid']=$cliente_id; }
  if ($operativa_id) { $sql.=" AND pr.operativa_id=:oid"; $params[':oid']=$operativa_id; }
  if ($q_producto)   { $sql.=" AND (pr.sku LIKE :qp OR pr.denominacion LIKE :qp)"; $params[':qp']="%$q_producto%"; }
  if ($q_pallet)     { $sql.=" AND pa.codigo LIKE :qpallet"; $params[':qpallet']="%$q_pallet%"; }
  if ($q_ubicacion)  { $sql.=" AND CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) LIKE :qub"; $params[':qub']="%$q_ubicacion%"; }

  $sql .= "
    GROUP BY pa.id, pa.codigo, c.razon_social, opv.nombre,
             pos.rack, pos.columna, pos.nivel, pos.fondo, occ.id
    HAVING tipo IS NOT NULL
  ";

  if ($tipo === 'REGISTRADO_SIN_FISICO') {
    $sql .= " AND tipo='REGISTRADO_SIN_FISICO'";
  } elseif ($tipo === 'FISICO_SIN_REGISTRO') {
    $sql .= " AND tipo='FISICO_SIN_REGISTRO'";
  }

  $sql .= " ORDER BY tipo ASC, pa.codigo ASC";

  $st=$pdo->prepare($sql);
  foreach($params as $k=>$v) $st->bindValue($k,$v);
  $st->execute();
  $rows=$st->fetchAll(PDO::FETCH_ASSOC)?:[];

  // Totales
  $tot=['pallets'=>count($rows),'items'=>0,'unidades'=>0,'productos'=>0];
  foreach($rows as $r){
    $tot['items']      += (int)($r['items_count'] ?? 0);
    $tot['unidades']   += (int)($r['unidades'] ?? 0);
    $tot['productos']  += (int)($r['productos_distintos'] ?? 0);
  }

  // ---------- PDF ----------
  $mpdf=new \Mpdf\Mpdf(['format'=>'A4','margin_left'=>10,'margin_right'=>10,'margin_top'=>14,'margin_bottom'=>12]);

  $title="Físico vs. Registrado (diferencias)";
  $styles='<style>
    body{font-family:DejaVu Sans,Arial,sans-serif;font-size:11px;}
    h1{font-size:18px;margin:0 0 6px 0;}
    .muted{color:#666;}
    .kv{display:flex;flex-wrap:wrap;gap:12px;margin:6px 0 8px 0;}
    .kv div{min-width:180px;}
    .grid{width:100%;border-collapse:collapse;margin-top:6px;}
    .grid th,.grid td{border:0.6px solid #999;padding:4px 6px;}
    .grid th{background:#f1f1f1;}
    .right{text-align:right;}
    .badge{display:inline-block;padding:1px 6px;border:1px solid #999;border-radius:4px;}
    .b-reg{background:#fcf8e3;border-color:#f0e1a0;}
    .b-fis{background:#f2dede;border-color:#e0b1b8;}
  </style>';

  ob_start(); ?>
  <?= $styles ?>
  <h1><?= htmlspecialchars($title) ?></h1>
  <div class="muted small">Generado: <?= date('Y-m-d H:i') ?> · Sistema: <?= htmlspecialchars((string)env('APP_NAME','SOL')) ?></div>

  <div class="kv">
    <div><b>Cliente:</b> <?= htmlspecialchars($clienteLabel) ?></div>
    <div><b>Operativa:</b> <?= htmlspecialchars($operativaLabel) ?></div>
    <div><b>Producto filtro:</b> <?= htmlspecialchars($q_producto ?: '—') ?></div>
    <div><b>Pallet filtro:</b> <?= htmlspecialchars($q_pallet ?: '—') ?></div>
    <div><b>Ubicación filtro:</b> <?= htmlspecialchars($q_ubicacion ?: '—') ?></div>
    <div><b>Tipo diferencia:</b> <?= htmlspecialchars($tipo ?: '(Todos)') ?></div>
    <div><b>Total pallets afectados:</b> <?= $tot['pallets'] ?></div>
  </div>

  <table class="grid">
    <thead>
      <tr>
        <th>Tipo</th><th>Cliente</th><th>Operativa</th><th>Pallet</th><th>Ubicación</th>
        <th class="right">Ítems</th><th class="right">Unidades</th><th class="right">Productos</th>
        <th>Observación</th>
      </tr>
    </thead>
    <tbody>
    <?php foreach($rows as $r):
      $tipo=strtoupper((string)($r['tipo']??''));
      $cls='badge';
      if($tipo==='REGISTRADO_SIN_FISICO')$cls.=' b-reg';
      elseif($tipo==='FISICO_SIN_REGISTRO')$cls.=' b-fis'; ?>
      <tr>
        <td><span class="<?= $cls ?>"><?= htmlspecialchars($tipo) ?></span></td>
        <td><?= htmlspecialchars((string)$r['cliente']) ?></td>
        <td><?= htmlspecialchars((string)$r['operativa']) ?></td>
        <td><?= htmlspecialchars((string)$r['pallet_codigo']) ?></td>
        <td><?= htmlspecialchars((string)$r['ubicacion']) ?></td>
        <td class="right"><?= nf($r['items_count']) ?></td>
        <td class="right"><?= nf($r['unidades']) ?></td>
        <td class="right"><?= nf($r['productos_distintos']) ?></td>
        <td><?= htmlspecialchars((string)$r['observacion']) ?></td>
      </tr>
    <?php endforeach; ?>
    </tbody>
    <tfoot>
      <tr>
        <th colspan="5" class="right">Totales</th>
        <th class="right"><?= nf($tot['items']) ?></th>
        <th class="right"><?= nf($tot['unidades']) ?></th>
        <th class="right"><?= nf($tot['productos']) ?></th>
        <th></th>
      </tr>
    </tfoot>
  </table>
  <?php
  $html=ob_get_clean();

  $mpdf->SetTitle($title);
  $mpdf->SetAuthor((string)env('APP_NAME','SOL'));
  $mpdf->WriteHTML($html);
  $mpdf->Output('fisico_vs_registrado.pdf', \Mpdf\Output\Destination::INLINE);

} catch (Throwable $e) {
  http_response_code(500);
  header('Content-Type: text/plain; charset=utf-8');
  echo (env('APP_ENV')==='local')?("Error generando PDF: ".$e->getMessage()):"No se pudo generar el PDF";
}
