<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function hasTable(PDO $pdo, string $t): bool {
  static $cache = [];
  if (array_key_exists($t, $cache)) return $cache[$t];
  $stmt = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
  $stmt->execute([$t]);
  return $cache[$t] = ((int)$stmt->fetchColumn() > 0);
}
/**
 * DataTables server-side endpoint:
 * - Lista existencias por producto vs. mínimos y máximos.
 * - Filtros: cliente_id, operativa_id, q_producto, estado [BAJO|OK|SOBRE], criterio [disponibles|stock],
 *            solo_con_min, solo_con_max
 * - Meta: ?meta=clientes | ?meta=operativas
 */
function hasColumn(PDO $pdo, string $t, string $c): bool {
  static $cache = [];
  $k = $t.'|'.$c;
  if (array_key_exists($k, $cache)) return $cache[$k];
  $stmt = $pdo->prepare("
    SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?
  ");
  $stmt->execute([$t,$c]);
  $cache[$k] = ((int)$stmt->fetchColumn() > 0);
  return $cache[$k];
}

try {
  $pdo = get_pdo();

  // ---- Meta combos ----
  if (isset($_GET['meta'])) {
    $meta = (string)$_GET['meta'];
    if ($meta === 'clientes') {
      $rows = $pdo->query("
        SELECT id, razon_social
        FROM para_clientes
        WHERE deleted_at IS NULL OR deleted_at IS NULL
        ORDER BY razon_social
      ")->fetchAll(PDO::FETCH_ASSOC);
      echo json_encode(['clientes'=>$rows], JSON_UNESCAPED_UNICODE);
      exit;
    }
    if ($meta === 'operativas') {
      $rows = $pdo->query("
        SELECT id, nombre
        FROM sys_operativas
        WHERE deleted_at IS NULL OR deleted_at IS NULL
        ORDER BY nombre
      ")->fetchAll(PDO::FETCH_ASSOC);
      echo json_encode(['operativas'=>$rows], JSON_UNESCAPED_UNICODE);
      exit;
    }
  }

  // ---- DataTables params ----
  $draw   = (int)($_GET['draw']   ?? 1);
  $start  = (int)($_GET['start']  ?? 0);
  $length = (int)($_GET['length'] ?? 25);

  // ---- Filtros ----
  $cliente_id   = trim((string)($_GET['cliente_id']   ?? ''));
  $operativa_id = trim((string)($_GET['operativa_id'] ?? ''));
  $q_producto   = trim((string)($_GET['q_producto']   ?? ''));
  $estado       = strtoupper(trim((string)($_GET['estado'] ?? ''))); // BAJO|OK|SOBRE
  $criterio     = trim((string)($_GET['criterio']     ?? 'disponibles')); // disponibles|stock
  $solo_con_min = isset($_GET['solo_con_min']) && $_GET['solo_con_min'] !== '' ? 1 : 0;
  $solo_con_max = isset($_GET['solo_con_max']) && $_GET['solo_con_max'] !== '' ? 1 : 0;

  // ---- Columna de máximo (si existe en para_productos) ----
  $hasMax = hasColumn($pdo, 'para_productos', 'stock_max_default');

  // ---- Subtotales por producto con detección de esquema ----
  // Si existe wh_pallet_items (warehouse-core), agregamos desde allí; si no, usamos wh_stock (legacy) con reservados=0.
  $useNew = hasTable($pdo, 'wh_pallet_items');
  if ($useNew) {
    $sub = "
      SELECT
        it.producto_id,
        SUM(COALESCE(it.uc_total_cache,
            (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
        )) AS stock,
        SUM(CASE WHEN it.estado = 'RESERVADO' THEN
            COALESCE(it.uc_total_cache,
              (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
            )
          ELSE 0 END
        ) AS reservados
      FROM wh_pallet_items it
      GROUP BY it.producto_id
    ";
  } else {
    $sub = "
      SELECT
        s.producto_id,
        SUM(COALESCE(s.qty_uc,0)) AS stock,
        0 AS reservados
      FROM wh_stock s
      GROUP BY s.producto_id
    ";
  }

  // ---- SQL principal: por producto ----
  $sql = "
    SELECT
      c.razon_social AS cliente,
      op.nombre      AS operativa,
      p.id           AS producto_id,
      p.sku,
      p.denominacion,

      -- Mín/Máx configurados
      COALESCE(p.stock_min_default, 0) AS minimo,
      ".($hasMax ? "p.stock_max_default" : "NULL")." AS maximo,

      -- Totales actuales
      COALESCE(agg.stock, 0)      AS stock,
      COALESCE(agg.reservados, 0) AS reservados,
      (COALESCE(agg.stock, 0) - COALESCE(agg.reservados, 0)) AS disponibles
    FROM para_productos p
    LEFT JOIN ($sub) agg ON agg.producto_id = p.id
    LEFT JOIN para_clientes c ON c.id = p.cliente_id
    LEFT JOIN sys_operativas op ON op.id = p.operativa_id
    WHERE (p.deleted_at IS NULL OR p.deleted_at IS NULL)
  ";

  $params = [];

  if ($cliente_id !== '') {
    $sql .= " AND p.cliente_id = :cliente_id";
    $params[':cliente_id'] = $cliente_id;
  }
  if ($operativa_id !== '') {
    $sql .= " AND p.operativa_id = :operativa_id";
    $params[':operativa_id'] = $operativa_id;
  }
  if ($q_producto !== '') {
    $sql .= " AND (p.sku LIKE :qp OR p.denominacion LIKE :qp)";
    $params[':qp'] = "%{$q_producto}%";
  }
  if ($solo_con_min) {
    $sql .= " AND COALESCE(p.stock_min_default,0) > 0";
  }
  if ($solo_con_max && $hasMax) {
    $sql .= " AND COALESCE(p.stock_max_default,0) > 0";
  } elseif ($solo_con_max && !$hasMax) {
    // Si pidieron 'solo con max' pero no existe la columna, forzamos vacío:
    $sql .= " AND 1=0";
  }

  // ---- Envoltorio para calcular derivadas y filtrar por estado/criterio ----
  $outer = "
    SELECT
      t.cliente, t.operativa, t.producto_id, t.sku, t.denominacion,
      t.minimo, t.maximo, t.stock, t.reservados, t.disponibles,
      GREATEST(t.minimo - ".($criterio === 'stock' ? "t.stock" : "t.disponibles").", 0) AS faltante_min,
      CASE
        WHEN t.maximo IS NULL THEN 0
        ELSE GREATEST(".($criterio === 'stock' ? "t.stock" : "t.disponibles")." - t.maximo, 0)
      END AS exceso_max,
      CASE
        WHEN ".($criterio === 'stock' ? "t.stock" : "t.disponibles")." < t.minimo THEN 'BAJO'
        WHEN t.maximo IS NOT NULL AND ".($criterio === 'stock' ? "t.stock" : "t.disponibles")." > t.maximo THEN 'SOBRE'
        ELSE 'OK'
      END AS estado
    FROM (
      $sql
    ) t
    WHERE 1=1
  ";

  // Filtro por estado
  if ($estado === 'BAJO') {
    $outer .= " AND (".($criterio === 'stock' ? "t.stock" : "t.disponibles")." < t.minimo)";
  } elseif ($estado === 'SOBRE') {
    $outer .= " AND (t.maximo IS NOT NULL AND ".($criterio === 'stock' ? "t.stock" : "t.disponibles")." > t.maximo)";
  } elseif ($estado === 'OK') {
    $outer .= " AND (".($criterio === 'stock' ? "t.stock" : "t.disponibles")." >= t.minimo) AND (t.maximo IS NULL OR ".($criterio === 'stock' ? "t.stock" : "t.disponibles")." <= t.maximo)";
  }

  // Orden por defecto
  $outer .= " ORDER BY t.operativa ASC, t.sku ASC";

  // ---- Conteo total
  $stmtCount = $pdo->prepare("SELECT COUNT(*) FROM ($outer) x");
  foreach ($params as $k=>$v) $stmtCount->bindValue($k, $v);
  $stmtCount->execute();
  $recordsTotal = (int)$stmtCount->fetchColumn();

  // ---- Paginación
  $sqlPaged = $outer . " LIMIT :start, :length";
  $stmt = $pdo->prepare($sqlPaged);
  foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
  $stmt->bindValue(':start', $start, PDO::PARAM_INT);
  $stmt->bindValue(':length', $length, PDO::PARAM_INT);
  $stmt->execute();
  $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

  echo json_encode([
    'draw'            => $draw,
    'recordsTotal'    => $recordsTotal,
    'recordsFiltered' => $recordsTotal,
    'data'            => $rows,
  ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'ok'    => false,
    'error' => 'Error cargando min/max',
    'msg'   => (env('APP_ENV')==='local') ? $e->getMessage() : ''
  ], JSON_UNESCAPED_UNICODE);
}
