<?php
declare(strict_types=1);

header('Content-Type: application/pdf');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

/* ======================
 * Dependencias (mPDF)
 * ====================== */
$aut = $ROOT . '/vendor/autoload.php';
if (!is_file($aut)) {
  http_response_code(500);
  echo "Falta vendor/autoload.php. Ejecuta: composer install";
  exit;
}
require_once $aut;
if (!class_exists(\Mpdf\Mpdf::class)) {
  http_response_code(500);
  echo "No está instalado mpdf/mpdf. Ejecuta: composer require mpdf/mpdf";
  exit;
}

/* ======================
 * Helpers
 * ====================== */
function nf($n, int $dec = 0): string {
  if ($n === null || $n === '') return '0';
  return number_format((float)$n, $dec, ',', '.'); // 1.234.567,89
}
function hasColumn(PDO $pdo, string $t, string $c): bool {
  static $cache = [];
  $k = $t.'|'.$c;
  if (array_key_exists($k, $cache)) return $cache[$k];
  $stmt = $pdo->prepare("
    SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?
  ");
  $stmt->execute([$t,$c]);
  return $cache[$k] = ((int)$stmt->fetchColumn() > 0);
}

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  // -------- Filtros (idénticos al listado) --------
  $cliente_id   = trim((string)($_GET['cliente_id']   ?? ''));
  $operativa_id = trim((string)($_GET['operativa_id'] ?? ''));
  $q_producto   = trim((string)($_GET['q_producto']   ?? ''));
  $estado       = strtoupper(trim((string)($_GET['estado'] ?? ''))); // BAJO|OK|SOBRE|'' (todos)
  $criterio     = trim((string)($_GET['criterio']     ?? 'disponibles')); // disponibles|stock
  $solo_con_min = isset($_GET['solo_con_min']) && $_GET['solo_con_min'] !== '' ? 1 : 0;
  $solo_con_max = isset($_GET['solo_con_max']) && $_GET['solo_con_max'] !== '' ? 1 : 0;

  // Labels para encabezado
  $clienteLabel = '(Todos)';
  if ($cliente_id !== '') {
    $st = $pdo->prepare("SELECT razon_social FROM para_clientes WHERE id=? LIMIT 1");
    $st->execute([$cliente_id]);
    $clienteLabel = (string)($st->fetchColumn() ?: '#'.$cliente_id);
  }
  $operativaLabel = '(Todas)';
  if ($operativa_id !== '') {
    $st = $pdo->prepare("SELECT nombre FROM sys_operativas WHERE id=? LIMIT 1");
    $st->execute([$operativa_id]);
    $operativaLabel = (string)($st->fetchColumn() ?: '#'.$operativa_id);
  }
  $estadoLabel = $estado !== '' ? $estado : '(Todos)';
  $criterioLabel = ($criterio === 'stock') ? 'Stock total' : 'Disponibles (stock - reservados)';

  // ¿Existe columna de máximo?
  $hasMax = hasColumn($pdo, 'para_productos', 'stock_max_default');

  // ----- Subtotales (stock/reservados) por producto -----
  $sub = "
    SELECT
      it.producto_id,
      SUM(COALESCE(it.uc_total_cache,
          (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
      )) AS stock,
      SUM(CASE WHEN it.estado = 'RESERVADO' THEN
          COALESCE(it.uc_total_cache,
            (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
          )
        ELSE 0 END
      ) AS reservados
    FROM wh_pallet_items it
    GROUP BY it.producto_id
  ";

  // ----- SQL base (por producto) -----
  $sql = "
    SELECT
      c.razon_social AS cliente,
      op.nombre      AS operativa,
      p.id           AS producto_id,
      p.sku,
      p.denominacion,
      COALESCE(p.stock_min_default, 0) AS minimo,
      ".($hasMax ? "p.stock_max_default" : "NULL")." AS maximo,
      COALESCE(agg.stock, 0)      AS stock,
      COALESCE(agg.reservados, 0) AS reservados,
      (COALESCE(agg.stock, 0) - COALESCE(agg.reservados, 0)) AS disponibles
    FROM para_productos p
    LEFT JOIN ($sub) agg ON agg.producto_id = p.id
    LEFT JOIN para_clientes c ON c.id = p.cliente_id
    LEFT JOIN sys_operativas op ON op.id = p.operativa_id
    WHERE (p.deleted_at IS NULL OR p.deleted_at IS NULL)
  ";

  $params = [];

  if ($cliente_id !== '') {
    $sql .= " AND p.cliente_id = :cliente_id";
    $params[':cliente_id'] = $cliente_id;
  }
  if ($operativa_id !== '') {
    $sql .= " AND p.operativa_id = :operativa_id";
    $params[':operativa_id'] = $operativa_id;
  }
  if ($q_producto !== '') {
    $sql .= " AND (p.sku LIKE :qp OR p.denominacion LIKE :qp)";
    $params[':qp'] = "%{$q_producto}%";
  }
  if ($solo_con_min) {
    $sql .= " AND COALESCE(p.stock_min_default,0) > 0";
  }
  if ($solo_con_max && $hasMax) {
    $sql .= " AND COALESCE(p.stock_max_default,0) > 0";
  } elseif ($solo_con_max && !$hasMax) {
    // Forzar vacío si pidieron 'solo con max' pero no existe la columna
    $sql .= " AND 1=0";
  }

  // ----- Cálculos derivados + filtro por estado y orden dinámico -----
  $outer = "
    SELECT
      t.cliente, t.operativa, t.producto_id, t.sku, t.denominacion,
      t.minimo, t.maximo, t.stock, t.reservados, t.disponibles,
      GREATEST(t.minimo - ".($criterio === 'stock' ? "t.stock" : "t.disponibles").", 0) AS faltante_min,
      CASE
        WHEN t.maximo IS NULL THEN 0
        ELSE GREATEST(".($criterio === 'stock' ? "t.stock" : "t.disponibles")." - t.maximo, 0)
      END AS exceso_max,
      CASE
        WHEN ".($criterio === 'stock' ? "t.stock" : "t.disponibles")." < t.minimo THEN 'BAJO'
        WHEN t.maximo IS NOT NULL AND ".($criterio === 'stock' ? "t.stock" : "t.disponibles")." > t.maximo THEN 'SOBRE'
        ELSE 'OK'
      END AS estado
    FROM (
      $sql
    ) t
    WHERE 1=1
  ";

  if ($estado === 'BAJO') {
    $outer .= " AND (".($criterio === 'stock' ? "t.stock" : "t.disponibles")." < t.minimo)";
  } elseif ($estado === 'SOBRE') {
    $outer .= " AND (t.maximo IS NOT NULL AND ".($criterio === 'stock' ? "t.stock" : "t.disponibles")." > t.maximo)";
  } elseif ($estado === 'OK') {
    $outer .= " AND (".($criterio === 'stock' ? "t.stock" : "t.disponibles")." >= t.minimo) AND (t.maximo IS NULL OR ".($criterio === 'stock' ? "t.stock" : "t.disponibles")." <= t.maximo)";
  }

  // Orden por defecto
  $orderSql = " ORDER BY operativa ASC, sku ASC";

  // Orden desde DataTable (dt_order = JSON [[idx,'asc'|'desc'],...])
  $colMap = [
    0  => 'cliente',
    1  => 'operativa',
    2  => 'sku',
    3  => 'denominacion',
    4  => 'minimo',
    5  => 'maximo',
    6  => 'stock',
    7  => 'reservados',
    8  => 'disponibles',
    9  => 'faltante_min',
    10 => 'exceso_max',
    11 => 'estado',
  ];
  if (!empty($_GET['dt_order'])) {
    $raw = json_decode((string)$_GET['dt_order'], true);
    if (is_array($raw) && !empty($raw)) {
      $parts = [];
      foreach ($raw as $o) {
        if (!is_array($o) || count($o) < 2) continue;
        [$idx, $dir] = $o;
        $idx = (int)$idx;
        $dir = strtolower((string)$dir) === 'desc' ? 'DESC' : 'ASC';
        if (array_key_exists($idx, $colMap)) {
          $parts[] = $colMap[$idx] . ' ' . $dir;
        }
        if (count($parts) >= 4) break;
      }
      if ($parts) $orderSql = ' ORDER BY ' . implode(', ', $parts);
    }
  }

  $outer .= $orderSql;

  // Ejecutar
  $st = $pdo->prepare($outer);
  foreach ($params as $k=>$v) $st->bindValue($k, $v);
  $st->execute();
  $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

  // Totales globales
  $tot = [
    'minimo' => 0, 'maximo' => 0,
    'stock' => 0, 'reservados' => 0, 'disponibles' => 0,
    'faltante_min' => 0, 'exceso_max' => 0
  ];
  foreach ($rows as $r) {
    $tot['minimo']       += (int)($r['minimo'] ?? 0);
    $tot['maximo']       += (int)($r['maximo'] ?? 0);
    $tot['stock']        += (int)($r['stock'] ?? 0);
    $tot['reservados']   += (int)($r['reservados'] ?? 0);
    $tot['disponibles']  += (int)($r['disponibles'] ?? 0);
    $tot['faltante_min'] += (int)($r['faltante_min'] ?? 0);
    $tot['exceso_max']   += (int)($r['exceso_max'] ?? 0);
  }

  // ---------- Render PDF ----------
  $mpdf = new \Mpdf\Mpdf([
    'format' => 'A4',
    'margin_left'   => 10,
    'margin_right'  => 10,
    'margin_top'    => 14,
    'margin_bottom' => 12,
  ]);

  $title = "Existencias mínimas y máximas";
  $styles = '
    <style>
      body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 11px; }
      h1 { font-size: 18px; margin: 0 0 6px 0; }
      .muted { color:#666; }
      .kv { display:flex; flex-wrap:wrap; gap:12px; margin:6px 0 8px 0; }
      .kv div { min-width: 180px; }
      .grid { width:100%; border-collapse: collapse; margin-top:6px; }
      .grid th, .grid td { border: 0.6px solid #999; padding: 4px 6px; }
      .grid th { background: #f1f1f1; }
      .right { text-align:right; }
      .badge { display:inline-block; padding:1px 6px; border:1px solid #999; border-radius:4px; }
      .b-ok { background:#dff0d8; border-color:#adcea1; }
      .b-low { background:#f2dede; border-color:#e0b1b8; }
      .b-hi  { background:#fcf8e3; border-color:#f0e1a0; }
      .small { font-size: 10px; }
    </style>
  ';

  ob_start();
  ?>
  <?= $styles ?>
  <h1><?= htmlspecialchars($title) ?></h1>
  <div class="muted small">
    Generado: <?= date('Y-m-d H:i') ?> · Sistema: <?= htmlspecialchars((string)env('APP_NAME','SOL')) ?>
  </div>

  <div class="kv">
    <div><b>Cliente:</b> <?= htmlspecialchars($clienteLabel) ?></div>
    <div><b>Operativa:</b> <?= htmlspecialchars($operativaLabel) ?></div>
    <div><b>Producto:</b> <?= htmlspecialchars($q_producto !== '' ? $q_producto : '—') ?></div>
    <div><b>Estado:</b> <?= htmlspecialchars($estadoLabel) ?></div>
    <div><b>Criterio:</b> <?= htmlspecialchars($criterioLabel) ?></div>
    <div><b>Solo con mínimo:</b> <?= $solo_con_min ? 'Sí' : 'No' ?></div>
    <div><b>Solo con máximo:</b> <?= $solo_con_max ? 'Sí' : 'No' ?></div>
    <div><b>Filas:</b> <?= count($rows) ?></div>
  </div>

  <table class="grid">
    <thead>
      <tr>
        <th>Cliente</th>
        <th>Operativa</th>
        <th>SKU</th>
        <th>Producto</th>
        <th class="right">Mínimo</th>
        <th class="right">Máximo</th>
        <th class="right">Stock</th>
        <th class="right">Reservados</th>
        <th class="right">Disponibles</th>
        <th class="right">Faltante a Mín</th>
        <th class="right">Exceso sobre Máx</th>
        <th>Estado</th>
      </tr>
    </thead>
    <tbody>
      <?php foreach ($rows as $r): ?>
        <?php
          $estadoCell = strtoupper((string)($r['estado'] ?? ''));
          $cls = 'badge';
          if ($estadoCell === 'BAJO')  $cls .= ' b-low';
          elseif ($estadoCell === 'SOBRE') $cls .= ' b-hi';
          else $cls .= ' b-ok';
        ?>
        <tr>
          <td><?= htmlspecialchars((string)($r['cliente'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($r['operativa'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($r['sku'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($r['denominacion'] ?? '')) ?></td>
          <td class="right"><?= nf($r['minimo']) ?></td>
          <td class="right"><?= $r['maximo'] !== null ? nf($r['maximo']) : '—' ?></td>
          <td class="right"><?= nf($r['stock']) ?></td>
          <td class="right"><?= nf($r['reservados']) ?></td>
          <td class="right"><?= nf($r['disponibles']) ?></td>
          <td class="right"><?= nf($r['faltante_min']) ?></td>
          <td class="right"><?= nf($r['exceso_max']) ?></td>
          <td><span class="<?= $cls ?>"><?= htmlspecialchars($estadoCell ?: '-') ?></span></td>
        </tr>
      <?php endforeach; ?>
    </tbody>
    <tfoot>
      <tr>
        <th colspan="4" class="right">Totales</th>
        <th class="right"><?= nf($tot['minimo']) ?></th>
        <th class="right"><?= $hasMax ? nf($tot['maximo']) : '—' ?></th>
        <th class="right"><?= nf($tot['stock']) ?></th>
        <th class="right"><?= nf($tot['reservados']) ?></th>
        <th class="right"><?= nf($tot['disponibles']) ?></th>
        <th class="right"><?= nf($tot['faltante_min']) ?></th>
        <th class="right"><?= nf($tot['exceso_max']) ?></th>
        <th></th>
      </tr>
    </tfoot>
  </table>
  <?php
  $html = ob_get_clean();

  $mpdf->SetTitle($title);
  $mpdf->SetAuthor((string)env('APP_NAME','SOL'));
  $mpdf->WriteHTML($html);
  $mpdf->Output('inventario_min_max.pdf', \Mpdf\Output\Destination::INLINE);

} catch (Throwable $e) {
  http_response_code(500);
  header('Content-Type: text/plain; charset=utf-8');
  echo (env('APP_ENV')==='local') ? ("Error generando PDF: ".$e->getMessage()) : "No se pudo generar el PDF";
}
