<?php

declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);

require_once $ROOT . '/vendor/Editor/lib/DataTables.php';

$inventarioHelperPath = $ROOT . '/app/Support/InventarioReport.php';
if (!is_file($inventarioHelperPath)) {
    $altCandidates = [
        $ROOT . '/app/support/InventarioReport.php',
        $ROOT . '/app/Support/inventarioreport.php',
        $ROOT . '/app/support/inventarioreport.php',
    ];
    foreach ($altCandidates as $candidate) {
        if (is_file($candidate)) {
            $inventarioHelperPath = $candidate;
            break;
        }
    }

    if (!is_file($inventarioHelperPath) && is_dir($ROOT . '/app')) {
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($ROOT . '/app', FilesystemIterator::SKIP_DOTS)
        );
        foreach ($iterator as $fileInfo) {
            if (!$fileInfo->isFile()) {
                continue;
            }
            if (strcasecmp($fileInfo->getFilename(), 'InventarioReport.php') === 0) {
                $inventarioHelperPath = $fileInfo->getPathname();
                break;
            }
        }
    }
}

if (!is_file($inventarioHelperPath)) {
    throw new RuntimeException('No se encontró el helper InventarioReport.php.');
}

require_once $inventarioHelperPath;

use DataTables\Database;

try {
    global $db;

    if (!isset($db) || !$db instanceof Database) {
        throw new RuntimeException('No se pudo inicializar la conexión a la base de datos.');
    }

    $pdo = $db->resource();

    $request = $_POST;
    if (!$request) {
        $request = $_GET;
    }

    $filters = [
        'desde' => isset($request['desde']) ? (string) $request['desde'] : '',
        'hasta' => isset($request['hasta']) ? (string) $request['hasta'] : '',
    ];

    $dataset = inventario_movimientos_dataset($pdo, $filters);
    if (($dataset['ok'] ?? false) !== true) {
        throw new RuntimeException($dataset['error'] ?? 'No se pudo obtener movimientos.');
    }

    $allRows = $dataset['rows'] ?? [];

    $draw   = isset($request['draw']) ? (int) $request['draw'] : 0;
    $start  = isset($request['start']) ? max(0, (int) $request['start']) : 0;
    $length = isset($request['length']) ? (int) $request['length'] : -1;

    $searchValue = '';
    if (isset($request['search']) && is_array($request['search'])) {
        $searchValue = (string) ($request['search']['value'] ?? '');
    }

    $orders = [];
    if (isset($request['order']) && is_array($request['order'])) {
        $orders = $request['order'];
    }

    $columnMap = [
        0  => null,
        1  => 'codigo',
        2  => 'descripcion',
        3  => 'cajas_por_pallet',
        4  => 'unidades_por_caja',
        5  => 'saldo_inicial_pallets',
        6  => 'saldo_inicial_cajas',
        7  => 'saldo_inicial_unidades_sueltas',
        8  => 'saldo_inicial',
        9  => 'ingresos_pallets',
        10 => 'ingresos_cajas',
        11 => 'ingresos_unidades_sueltas',
        12 => 'ingresos',
        13 => 'salidas_pallets',
        14 => 'salidas_cajas',
        15 => 'salidas_unidades_sueltas',
        16 => 'salidas',
        17 => 'saldo_final_pallets',
        18 => 'saldo_final_cajas',
        19 => 'saldo_final_unidades_sueltas',
        20 => 'saldo_final',
        21 => 'inventario_fisico_pallets',
        22 => 'inventario_fisico_cajas',
        23 => 'inventario_fisico_unidades_sueltas',
        24 => 'inventario_fisico',
        25 => 'diferencia_unidades',
        26 => 'valor_unidades',
        27 => 'valor_diferencia',
    ];

    $numericFields = [
        'cajas_por_pallet',
        'unidades_por_caja',
        'saldo_inicial_pallets',
        'saldo_inicial_cajas',
        'saldo_inicial_unidades_sueltas',
        'saldo_inicial',
        'ingresos_pallets',
        'ingresos_cajas',
        'ingresos_unidades_sueltas',
        'ingresos',
        'salidas_pallets',
        'salidas_cajas',
        'salidas_unidades_sueltas',
        'salidas',
        'saldo_final_pallets',
        'saldo_final_cajas',
        'saldo_final_unidades_sueltas',
        'saldo_final',
        'inventario_fisico_pallets',
        'inventario_fisico_cajas',
        'inventario_fisico_unidades_sueltas',
        'inventario_fisico',
        'diferencia_unidades',
        'valor_unidades',
        'valor_diferencia',
    ];

    $searchableFields = ['codigo', 'descripcion'];

    $filteredRows = inventario_filter_rows($allRows, $searchValue, $searchableFields);
    $recordsTotal = count($allRows);
    $recordsFiltered = count($filteredRows);

    $orderedRows = inventario_order_rows($filteredRows, $orders, $columnMap, $numericFields);
    $pageRows = inventario_slice_rows($orderedRows, $start, $length);

    $filteredTotals = inventario_movimientos_compute_totals($filteredRows);
    $grandTotals = $dataset['totals'] ?? inventario_movimientos_compute_totals($allRows);

    $meta = $dataset['meta'] ?? [];
    $metaFilters = [
        'desde' => $filters['desde'] ?? '',
        'hasta' => $filters['hasta'] ?? '',
    ];

    if (isset($dataset['meta']['filters']) && is_array($dataset['meta']['filters'])) {
        foreach (['desde', 'hasta'] as $key) {
            if (isset($dataset['meta']['filters'][$key]) && $dataset['meta']['filters'][$key] !== null) {
                $metaFilters[$key] = (string) $dataset['meta']['filters'][$key];
            }
        }
    }

    $meta['filters'] = $metaFilters;

    $response = [
        'ok'              => true,
        'draw'            => $draw,
        'recordsTotal'    => $recordsTotal,
        'recordsFiltered' => $recordsFiltered,
        'data'            => $pageRows,
        'rows'            => $pageRows,
        'totals'          => $filteredTotals,
        'grandTotals'     => $grandTotals,
        'meta'            => $meta,
    ];

    $jsonFlags = JSON_UNESCAPED_UNICODE | JSON_PRESERVE_ZERO_FRACTION;
    echo json_encode($response, $jsonFlags);
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode([
        'ok'     => false,
        'error'  => 'Error cargando movimientos de inventario',
        'detail' => inventario_is_local_env() ? $e->getMessage() : null,
    ], JSON_UNESCAPED_UNICODE);
}
