<?php

declare(strict_types=1);

header('Content-Type: application/pdf');

$ROOT = dirname(__DIR__, 2);

require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

$autoloader = $ROOT . '/vendor/autoload.php';
if (!is_file($autoloader)) {
    http_response_code(500);
    echo 'Falta vendor/autoload.php. Ejecuta: composer install';
    exit;
}
require_once $autoloader;

if (!class_exists(\Mpdf\Mpdf::class)) {
    http_response_code(500);
    echo 'No está instalado mpdf/mpdf. Ejecuta: composer require mpdf/mpdf';
    exit;
}

$inventarioHelperPath = $ROOT . '/app/Support/InventarioReport.php';
if (!is_file($inventarioHelperPath)) {
    $altCandidates = [
        $ROOT . '/app/support/InventarioReport.php',
        $ROOT . '/app/Support/inventarioreport.php',
        $ROOT . '/app/support/inventarioreport.php',
    ];
    foreach ($altCandidates as $candidate) {
        if (is_file($candidate)) {
            $inventarioHelperPath = $candidate;
            break;
        }
    }

    if (!is_file($inventarioHelperPath) && is_dir($ROOT . '/app')) {
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($ROOT . '/app', FilesystemIterator::SKIP_DOTS)
        );
        foreach ($iterator as $fileInfo) {
            if (!$fileInfo->isFile()) {
                continue;
            }
            if (strcasecmp($fileInfo->getFilename(), 'InventarioReport.php') === 0) {
                $inventarioHelperPath = $fileInfo->getPathname();
                break;
            }
        }
    }
}

if (!is_file($inventarioHelperPath)) {
    http_response_code(500);
    echo 'No se encontró el helper InventarioReport.php.';
    exit;
}

require_once $inventarioHelperPath;

function nf0($value): string
{
    if ($value === null || $value === '') {
        return '0';
    }
    return number_format((float) $value, 0, ',', '.');
}

function nf2($value): string
{
    if ($value === null || $value === '') {
        return '0,00';
    }
    return number_format((float) $value, 2, ',', '.');
}

function dash_if_null($value): string
{
    if ($value === null || $value === '') {
        return '—';
    }
    return (string) $value;
}

function nf0_nullable($value): string
{
    if ($value === null || $value === '') {
        return '—';
    }
    return nf0($value);
}

function nf2_nullable($value): string
{
    if ($value === null || $value === '') {
        return '—';
    }
    return nf2($value);
}

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $filters = [
        'desde' => isset($_GET['desde']) ? (string) $_GET['desde'] : '',
        'hasta' => isset($_GET['hasta']) ? (string) $_GET['hasta'] : '',
    ];

    $dataset = inventario_movimientos_dataset($pdo, $filters);
    if (($dataset['ok'] ?? false) !== true) {
        throw new RuntimeException($dataset['error'] ?? 'No se pudo obtener el dataset de movimientos.');
    }

    $rows = $dataset['rows'] ?? [];
    $totals = $dataset['totals'] ?? inventario_movimientos_compute_totals($rows);

    $metaFilters = $dataset['meta']['filters'] ?? ['desde' => '', 'hasta' => ''];
    $filterDesde = (string) ($metaFilters['desde'] ?? '');
    $filterHasta = (string) ($metaFilters['hasta'] ?? '');

    $mpdf = new \Mpdf\Mpdf([
        'format' => 'A4',
        'margin_left'   => 10,
        'margin_right'  => 10,
        'margin_top'    => 15,
        'margin_bottom' => 12,
    ]);

    $styles = <<<HTML
<style>
  body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 11px; }
  h1 { font-size: 18px; margin: 0 0 8px 0; }
  .muted { color: #666; }
  .filters { display: flex; flex-wrap: wrap; gap: 12px; margin: 6px 0 10px 0; }
  .filters div { min-width: 160px; }
  table.report { width: 100%; border-collapse: collapse; }
  table.report th,
  table.report td { border: 0.6px solid #999; padding: 4px 6px; }
  table.report th { background: #f1f1f1; }
  .right { text-align: right; }
  .small { font-size: 10px; }
    .positive { color: #1a7f37; }
    .negative { color: #c03221; }
    .fw-semibold { font-weight: 600; }
</style>
HTML;

    ob_start();
    ?>
    <?= $styles ?>
    <h1>Movimientos de inventario</h1>
    <div class="muted small">Generado: <?= date('Y-m-d H:i') ?> · Sistema: <?= htmlspecialchars((string) env('APP_NAME', 'SOL')) ?></div>

    <div class="filters">
        <div><strong>Desde:</strong> <?= $filterDesde !== '' ? htmlspecialchars($filterDesde) : '—' ?></div>
        <div><strong>Hasta:</strong> <?= $filterHasta !== '' ? htmlspecialchars($filterHasta) : '—' ?></div>
        <div><strong>Filas:</strong> <?= count($rows) ?></div>
    </div>

    <table class="report">
        <thead>
            <tr>
                <th rowspan="2">Código</th>
                <th rowspan="2">Descripción</th>
                <th colspan="2">Presentación</th>
                <th colspan="4">Saldo inicial</th>
                <th colspan="4">Ingresos</th>
                <th colspan="4">Salidas</th>
                <th colspan="4">Saldo final</th>
                <th colspan="4">Inventario físico</th>
                <th rowspan="2">Dif. unidades</th>
                <th rowspan="2">Valor unidades</th>
                <th rowspan="2">Valor diferencia</th>
            </tr>
            <tr>
                <th class="right">Cajas / pallet</th>
                <th class="right">Unid. / caja</th>
                <th class="right">Pallets</th>
                <th class="right">Cajas</th>
                <th class="right">Unid. sueltas</th>
                <th class="right">Total unid.</th>
                <th class="right">Pallets</th>
                <th class="right">Cajas</th>
                <th class="right">Unid. sueltas</th>
                <th class="right">Total unid.</th>
                <th class="right">Pallets</th>
                <th class="right">Cajas</th>
                <th class="right">Unid. sueltas</th>
                <th class="right">Total unid.</th>
                <th class="right">Pallets</th>
                <th class="right">Cajas</th>
                <th class="right">Unid. sueltas</th>
                <th class="right">Total unid.</th>
                <th class="right">Pallets</th>
                <th class="right">Cajas</th>
                <th class="right">Unid. sueltas</th>
                <th class="right">Total unid.</th>
            </tr>
        </thead>
        <tbody>
            <?php if (!$rows): ?>
                <tr>
                    <td colspan="27" class="muted" style="text-align:center;">Sin resultados para los filtros seleccionados.</td>
                </tr>
            <?php else: ?>
                <?php foreach ($rows as $row): ?>
                    <tr>
                        <td><?= htmlspecialchars((string) ($row['codigo'] ?? '')) ?></td>
                        <td><?= htmlspecialchars((string) ($row['descripcion'] ?? '')) ?></td>
                        <td class="right"><?= $row['cajas_por_pallet'] !== null ? nf2($row['cajas_por_pallet']) : '—' ?></td>
                        <td class="right"><?= $row['unidades_por_caja'] !== null ? nf0($row['unidades_por_caja']) : '—' ?></td>
                        <td class="right"><?= nf0_nullable($row['saldo_inicial_pallets'] ?? null) ?></td>
                        <td class="right"><?= nf0_nullable($row['saldo_inicial_cajas'] ?? null) ?></td>
                        <td class="right"><?= nf0_nullable($row['saldo_inicial_unidades_sueltas'] ?? null) ?></td>
                        <td class="right fw-semibold"><?= nf0($row['saldo_inicial'] ?? 0) ?></td>
                        <td class="right positive"><?= nf0_nullable($row['ingresos_pallets'] ?? null) ?></td>
                        <td class="right positive"><?= nf0_nullable($row['ingresos_cajas'] ?? null) ?></td>
                        <td class="right positive"><?= nf0_nullable($row['ingresos_unidades_sueltas'] ?? null) ?></td>
                        <td class="right positive fw-semibold"><?= nf0($row['ingresos'] ?? 0) ?></td>
                        <td class="right negative"><?= nf0_nullable($row['salidas_pallets'] ?? null) ?></td>
                        <td class="right negative"><?= nf0_nullable($row['salidas_cajas'] ?? null) ?></td>
                        <td class="right negative"><?= nf0_nullable($row['salidas_unidades_sueltas'] ?? null) ?></td>
                        <td class="right negative fw-semibold"><?= nf0($row['salidas'] ?? 0) ?></td>
                        <td class="right"><?= nf0_nullable($row['saldo_final_pallets'] ?? null) ?></td>
                        <td class="right"><?= nf0_nullable($row['saldo_final_cajas'] ?? null) ?></td>
                        <td class="right"><?= nf0_nullable($row['saldo_final_unidades_sueltas'] ?? null) ?></td>
                        <td class="right fw-semibold"><?= nf0($row['saldo_final'] ?? 0) ?></td>
                        <td class="right"><?= nf0_nullable($row['inventario_fisico_pallets'] ?? null) ?></td>
                        <td class="right"><?= nf0_nullable($row['inventario_fisico_cajas'] ?? null) ?></td>
                        <td class="right"><?= nf0_nullable($row['inventario_fisico_unidades_sueltas'] ?? null) ?></td>
                        <td class="right"><?= nf0_nullable($row['inventario_fisico'] ?? null) ?></td>
                        <td class="right"><?= nf0_nullable($row['diferencia_unidades'] ?? null) ?></td>
                        <td class="right"><?= $row['valor_unidades'] !== null ? nf2($row['valor_unidades']) : '—' ?></td>
                        <td class="right"><?= nf2_nullable($row['valor_diferencia'] ?? null) ?></td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
        <tfoot>
            <tr>
                <th colspan="4" class="right">Totales</th>
                <th class="right"><?= nf0_nullable($totals['saldo_inicial_pallets'] ?? null) ?></th>
                <th class="right"><?= nf0_nullable($totals['saldo_inicial_cajas'] ?? null) ?></th>
                <th class="right"><?= nf0_nullable($totals['saldo_inicial_unidades_sueltas'] ?? null) ?></th>
                <th class="right"><?= nf0($totals['saldo_inicial'] ?? 0) ?></th>
                <th class="right"><?= nf0_nullable($totals['ingresos_pallets'] ?? null) ?></th>
                <th class="right"><?= nf0_nullable($totals['ingresos_cajas'] ?? null) ?></th>
                <th class="right"><?= nf0_nullable($totals['ingresos_unidades_sueltas'] ?? null) ?></th>
                <th class="right"><?= nf0($totals['ingresos'] ?? 0) ?></th>
                <th class="right"><?= nf0_nullable($totals['salidas_pallets'] ?? null) ?></th>
                <th class="right"><?= nf0_nullable($totals['salidas_cajas'] ?? null) ?></th>
                <th class="right"><?= nf0_nullable($totals['salidas_unidades_sueltas'] ?? null) ?></th>
                <th class="right"><?= nf0($totals['salidas'] ?? 0) ?></th>
                <th class="right"><?= nf0_nullable($totals['saldo_final_pallets'] ?? null) ?></th>
                <th class="right"><?= nf0_nullable($totals['saldo_final_cajas'] ?? null) ?></th>
                <th class="right"><?= nf0_nullable($totals['saldo_final_unidades_sueltas'] ?? null) ?></th>
                <th class="right"><?= nf0($totals['saldo_final'] ?? 0) ?></th>
                <th class="right"><?= nf0_nullable($totals['inventario_fisico_pallets'] ?? null) ?></th>
                <th class="right"><?= nf0_nullable($totals['inventario_fisico_cajas'] ?? null) ?></th>
                <th class="right"><?= nf0_nullable($totals['inventario_fisico_unidades_sueltas'] ?? null) ?></th>
                <th class="right"><?= nf0_nullable($totals['inventario_fisico'] ?? null) ?></th>
                <th class="right"><?= nf0_nullable($totals['diferencia_unidades'] ?? null) ?></th>
                <th class="right"><?= nf2($totals['valor_unidades'] ?? 0) ?></th>
                <th class="right"><?= nf2_nullable($totals['valor_diferencia'] ?? null) ?></th>
            </tr>
        </tfoot>
    </table>
    <?php
    $html = ob_get_clean();

    $mpdf->SetTitle('Movimientos de inventario');
    $mpdf->SetAuthor((string) env('APP_NAME', 'SOL'));
    $mpdf->WriteHTML($html);
    $mpdf->Output('movimientos_inventario.pdf', \Mpdf\Output\Destination::INLINE);
} catch (Throwable $e) {
    http_response_code(500);
    header('Content-Type: text/plain; charset=utf-8');
    echo (env('APP_ENV') === 'local') ? ('Error generando PDF: ' . $e->getMessage()) : 'No se pudo generar el PDF';
}
