<?php
declare(strict_types=1);

/**
 * API: Inventario - Stock actual por producto
 * Ruta: api/inventario/stock_producto.php
 * Método: GET
 *
 * Parámetros:
 * - q: string (SKU exacto o parte de nombre/sku); si es numérico, también intenta producto_id
 * - producto_id: int (opcional)
 * - deposito_code | deposito_id: para filtrar (opcional; por defecto DEP1 si existe)
 *
 * Respuesta ok:true
 * {
 *   ok: true,
 *   producto: { id, sku, denominacion },
 *   deposito: { id, code, nombre } | null,
 *   uc_per_uv: int,
 *   totals: { total_uv, total_uc, total_uv_equiv_from_uc },
 *   rows: [ { ambiente, pos_code, lote_codigo, venc, qty_uv, qty_uc, uv_equiv_from_uc } ]
 * }
 *
 * Respuesta ambiguo/errores:
 * { ok:false, error:'...', matches:[ {id,sku,denominacion}, ... ] }
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/db.php';

function outJson(array $p, int $c=200){ http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }

try {
  $q = isset($_GET['q']) ? trim((string)$_GET['q']) : '';
  $prodIdIn = isset($_GET['producto_id']) ? (int)$_GET['producto_id'] : 0;
  $depCodeIn = isset($_GET['deposito_code']) ? trim((string)$_GET['deposito_code']) : '';
  $depIdIn = isset($_GET['deposito_id']) ? (int)$_GET['deposito_id'] : 0;

  if ($prodIdIn <= 0 && $q === '') {
    outJson(['ok'=>false,'error'=>'Parámetro q o producto_id requerido'], 422);
  }

  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $pdo->exec('SET NAMES utf8mb4');

  // Resolver producto
  $prod = null;
  if ($prodIdIn > 0) {
    $st = $pdo->prepare('SELECT id, sku, denominacion FROM para_productos WHERE id=? LIMIT 1');
    $st->execute([$prodIdIn]);
    $prod = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  } else {
    $cands = [];
    if ($q !== '') {
      // Si es numérico, probá como id primero
      if (ctype_digit($q)) {
        $st = $pdo->prepare('SELECT id, sku, denominacion FROM para_productos WHERE id=? LIMIT 1');
        $st->execute([(int)$q]);
        $row = $st->fetch(PDO::FETCH_ASSOC);
        if ($row) $cands[] = $row;
      }
      // SKU exacto
      $st = $pdo->prepare('SELECT id, sku, denominacion FROM para_productos WHERE sku=? LIMIT 1');
      $st->execute([$q]);
      $row = $st->fetch(PDO::FETCH_ASSOC);
      if ($row) $cands[] = $row;
      // Búsqueda LIKE por sku/denominación (top 5) si todavía no hay candidato
      if (!$cands) {
        $st = $pdo->prepare("SELECT id, sku, denominacion FROM para_productos WHERE sku LIKE ? OR denominacion LIKE ? ORDER BY denominacion ASC LIMIT 5");
        $like = '%' . $q . '%';
        $st->execute([$like, $like]);
        $cands = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
      }
    }
    if (!$cands) outJson(['ok'=>false,'error'=>'Producto no encontrado']);
    if (count($cands) > 1) outJson(['ok'=>false,'error'=>'Búsqueda ambigua, elegí un producto.', 'matches'=>$cands], 422);
    $prod = $cands[0];
  }

  if (!$prod) outJson(['ok'=>false,'error'=>'Producto no encontrado'], 404);
  $prodId = (int)$prod['id'];

  // Resolver depósito (opcional) -> si no envía, usar DEP1 si existe
  $dep = null; $depId = 0;
  if ($depIdIn > 0) {
    $st = $pdo->prepare('SELECT id, code, nombre FROM wh_deposito WHERE id=? LIMIT 1');
    $st->execute([$depIdIn]);
    $dep = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  } elseif ($depCodeIn !== '') {
    $st = $pdo->prepare('SELECT id, code, nombre FROM wh_deposito WHERE code=? LIMIT 1');
    $st->execute([$depCodeIn]);
    $dep = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  } else {
    $st = $pdo->query("SELECT id, code, nombre FROM wh_deposito WHERE code='DEP1' LIMIT 1");
    $dep = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  }
  if ($dep) $depId = (int)$dep['id'];

  // Factor UC por UV
  $ucPerUv = 0;
  try {
    $st = $pdo->prepare('SELECT COALESCE(MAX(unidades_por_uv),0) FROM para_producto_pack WHERE producto_id=?');
    $st->execute([$prodId]);
    $ucPerUv = (int)($st->fetchColumn() ?: 0);
  } catch (Throwable $e) { $ucPerUv = 0; }

  // Query de stock por filas
  $params = [$prodId];
  $depSql = '';
  if ($depId > 0) { $depSql = ' AND s.deposito_id = ? '; $params[] = $depId; }
  $sql = "SELECT a.code AS ambiente, p.code AS pos_code, l.codigo AS lote_codigo, l.fecha_vencimiento AS venc,
                 s.qty_uv, s.qty_uc
            FROM wh_stock s
            JOIN wh_posicion p ON p.id=s.posicion_id
            JOIN wh_ambiente a ON a.id=p.ambiente_id
            LEFT JOIN wh_lote l ON l.id=s.lote_id
           WHERE s.producto_id=? $depSql AND (s.qty_uv>0 OR s.qty_uc>0)
           ORDER BY a.code, p.code, COALESCE(l.fecha_vencimiento,'9999-12-31') ASC";
  $st = $pdo->prepare($sql);
  $st->execute($params);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

  $outRows = [];
  $totUv = 0; $totUc = 0; $totUvEq = 0;
  foreach ($rows as $r) {
    $uv = (int)($r['qty_uv'] ?? 0);
    $uc = (int)($r['qty_uc'] ?? 0);
    $uvEq = ($ucPerUv > 0) ? intdiv($uc, $ucPerUv) : 0;
    $totUv += $uv; $totUc += $uc; $totUvEq += $uvEq;
    $outRows[] = [
      'ambiente' => (string)($r['ambiente'] ?? ''),
      'pos_code' => (string)($r['pos_code'] ?? ''),
      'lote_codigo' => $r['lote_codigo'] !== null ? (string)$r['lote_codigo'] : null,
      'venc' => $r['venc'] !== null ? (string)$r['venc'] : null,
      'qty_uv' => $uv,
      'qty_uc' => $uc,
      'uv_equiv_from_uc' => $uvEq,
    ];
  }

  outJson([
    'ok' => true,
    'producto' => [ 'id'=>$prodId, 'sku'=>$prod['sku'], 'denominacion'=>$prod['denominacion'] ],
    'deposito' => $dep,
    'uc_per_uv' => $ucPerUv,
    'totals' => [ 'total_uv'=>$totUv, 'total_uc'=>$totUc, 'total_uv_equiv_from_uc'=>$totUvEq ],
    'rows' => $outRows
  ]);

} catch (Throwable $e) {
  outJson(['ok'=>false,'error'=>'Error en consulta de stock','message'=>$e->getMessage()], 500);
}
