<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  // ---------------------------
  // Helpers
  // ---------------------------
  $hasTable = function (PDO $pdo, string $t): bool {
    static $cache = [];
    if (array_key_exists($t, $cache)) return $cache[$t];
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
    $st->execute([$t]);
    return $cache[$t] = ((int)$st->fetchColumn() > 0);
  };

  // ---------------------------
  // Meta endpoints (combos)
  // ---------------------------
  if (isset($_GET['meta'])) {
    $meta = (string)$_GET['meta'];
    if ($meta === 'clientes') {
      $rows = $pdo->query("
        SELECT id, razon_social
        FROM para_clientes
        WHERE deleted_at IS NULL
        ORDER BY razon_social
      ")->fetchAll(PDO::FETCH_ASSOC);
      echo json_encode(['clientes' => $rows], JSON_UNESCAPED_UNICODE);
      exit;
    }
    if ($meta === 'operativas') {
      $rows = $pdo->query("
        SELECT id, nombre
        FROM sys_operativas
        WHERE deleted_at IS NULL
        ORDER BY nombre
      ")->fetchAll(PDO::FETCH_ASSOC);
      echo json_encode(['operativas' => $rows], JSON_UNESCAPED_UNICODE);
      exit;
    }
  }

  // ---------------------------
  // DataTables params
  // ---------------------------
  $draw   = (int)($_GET['draw']   ?? 1);
  $start  = (int)($_GET['start']  ?? 0);
  $length = (int)($_GET['length'] ?? 25);

  // ---------------------------
  // Filtros
  // ---------------------------
  $cliente_id   = trim((string)($_GET['cliente_id']   ?? ''));
  $operativa_id = trim((string)($_GET['operativa_id'] ?? ''));
  $q_producto   = trim((string)($_GET['q_producto']   ?? ''));
  $q_pallet     = trim((string)($_GET['q_pallet']     ?? ''));
  $q_ubicacion  = trim((string)($_GET['q_ubicacion']  ?? ''));
  $rack         = trim((string)($_GET['rack']         ?? ''));
  $columna      = trim((string)($_GET['columna']      ?? ''));
  $nivel        = trim((string)($_GET['nivel']        ?? ''));
  $fondo        = trim((string)($_GET['fondo']        ?? ''));
  $estado_pos   = strtolower(trim((string)($_GET['estado_pos'] ?? ''))); // ocupada|libre|''

  // ---------------------------
  // SQL base (una fila por posición) con detección de esquema
  // ---------------------------
  $useNew = $hasTable($pdo, 'wh_positions') && $hasTable($pdo, 'wh_position_occupancy')
         && $hasTable($pdo, 'wh_pallets') && $hasTable($pdo, 'wh_pallet_items');

  if ($useNew) {
    // Warehouse-core: ocupación vigente por posición
    $sql = "
      SELECT
        pos.id                                                AS position_id,
        pos.rack,
        pos.columna,
        pos.nivel,
        pos.fondo,
        CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) AS ubicacion,

        -- ocupación vigente
        occ.id                                                AS occ_id,
        pa.codigo                                             AS pallet_codigo,

        -- agregados por pallet (si ocupado)
        COALESCE(SUM(COALESCE(it.uv_cajas,0)), 0)                                    AS cajas,
        COALESCE(SUM(COALESCE(it.uc_total_cache,
                  (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
                )), 0)                                                               AS unidades,
        COALESCE(SUM(COALESCE(it.uc_total_cache,
                  (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
                )), 0)                                                               AS stock,
        COALESCE(SUM(CASE WHEN it.estado = 'RESERVADO'
                  THEN COALESCE(it.uc_total_cache,
                        (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
                      )
                  ELSE 0 END), 0)                                                    AS reservados,
        ( COALESCE(SUM(COALESCE(it.uc_total_cache,
                  (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
                )), 0)
          - COALESCE(SUM(CASE WHEN it.estado = 'RESERVADO'
                  THEN COALESCE(it.uc_total_cache,
                        (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
                      )
                  ELSE 0 END), 0)
        )                                                                              AS disponibles,

        -- cliente/operativa/producto principal (solo si el pallet tiene 1 producto)
        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(c.razon_social) END              AS cliente,
        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(opv.nombre) END                  AS operativa,
        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(pr.sku) END                      AS sku,
        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(pr.denominacion) END             AS denominacion,
        CASE WHEN COUNT(DISTINCT pr.id) > 1 THEN 'MIX' END                            AS producto_mix,

        -- estado de la posición
        CASE WHEN occ.id IS NULL THEN 'libre' ELSE 'ocupada' END                      AS estado_pos
      FROM wh_positions pos
      LEFT JOIN wh_position_occupancy occ ON occ.position_id = pos.id AND occ.hasta IS NULL
      LEFT JOIN wh_pallets pa ON pa.id = occ.pallet_id
      LEFT JOIN wh_pallet_items it ON it.pallet_id = pa.id
      LEFT JOIN para_productos pr ON pr.id = it.producto_id
      LEFT JOIN para_clientes c ON c.id = pr.cliente_id
      LEFT JOIN sys_operativas opv ON opv.id = pr.operativa_id
      WHERE 1=1
    ";
  } else {
    // Legacy: ocupar "ocupada" si hay stock en la posición; sin pallets/occ
    $sql = "
      SELECT
        pos.id                                                AS position_id,
        pos.rack,
        pos.columna,
        pos.nivel,
        pos.fondo,
        CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) AS ubicacion,

        NULL                                                  AS occ_id,
        ''                                                    AS pallet_codigo,

        COALESCE(SUM(COALESCE(s.qty_uv,0)), 0)                                        AS cajas,
        COALESCE(SUM(COALESCE(s.qty_uc,0)), 0)                                         AS unidades,
        COALESCE(SUM(COALESCE(s.qty_uc,0)), 0)                                         AS stock,
        0                                                                              AS reservados,
        COALESCE(SUM(COALESCE(s.qty_uc,0)), 0)                                         AS disponibles,

        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(c.razon_social) END              AS cliente,
        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(opv.nombre) END                  AS operativa,
        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(pr.sku) END                      AS sku,
        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(pr.denominacion) END             AS denominacion,
        CASE WHEN COUNT(DISTINCT pr.id) > 1 THEN 'MIX' END                            AS producto_mix,

        CASE WHEN COALESCE(SUM(COALESCE(s.qty_uc,0)),0) > 0 THEN 'ocupada' ELSE 'libre' END AS estado_pos
      FROM wh_posicion pos
      LEFT JOIN wh_stock s       ON s.posicion_id = pos.id
      LEFT JOIN para_productos pr ON pr.id = s.producto_id
      LEFT JOIN para_clientes c   ON c.id = pr.cliente_id
      LEFT JOIN sys_operativas opv ON opv.id = pr.operativa_id
      WHERE 1=1
    ";
  }

  $params = [];

  // Filtros “geométricos” de posición
  if ($rack !== '') {
    $sql .= " AND pos.rack LIKE :frack";
    $params[':frack'] = "%{$rack}%";
  }
  if ($columna !== '') {
    $sql .= " AND CAST(pos.columna AS CHAR) LIKE :fcol";
    $params[':fcol'] = "%{$columna}%";
  }
  if ($nivel !== '') {
    $sql .= " AND CAST(pos.nivel AS CHAR) LIKE :fniv";
    $params[':fniv'] = "%{$nivel}%";
  }
  if ($fondo !== '') {
    $sql .= " AND CAST(pos.fondo AS CHAR) LIKE :ffon";
    $params[':ffon'] = "%{$fondo}%";
  }
  if ($q_ubicacion !== '') {
    $sql .= " AND CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) LIKE :qub";
    $params[':qub'] = "%{$q_ubicacion}%";
  }

  // Filtros que aplican sobre ocupadas (se ignorarán para libres) y generales
  if ($q_pallet !== '') {
    if ($useNew) {
      $sql .= " AND pa.codigo LIKE :qpallet";
    } else {
      // en legacy no hay código pallet; intentar por id numérico si existe
      $sql .= " AND CAST(s.pallet_id AS CHAR) LIKE :qpallet";
    }
    $params[':qpallet'] = "%{$q_pallet}%";
  }
  if ($cliente_id !== '') {
    $sql .= " AND pr.cliente_id = :cliente_id";
    $params[':cliente_id'] = $cliente_id;
  }
  if ($operativa_id !== '') {
    $sql .= " AND pr.operativa_id = :operativa_id";
    $params[':operativa_id'] = $operativa_id;
  }
  if ($q_producto !== '') {
    $sql .= " AND (pr.sku LIKE :qp OR pr.denominacion LIKE :qp)";
    $params[':qp'] = "%{$q_producto}%";
  }

  if ($useNew) {
    $sql .= "
      GROUP BY
        pos.id, pos.rack, pos.columna, pos.nivel, pos.fondo, occ.id, pa.codigo
    ";
  } else {
    $sql .= "
      GROUP BY
        pos.id, pos.rack, pos.columna, pos.nivel, pos.fondo
    ";
  }

  // Filtro por estado_pos (después del GROUP BY para no romper agregaciones)
  if ($estado_pos === 'ocupada') {
    $sql .= " HAVING estado_pos = 'ocupada'";
  } elseif ($estado_pos === 'libre') {
    $sql .= " HAVING estado_pos = 'libre'";
  }

  // Orden natural por ubicación
  $sql .= " ORDER BY pos.rack ASC, pos.columna ASC, pos.nivel ASC, pos.fondo ASC";

  // ---------------------------
  // Conteo total
  // ---------------------------
  $stmtCount = $pdo->prepare("SELECT COUNT(*) FROM ($sql) t");
  foreach ($params as $k=>$v) $stmtCount->bindValue($k, $v);
  $stmtCount->execute();
  $recordsTotal = (int)$stmtCount->fetchColumn();

  // ---------------------------
  // Paginación
  // ---------------------------
  $sqlPaged = $sql . " LIMIT :start, :length";
  $stmt = $pdo->prepare($sqlPaged);
  foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
  $stmt->bindValue(':start', $start, PDO::PARAM_INT);
  $stmt->bindValue(':length', $length, PDO::PARAM_INT);
  $stmt->execute();
  $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];

  // Post-procesar “producto principal”
  foreach ($rows as &$r) {
    if (!empty($r['producto_mix'])) {
      $r['sku'] = null;
      $r['denominacion'] = 'MIX';
    }
    // Si está libre, normalizar campos numéricos y de pallet
    if (($r['estado_pos'] ?? '') === 'libre') {
      $r['pallet_codigo'] = '';
      $r['cliente'] = '';
      $r['operativa'] = '';
      $r['sku'] = '';
      $r['denominacion'] = '';
      $r['cajas'] = 0;
      $r['unidades'] = 0;
      $r['stock'] = 0;
      $r['reservados'] = 0;
      $r['disponibles'] = 0;
    }
  }
  unset($r);

  echo json_encode([
    'draw'            => $draw,
    'recordsTotal'    => $recordsTotal,
    'recordsFiltered' => $recordsTotal,
    'data'            => $rows,
  ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'ok'    => false,
    'error' => 'Error cargando ubicaciones',
    'msg'   => (env('APP_ENV')==='local') ? $e->getMessage() : ''
  ], JSON_UNESCAPED_UNICODE);
}
