<?php
declare(strict_types=1);

header('Content-Type: application/pdf');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

/* ======================
 * Dependencias (mPDF)
 * ====================== */
$aut = $ROOT . '/vendor/autoload.php';
if (!is_file($aut)) {
  http_response_code(500);
  echo "Falta vendor/autoload.php. Ejecuta: composer install";
  exit;
}
require_once $aut;
if (!class_exists(\Mpdf\Mpdf::class)) {
  http_response_code(500);
  echo "No está instalado mpdf/mpdf. Ejecuta: composer require mpdf/mpdf";
  exit;
}

/* ======================
 * Helpers
 * ====================== */
function nf($n, int $dec = 0): string {
  if ($n === null || $n === '') return '0';
  return number_format((float)$n, $dec, ',', '.'); // 1.234.567,89
}

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  // helpers
  $hasTable = function (PDO $pdo, string $t): bool {
    static $cache = [];
    if (array_key_exists($t, $cache)) return $cache[$t];
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
    $st->execute([$t]);
    return $cache[$t] = ((int)$st->fetchColumn() > 0);
  };

  // -------- Filtros (idénticos al listado) --------
  $cliente_id   = trim((string)($_GET['cliente_id']   ?? ''));
  $operativa_id = trim((string)($_GET['operativa_id'] ?? ''));
  $q_producto   = trim((string)($_GET['q_producto']   ?? ''));
  $q_pallet     = trim((string)($_GET['q_pallet']     ?? ''));
  $q_ubicacion  = trim((string)($_GET['q_ubicacion']  ?? ''));
  $rack         = trim((string)($_GET['rack']         ?? ''));
  $columna      = trim((string)($_GET['columna']      ?? ''));
  $nivel        = trim((string)($_GET['nivel']        ?? ''));
  $fondo        = trim((string)($_GET['fondo']        ?? ''));
  $estado_pos   = strtolower(trim((string)($_GET['estado_pos'] ?? ''))); // ocupada|libre|''

  // Labels
  $clienteLabel   = $cliente_id   !== '' ? ($pdo->query("SELECT razon_social FROM para_clientes WHERE id=".(int)$cliente_id)->fetchColumn() ?: '#'.$cliente_id) : '(Todos)';
  $operativaLabel = $operativa_id !== '' ? ($pdo->query("SELECT nombre FROM sys_operativas WHERE id=".(int)$operativa_id)->fetchColumn() ?: '#'.$operativa_id) : '(Todas)';

  // -------- SQL (mismo origen que /api/inventario/ubicacion.php, sin LIMIT) --------
  $useNew = $hasTable($pdo,'wh_positions') && $hasTable($pdo,'wh_position_occupancy')
         && $hasTable($pdo,'wh_pallets') && $hasTable($pdo,'wh_pallet_items');

  if ($useNew) {
    $sql = "
      SELECT
        pos.id AS position_id,
        pos.rack,
        pos.columna,
        pos.nivel,
        pos.fondo,
        CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) AS ubicacion,
        occ.id AS occ_id,
        pa.codigo AS pallet_codigo,

        COALESCE(SUM(COALESCE(it.uv_cajas,0)), 0) AS cajas,
        COALESCE(SUM(COALESCE(it.uc_total_cache,
              (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
            )), 0) AS unidades,
        COALESCE(SUM(COALESCE(it.uc_total_cache,
              (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
            )), 0) AS stock,
        COALESCE(SUM(CASE WHEN it.estado='RESERVADO'
              THEN COALESCE(it.uc_total_cache,
                (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
              )
              ELSE 0 END), 0) AS reservados,
        (
          COALESCE(SUM(COALESCE(it.uc_total_cache,
            (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
          )), 0)
          - COALESCE(SUM(CASE WHEN it.estado='RESERVADO'
            THEN COALESCE(it.uc_total_cache,
              (COALESCE(it.uc_por_caja,0)*COALESCE(it.uv_cajas,0) + COALESCE(it.uc_sueltas,0))
            )
            ELSE 0 END), 0)
        ) AS disponibles,

        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(c.razon_social) END  AS cliente,
        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(opv.nombre) END      AS operativa,
        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(pr.sku) END          AS sku,
        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(pr.denominacion) END AS denominacion,
        CASE WHEN COUNT(DISTINCT pr.id) > 1 THEN 'MIX' END                AS producto_mix,

        CASE WHEN occ.id IS NULL THEN 'libre' ELSE 'ocupada' END          AS estado_pos
      FROM wh_positions pos
      LEFT JOIN wh_position_occupancy occ ON occ.position_id = pos.id AND occ.hasta IS NULL
      LEFT JOIN wh_pallets pa ON pa.id = occ.pallet_id
      LEFT JOIN wh_pallet_items it ON it.pallet_id = pa.id
      LEFT JOIN para_productos pr ON pr.id = it.producto_id
      LEFT JOIN para_clientes c ON c.id = pr.cliente_id
      LEFT JOIN sys_operativas opv ON opv.id = pr.operativa_id
      WHERE 1=1
    ";
  } else {
    $sql = "
      SELECT
        pos.id AS position_id,
        pos.rack,
        pos.columna,
        pos.nivel,
        pos.fondo,
        CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) AS ubicacion,
        NULL AS occ_id,
        ''   AS pallet_codigo,

        COALESCE(SUM(COALESCE(s.qty_uv,0)), 0) AS cajas,
        COALESCE(SUM(COALESCE(s.qty_uc,0)), 0) AS unidades,
        COALESCE(SUM(COALESCE(s.qty_uc,0)), 0) AS stock,
        0 AS reservados,
        COALESCE(SUM(COALESCE(s.qty_uc,0)), 0) AS disponibles,

        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(c.razon_social) END  AS cliente,
        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(opv.nombre) END      AS operativa,
        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(pr.sku) END          AS sku,
        CASE WHEN COUNT(DISTINCT pr.id) = 1 THEN MAX(pr.denominacion) END AS denominacion,
        CASE WHEN COUNT(DISTINCT pr.id) > 1 THEN 'MIX' END                AS producto_mix,

        CASE WHEN COALESCE(SUM(COALESCE(s.qty_uc,0)),0) > 0 THEN 'ocupada' ELSE 'libre' END AS estado_pos
      FROM wh_posicion pos
      LEFT JOIN wh_stock s        ON s.posicion_id = pos.id
      LEFT JOIN para_productos pr ON pr.id = s.producto_id
      LEFT JOIN para_clientes c   ON c.id = pr.cliente_id
      LEFT JOIN sys_operativas opv ON opv.id = pr.operativa_id
      WHERE 1=1
    ";
  }

  $params = [];

  if ($rack !== '') {
    $sql .= " AND pos.rack LIKE :frack";
    $params[':frack'] = "%{$rack}%";
  }
  if ($columna !== '') {
    $sql .= " AND CAST(pos.columna AS CHAR) LIKE :fcol";
    $params[':fcol'] = "%{$columna}%";
  }
  if ($nivel !== '') {
    $sql .= " AND CAST(pos.nivel AS CHAR) LIKE :fniv";
    $params[':fniv'] = "%{$nivel}%";
  }
  if ($fondo !== '') {
    $sql .= " AND CAST(pos.fondo AS CHAR) LIKE :ffon";
    $params[':ffon'] = "%{$fondo}%";
  }
  if ($q_ubicacion !== '') {
    $sql .= " AND CONCAT_WS('-', pos.rack, LPAD(pos.columna,2,'0'), CONCAT('N',pos.nivel), CONCAT('F',pos.fondo)) LIKE :qub";
    $params[':qub'] = "%{$q_ubicacion}%";
  }

  if ($q_pallet !== '') {
    if ($useNew) {
      $sql .= " AND pa.codigo LIKE :qpallet";
    } else {
      $sql .= " AND CAST(s.pallet_id AS CHAR) LIKE :qpallet";
    }
    $params[':qpallet'] = "%{$q_pallet}%";
  }
  if ($cliente_id !== '') {
    $sql .= " AND pr.cliente_id = :cliente_id";
    $params[':cliente_id'] = $cliente_id;
  }
  if ($operativa_id !== '') {
    $sql .= " AND pr.operativa_id = :operativa_id";
    $params[':operativa_id'] = $operativa_id;
  }
  if ($q_producto !== '') {
    $sql .= " AND (pr.sku LIKE :qp OR pr.denominacion LIKE :qp)";
    $params[':qp'] = "%{$q_producto}%";
  }

  if ($useNew) {
    $sql .= "
      GROUP BY pos.id, pos.rack, pos.columna, pos.nivel, pos.fondo, occ.id, pa.codigo
    ";
  } else {
    $sql .= "
      GROUP BY pos.id, pos.rack, pos.columna, pos.nivel, pos.fondo
    ";
  }

  if ($estado_pos === 'ocupada') {
    $sql .= " HAVING estado_pos = 'ocupada'";
  } elseif ($estado_pos === 'libre') {
    $sql .= " HAVING estado_pos = 'libre'";
  }

  // Orden por defecto
  $orderSql = " ORDER BY pos.rack ASC, pos.columna ASC, pos.nivel ASC, pos.fondo ASC";

  // Orden dinámico desde DataTable (dt_order = JSON [[idx, 'asc'|'desc'], ...])
  // Mapa según las columnas del listado:
  // 0 ubicacion, 1 rack, 2 columna, 3 nivel, 4 fondo, 5 estado_pos, 6 pallet_codigo,
  // 7 cliente, 8 operativa, 9 sku, 10 denominacion, 11 cajas, 12 unidades, 13 stock, 14 reservados, 15 disponibles
  $colMap = [
    0=>'ubicacion', 1=>'rack', 2=>'columna', 3=>'nivel', 4=>'fondo', 5=>'estado_pos',
    6=>'pallet_codigo', 7=>'cliente', 8=>'operativa', 9=>'sku', 10=>'denominacion',
    11=>'cajas', 12=>'unidades', 13=>'stock', 14=>'reservados', 15=>'disponibles'
  ];
  if (!empty($_GET['dt_order'])) {
    $raw = json_decode((string)$_GET['dt_order'], true);
    if (is_array($raw) && !empty($raw)) {
      $parts = [];
      foreach ($raw as $o) {
        if (!is_array($o) || count($o) < 2) continue;
        [$idx, $dir] = $o;
        $idx = (int)$idx;
        $dir = strtolower((string)$dir) === 'desc' ? 'DESC' : 'ASC';
        if (isset($colMap[$idx])) $parts[] = $colMap[$idx].' '.$dir;
        if (count($parts) >= 4) break;
      }
      if ($parts) $orderSql = ' ORDER BY '.implode(', ', $parts);
    }
  }

  $sql .= $orderSql;

  $st = $pdo->prepare($sql);
  foreach ($params as $k=>$v) $st->bindValue($k,$v);
  $st->execute();
  $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

  // Normalizaciones para libres y MIX
  foreach ($rows as &$r) {
    if (!empty($r['producto_mix'])) {
      $r['sku'] = null;
      $r['denominacion'] = 'MIX';
    }
    if (($r['estado_pos'] ?? '') === 'libre') {
      $r['pallet_codigo'] = '';
      $r['cliente'] = '';
      $r['operativa'] = '';
      $r['sku'] = '';
      $r['denominacion'] = '';
      $r['cajas'] = 0;
      $r['unidades'] = 0;
      $r['stock'] = 0;
      $r['reservados'] = 0;
      $r['disponibles'] = 0;
    }
  }
  unset($r);

  // Totales globales (numéricos)
  $tot = ['cajas'=>0,'unidades'=>0,'stock'=>0,'reservados'=>0,'disponibles'=>0];
  foreach ($rows as $r) {
    $tot['cajas']       += (int)($r['cajas'] ?? 0);
    $tot['unidades']    += (int)($r['unidades'] ?? 0);
    $tot['stock']       += (int)($r['stock'] ?? 0);
    $tot['reservados']  += (int)($r['reservados'] ?? 0);
    $tot['disponibles'] += (int)($r['disponibles'] ?? 0);
  }

  // ---------- PDF ----------
  $mpdf = new \Mpdf\Mpdf([
    'format' => 'A4',
    'margin_left'   => 10,
    'margin_right'  => 10,
    'margin_top'    => 14,
    'margin_bottom' => 12,
  ]);

  $title = "Inventario por ubicación";
  $styles = '
    <style>
      body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 11px; }
      h1 { font-size: 18px; margin: 0 0 6px 0; }
      .muted { color:#666; }
      .kv { display:flex; flex-wrap:wrap; gap:12px; margin:6px 0 8px 0; }
      .kv div { min-width: 180px; }
      .grid { width:100%; border-collapse: collapse; margin-top:6px; }
      .grid th, .grid td { border: 0.6px solid #999; padding: 4px 6px; }
      .grid th { background: #f1f1f1; }
      .right { text-align:right; }
      .badge { display:inline-block; padding:1px 6px; border:1px solid #999; border-radius:4px; }
      .b-occ { background:#dff0d8; border-color:#adcea1; }
      .b-free { background:#eee; border-color:#bbb; }
      .small { font-size: 10px; }
    </style>
  ';

  ob_start();
  ?>
  <?= $styles ?>
  <h1><?= htmlspecialchars($title) ?></h1>
  <div class="muted small">Generado: <?= date('Y-m-d H:i') ?> · Sistema: <?= htmlspecialchars((string)env('APP_NAME','SOL')) ?></div>

  <div class="kv">
    <div><b>Cliente:</b> <?= htmlspecialchars($clienteLabel) ?></div>
    <div><b>Operativa:</b> <?= htmlspecialchars($operativaLabel) ?></div>
    <div><b>Producto:</b> <?= htmlspecialchars($q_producto !== '' ? $q_producto : '—') ?></div>
    <div><b>Pallet:</b> <?= htmlspecialchars($q_pallet !== '' ? $q_pallet : '—') ?></div>
    <div><b>Ubicación contiene:</b> <?= htmlspecialchars($q_ubicacion !== '' ? $q_ubicacion : '—') ?></div>
    <div><b>Rack/Col/Nivel/Fondo:</b> <?= htmlspecialchars(($rack ?: '•').'/'.($columna ?: '•').'/'.($nivel ?: '•').'/'.($fondo ?: '•')) ?></div>
    <div><b>Estado posición:</b> <?= htmlspecialchars($estado_pos !== '' ? $estado_pos : '(Todas)') ?></div>
    <div><b>Filas:</b> <?= count($rows) ?></div>
  </div>

  <table class="grid">
    <thead>
      <tr>
        <th>Ubicación</th>
        <th>Rack</th>
        <th>Col</th>
        <th>Nivel</th>
        <th>Fondo</th>
        <th>Estado</th>
        <th>Pallet</th>
        <th>Cliente</th>
        <th>Operativa</th>
        <th>SKU</th>
        <th>Producto</th>
        <th class="right">Cajas</th>
        <th class="right">Unidades</th>
        <th class="right">Stock</th>
        <th class="right">Reservados</th>
        <th class="right">Disponibles</th>
      </tr>
    </thead>
    <tbody>
      <?php foreach ($rows as $r): ?>
        <?php
          $estado = strtolower((string)($r['estado_pos'] ?? ''));
          $cls = $estado === 'ocupada' ? 'badge b-occ' : 'badge b-free';
        ?>
        <tr>
          <td><?= htmlspecialchars((string)$r['ubicacion']) ?></td>
          <td><?= htmlspecialchars((string)$r['rack']) ?></td>
          <td class="right"><?= htmlspecialchars((string)$r['columna']) ?></td>
          <td class="right"><?= htmlspecialchars((string)$r['nivel']) ?></td>
          <td class="right"><?= htmlspecialchars((string)$r['fondo']) ?></td>
          <td><span class="<?= $cls ?>"><?= htmlspecialchars($estado ?: '-') ?></span></td>
          <td><?= htmlspecialchars((string)($r['pallet_codigo'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($r['cliente'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($r['operativa'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($r['sku'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($r['denominacion'] ?? '')) ?></td>
          <td class="right"><?= nf($r['cajas']) ?></td>
          <td class="right"><?= nf($r['unidades']) ?></td>
          <td class="right"><?= nf($r['stock']) ?></td>
          <td class="right"><?= nf($r['reservados']) ?></td>
          <td class="right"><?= nf($r['disponibles']) ?></td>
        </tr>
      <?php endforeach; ?>
    </tbody>
    <tfoot>
      <tr>
        <th colspan="11" class="right">Totales</th>
        <th class="right"><?= nf($tot['cajas']) ?></th>
        <th class="right"><?= nf($tot['unidades']) ?></th>
        <th class="right"><?= nf($tot['stock']) ?></th>
        <th class="right"><?= nf($tot['reservados']) ?></th>
        <th class="right"><?= nf($tot['disponibles']) ?></th>
      </tr>
    </tfoot>
  </table>
  <?php
  $html = ob_get_clean();

  $mpdf->SetTitle($title);
  $mpdf->SetAuthor((string)env('APP_NAME','SOL'));
  $mpdf->WriteHTML($html);
  $mpdf->Output('inventario_por_ubicacion.pdf', \Mpdf\Output\Destination::INLINE);

} catch (Throwable $e) {
  http_response_code(500);
  header('Content-Type: text/plain; charset=utf-8');
  echo (env('APP_ENV')==='local') ? ("Error generando PDF: ".$e->getMessage()) : "No se pudo generar el PDF";
}
