<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Verificar método POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['ok'=>false, 'error'=>'Método no permitido']);
        exit;
    }

    // Obtener datos del body
            // Pre-fetch estado IDs para actualizar estado al liberar de CUARENTENA
            $estadoPosEnteroId = 0;
            $estadoPosPickId   = 0;
            try {
                $estadoPosEnteroId = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='POS_ENTERO' LIMIT 1")->fetchColumn() ?: 0);
                $estadoPosPickId   = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='POS_PICKEADO' LIMIT 1")->fetchColumn() ?: 0);
            } catch (Throwable $e) { /* opcional: log */ }
    $input = json_decode(file_get_contents('php://input'), true);
    if (!$input) {
        http_response_code(400);
        echo json_encode(['ok'=>false, 'error'=>'JSON inválido']);
        exit;
    }

    $palletIds = $input['pallet_ids'] ?? [];
    $posicionId = (int)($input['posicion_id'] ?? 0);
    $depositoId = (int)($input['deposito_id'] ?? 0);
    $motivo = $input['motivo'] ?? 'UBICACION';

    // Validaciones
    if (empty($palletIds) || !is_array($palletIds)) {
        http_response_code(400);
        echo json_encode(['ok'=>false, 'error'=>'Faltan pallet_ids']);
        exit;
    }

    if ($posicionId <= 0) {
        http_response_code(400);
        echo json_encode(['ok'=>false, 'error'=>'Falta posicion_id']);
        exit;
    }

    if ($depositoId <= 0) {
        http_response_code(400);
        echo json_encode(['ok'=>false, 'error'=>'Falta deposito_id']);
        exit;
    }

    $pdo->beginTransaction();

    $movesCreated = [];
    $errors = [];

    foreach ($palletIds as $palletId) {
        $palletId = (int)$palletId;
        if ($palletId <= 0) continue;

        try {
            // Verificar que el pallet existe y está en cuarentena
            $stmtPallet = $pdo->prepare("
                SELECT p.id, p.codigo, p.posicion_id, pe.code as estado
                FROM wh_pallet p
                JOIN wh_pallet_estado pe ON pe.id = p.estado_id
                WHERE p.id = ? AND p.deposito_id = ? AND p.deleted_at IS NULL
            ");
            $stmtPallet->execute([$palletId, $depositoId]);
            $pallet = $stmtPallet->fetch(PDO::FETCH_ASSOC);

            if (!$pallet) {
                $errors[] = "Pallet ID {$palletId} no encontrado o no pertenece al depósito";
                continue;
            }

            if ($pallet['estado'] !== 'CUARENTENA') {
                $errors[] = "Pallet {$pallet['codigo']} no está en cuarentena (estado: {$pallet['estado']})";
                continue;
            }

            // Verificar capacidad de la posición destino
            $stmtPos = $pdo->prepare("
                SELECT p.id, p.capacidad_pallets, p.ocupado,
                       p.ambiente_id, a.code AS ambiente_code,
                       COUNT(s.pallet_id) AS pallets_actuales
                  FROM wh_posicion p
             LEFT JOIN wh_stock s ON s.posicion_id = p.id AND s.pallet_id IS NOT NULL
             LEFT JOIN wh_ambiente a ON a.id = p.ambiente_id
                 WHERE p.id = ? AND p.deposito_id = ? AND p.activo = 1
              GROUP BY p.id, p.capacidad_pallets, p.ocupado, p.ambiente_id, a.code
            ");
            $stmtPos->execute([$posicionId, $depositoId]);
            $posicion = $stmtPos->fetch(PDO::FETCH_ASSOC);

            if (!$posicion) {
                $errors[] = "Posición ID {$posicionId} no encontrada";
                continue;
            }

            if ($posicion['pallets_actuales'] >= $posicion['capacidad_pallets']) {
                $errors[] = "Posición sin capacidad disponible ({$posicion['pallets_actuales']}/{$posicion['capacidad_pallets']})";
                continue;
            }

            // Crear movimiento MOVE desde cuarentena a posición
            $stmtMove = $pdo->prepare("
                INSERT INTO wh_move (
                    deposito_id, tipo, motivo, pallet_id, 
                    from_pos_id, to_pos_id, delta_uv, delta_uc, 
                    referencia, created_at
                ) VALUES (?, 'MOVE', ?, ?, ?, ?, 0, 0, ?, NOW())
            ");

            $referencia = "ASIGNACION_INGRESO";
            $stmtMove->execute([
                $depositoId, $motivo, $palletId,
                $pallet['posicion_id'], $posicionId, $referencia
            ]);

            $moveId = $pdo->lastInsertId();

            // Actualizar posición del pallet
            $stmtUpdatePallet = $pdo->prepare("
                UPDATE wh_pallet 
                   SET posicion_id = ?, updated_at = NOW()
                 WHERE id = ?
            ");
            $stmtUpdatePallet->execute([$posicionId, $palletId]);

            // Obtener items del pallet para actualizar stock
            $stmtItems = $pdo->prepare("
                SELECT producto_id, lote_id, uv_cajas, uc_unidades
                FROM wh_pallet_item
                WHERE pallet_id = ? AND deleted_at IS NULL
            ");
            $stmtItems->execute([$palletId]);
            $items = $stmtItems->fetchAll(PDO::FETCH_ASSOC);

            // Actualizar stock en nueva posición
            foreach ($items as $item) {
                // Quitar stock de la posición anterior (si existe)
                if ($pallet['posicion_id']) {
                    $stmtDelStock = $pdo->prepare("
                        UPDATE wh_stock 
                        SET qty_uv = qty_uv - ?, qty_uc = qty_uc - ?, updated_at = NOW()
                        WHERE deposito_id = ? AND posicion_id = ? 
                          AND producto_id = ? AND lote_id = ? AND pallet_id = ?
                    ");
                    $stmtDelStock->execute([
                        $item['uv_cajas'], $item['uc_unidades'],
                        $depositoId, $pallet['posicion_id'],
                        $item['producto_id'], $item['lote_id'], $palletId
                    ]);
                }

                // Insertar/actualizar stock en nueva posición
                $stmtUpsertStock = $pdo->prepare("
                    INSERT INTO wh_stock (
                        deposito_id, posicion_id, producto_id, lote_id, pallet_id,
                        qty_uv, qty_uc, pickeado, updated_at
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, 0, NOW())
                    ON DUPLICATE KEY UPDATE
                        qty_uv = qty_uv + VALUES(qty_uv),
                        qty_uc = qty_uc + VALUES(qty_uc),
                        updated_at = NOW()
                ");
                $stmtUpsertStock->execute([
                    $depositoId, $posicionId, $item['producto_id'], 
                    $item['lote_id'], $palletId, $item['uv_cajas'], $item['uc_unidades']
                ]);
            }

            // Actualizar flags de ocupación en posiciones
            $stmtUpdatePosOld = $pdo->prepare("
                UPDATE wh_posicion 
                SET ocupado = (
                    SELECT IF(COUNT(*) > 0, 1, 0) 
                    FROM wh_stock 
                    WHERE posicion_id = ? AND qty_uv > 0
                ), updated_at = NOW()
                WHERE id = ?
            ");
            if ($pallet['posicion_id']) {
                $stmtUpdatePosOld->execute([$pallet['posicion_id'], $pallet['posicion_id']]);
            }
            $stmtUpdatePosOld->execute([$posicionId, $posicionId]);

            // Actualizar estado del pallet al salir de CUARENTENA
            try {
                $destEsPicking = isset($posicion['ambiente_code']) && strtoupper((string)$posicion['ambiente_code']) === 'PICKING';
                $nuevoEstadoId = $destEsPicking ? ($estadoPosPickId ?: 0) : ($estadoPosEnteroId ?: 0);
                if ($nuevoEstadoId > 0) {
                    $stUpdEst = $pdo->prepare("UPDATE wh_pallet SET estado_id = ?, updated_at = NOW() WHERE id = ?");
                    $stUpdEst->execute([$nuevoEstadoId, $palletId]);
                }
            } catch (Throwable $e) { /* opcional: log */ }

            $movesCreated[] = [
                'move_id' => $moveId,
                'pallet_id' => $palletId,
                'pallet_codigo' => $pallet['codigo'],
                'estado_dest' => ($destEsPicking ? 'POS_PICKEADO' : 'POS_ENTERO')
            ];

        } catch (Exception $e) {
            $errors[] = "Error procesando pallet {$palletId}: " . $e->getMessage();
        }
    }

    $pdo->commit();

    $response = [
        'ok' => true,
        'moves_created' => count($movesCreated),
        'moves' => $movesCreated
    ];

    if (!empty($errors)) {
        $response['warnings'] = $errors;
    }

    echo json_encode($response);

} catch (Throwable $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    http_response_code(500);
    echo json_encode([
        'ok'=>false,
        'error'=>'Error asignando pallets',
        'msg'=>(env('APP_ENV')==='local') ? $e->getMessage() : ''
    ]);
}