<?php
declare(strict_types=1);

// API: debug_stock_overview.php
// Quick stock projection overview by depósito and ambiente (uses wh_stock)
// GET params:
//   deposito_code (optional, default DEP1)
//   deposito_id   (optional)
//   limit         (optional, default 10) for sample rows in PICKING

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/db.php';

function outJson(array $p, int $c=200){ http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }

try {
  $depCode = isset($_GET['deposito_code']) ? trim((string)$_GET['deposito_code']) : '';
  $depIdIn = isset($_GET['deposito_id']) ? (int)$_GET['deposito_id'] : 0;
  $limit = isset($_GET['limit']) ? max(1,(int)$_GET['limit']) : 10;

  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $pdo->exec('SET NAMES utf8mb4');

  // Resolve depósito
  if ($depIdIn > 0) {
    $st = $pdo->prepare('SELECT id, code, nombre FROM wh_deposito WHERE id=? LIMIT 1');
    $st->execute([$depIdIn]);
    $dep = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  } elseif ($depCode !== '') {
    $st = $pdo->prepare('SELECT id, code, nombre FROM wh_deposito WHERE code=? LIMIT 1');
    $st->execute([$depCode]);
    $dep = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  } else {
    $st = $pdo->query("SELECT id, code, nombre FROM wh_deposito WHERE code='DEP1' LIMIT 1");
    $dep = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  }
  if (!$dep) outJson(['ok'=>false,'error'=>'Depósito no encontrado']);
  $depId = (int)$dep['id'];

  // Aggregates by ambiente
  $sqlAgg = "SELECT a.code AS ambiente, COUNT(DISTINCT s.posicion_id) AS posiciones, COUNT(*) AS filas, COALESCE(SUM(s.qty_uv),0) AS total_uv, COALESCE(SUM(s.qty_uc),0) AS total_uc
             FROM wh_stock s
             JOIN wh_posicion p ON p.id=s.posicion_id
             JOIN wh_ambiente a ON a.id=p.ambiente_id
            WHERE s.deposito_id=?
            GROUP BY a.code
            ORDER BY a.code";
  $st = $pdo->prepare($sqlAgg);
  $st->execute([$depId]);
  $agg = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

  // Sample rows in PICKING (incluye UV equivalentes desde UC si existe factor en para_producto_pack)
  $sqlPick = "SELECT p.code AS pos_code, pr.sku AS producto_code, pr.denominacion AS producto_nombre, l.codigo AS lote_codigo, l.fecha_vencimiento,
                      s.qty_uv, s.qty_uc,
                      COALESCE(pu.unidades_por_uv, 0) AS uc_per_uv,
                      CASE WHEN COALESCE(pu.unidades_por_uv, 0) > 0 THEN FLOOR(s.qty_uc / pu.unidades_por_uv) ELSE 0 END AS uv_equiv_from_uc
               FROM wh_stock s
               JOIN para_productos pr ON pr.id=s.producto_id
               LEFT JOIN (
                 SELECT producto_id, MAX(unidades_por_uv) AS unidades_por_uv
                   FROM para_producto_pack
                  GROUP BY producto_id
               ) pu ON pu.producto_id = s.producto_id
               LEFT JOIN wh_lote l ON l.id=s.lote_id
               JOIN wh_posicion p ON p.id=s.posicion_id
               JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PICKING'
              WHERE s.deposito_id=? AND (s.qty_uv>0 OR s.qty_uc>0)
              ORDER BY COALESCE(l.fecha_vencimiento,'9999-12-31') ASC, p.code ASC
              LIMIT " . (int)$limit;
  $st = $pdo->prepare($sqlPick);
  $st->execute([$depId]);
  $pick = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

  outJson(['ok'=>true,'deposito'=>$dep,'by_ambiente'=>$agg,'sample_picking'=>$pick]);
} catch (Throwable $e) {
  outJson(['ok'=>false,'error'=>'Error en overview','message'=>$e->getMessage()], 500);
}
