<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['ok'=>false,'error'=>'Método no permitido']);
        exit;
    }

    $input = json_decode(file_get_contents('php://input'), true);
    if (!$input) {
        http_response_code(400);
        echo json_encode(['ok'=>false,'error'=>'JSON inválido']);
        exit;
    }

    $palletId = isset($input['pallet_id']) ? (int)$input['pallet_id'] : 0;
    $posicionId = isset($input['posicion_id']) ? (int)$input['posicion_id'] : 0;

    if ($palletId <= 0 && $posicionId <= 0) {
        http_response_code(400);
        echo json_encode(['ok'=>false,'error'=>'Falta pallet_id o posicion_id']);
        exit;
    }

    $pdo->beginTransaction();

    $affected = [];

    if ($palletId > 0) {
        // Move single pallet to CUARENTENA and remove posicion
        $stmt = $pdo->prepare("SELECT id, codigo, deposito_id, posicion_id FROM wh_pallet WHERE id = ? AND deleted_at IS NULL");
        $stmt->execute([$palletId]);
        $p = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$p) {
            $pdo->rollBack();
            echo json_encode(['ok'=>false,'error'=>'Pallet no encontrado']);
            exit;
        }

        $depositoId = (int)$p['deposito_id'];
        $oldPos = (int)$p['posicion_id'];

        // create move record indicating desasignacion
        $stmtMove = $pdo->prepare("INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, from_pos_id, to_pos_id, referencia, created_at) VALUES (?, 'MOVE', 'DESASIGNACION', ?, ?, NULL, 'DESASIGN', NOW())");
        $stmtMove->execute([$depositoId, $palletId, $oldPos]);

        // set pallet posicion_id = NULL and estado to CUARENTENA
        $stmtUpd = $pdo->prepare("UPDATE wh_pallet SET posicion_id = NULL, updated_at = NOW() WHERE id = ?");
        $stmtUpd->execute([$palletId]);

        // update stock: remove from old position (already handled by wh_move/stock logic elsewhere) - attempt to subtract quantities from stock rows for that pallet
        $stmtItems = $pdo->prepare("SELECT producto_id, lote_id, uv_cajas, uc_unidades FROM wh_pallet_item WHERE pallet_id = ? AND deleted_at IS NULL");
        $stmtItems->execute([$palletId]);
        $items = $stmtItems->fetchAll(PDO::FETCH_ASSOC);
        foreach ($items as $it) {
            if ($oldPos) {
                $stmtDel = $pdo->prepare("UPDATE wh_stock SET qty_uv = GREATEST(0, qty_uv - ?), qty_uc = GREATEST(0, qty_uc - ?), updated_at = NOW() WHERE deposito_id = ? AND posicion_id = ? AND producto_id = ? AND lote_id = ? AND pallet_id = ?");
                $stmtDel->execute([$it['uv_cajas'], $it['uc_unidades'], $depositoId, $oldPos, $it['producto_id'], $it['lote_id'], $palletId]);
            }
            // insert stock row into CUARENTENA position is optional; in many designs CUARENTENA is represented as a special area - here we just clear posicion
        }

        // update occupied flags
        if ($oldPos) {
            $stmtUpdPos = $pdo->prepare("UPDATE wh_posicion SET ocupado = (SELECT IF(COUNT(*) > 0, 1, 0) FROM wh_stock WHERE posicion_id = ? AND qty_uv > 0), updated_at = NOW() WHERE id = ?");
            $stmtUpdPos->execute([$oldPos, $oldPos]);
        }

        $affected[] = $palletId;
    }

    if ($posicionId > 0) {
        // Find all pallets assigned to this position and desasign them
        $stmtP = $pdo->prepare("SELECT id FROM wh_pallet WHERE posicion_id = ? AND deleted_at IS NULL");
        $stmtP->execute([$posicionId]);
        $rows = $stmtP->fetchAll(PDO::FETCH_ASSOC);
        foreach ($rows as $r) {
            $pid = (int)$r['id'];
            // reuse the single-pallet logic
            // create move
            $stmtMove = $pdo->prepare("INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, from_pos_id, to_pos_id, referencia, created_at) VALUES ((SELECT deposito_id FROM wh_pallet WHERE id = ?), 'MOVE', 'DESASIGNACION', ?, ?, NULL, 'DESASIGN', NOW())");
            $stmtMove->execute([$pid, $pid, $posicionId]);

            // unset posicion
            $stmtUpd = $pdo->prepare("UPDATE wh_pallet SET posicion_id = NULL, updated_at = NOW() WHERE id = ?");
            $stmtUpd->execute([$pid]);

            // update stock lines
            $stmtItems = $pdo->prepare("SELECT producto_id, lote_id, uv_cajas, uc_unidades FROM wh_pallet_item WHERE pallet_id = ? AND deleted_at IS NULL");
            $stmtItems->execute([$pid]);
            $items = $stmtItems->fetchAll(PDO::FETCH_ASSOC);
            foreach ($items as $it) {
                $stmtDel = $pdo->prepare("UPDATE wh_stock SET qty_uv = GREATEST(0, qty_uv - ?), qty_uc = GREATEST(0, qty_uc - ?), updated_at = NOW() WHERE posicion_id = ? AND producto_id = ? AND lote_id = ? AND pallet_id = ?");
                $stmtDel->execute([$it['uv_cajas'], $it['uc_unidades'], $posicionId, $it['producto_id'], $it['lote_id'], $pid]);
            }

            $affected[] = $pid;
        }

        // update position occupied flag
        $stmtUpdPos = $pdo->prepare("UPDATE wh_posicion SET ocupado = (SELECT IF(COUNT(*) > 0, 1, 0) FROM wh_stock WHERE posicion_id = ? AND qty_uv > 0), updated_at = NOW() WHERE id = ?");
        $stmtUpdPos->execute([$posicionId, $posicionId]);
    }

    $pdo->commit();

    echo json_encode(['ok'=>true,'affected'=>$affected]);

} catch (Throwable $e) {
    if (isset($pdo) && $pdo->inTransaction()) $pdo->rollBack();
    http_response_code(500);
    echo json_encode(['ok'=>false,'error'=>'Error procesando desasignacion','msg'=> (env('APP_ENV')==='local') ? $e->getMessage() : '']);
}
