<?php
declare(strict_types=1);

header('Content-Type: application/pdf');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

// ---- Intenta cargar mPDF ----
$aut = $ROOT . '/vendor/autoload.php';
if (!is_file($aut)) {
  http_response_code(500);
  echo "Falta vendor/autoload.php. Instala dependencias con Composer.";
  exit;
}
require_once $aut;

if (!class_exists(\Mpdf\Mpdf::class)) {
  http_response_code(500);
  echo "No está instalado mpdf/mpdf. Ejecuta: composer require mpdf/mpdf";
  exit;
}

// ---- Helpers mínimos ----
function nf($n, int $dec = 0): string {
  if ($n === null || $n === '') return '0';
  return number_format((float)$n, $dec, ',', '.'); // 1.234.567,89
}
function hasCol(PDO $pdo, string $t, string $c): bool {
  static $cache = [];
  $k = $t.'|'.$c;
  if (array_key_exists($k,$cache)) return $cache[$k];
  $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME=? AND COLUMN_NAME=?");
  $st->execute([$t,$c]);
  return $cache[$k] = ((int)$st->fetchColumn() > 0);
}
function hasTbl(PDO $pdo, string $t): bool {
  static $cache = [];
  if (array_key_exists($t,$cache)) return $cache[$t];
  $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME=?");
  $st->execute([$t]);
  return $cache[$t] = ((int)$st->fetchColumn() > 0);
}

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  $ingresoId = (int)($_GET['ingreso_id'] ?? 0);
  if ($ingresoId <= 0) {
    http_response_code(400);
    echo "ingreso_id es requerido";
    exit;
  }

  // ---------- CABECERA ----------
  $tblIng = 'pl_ingreso';
  $ci = [
    'fecha_ingreso'        => hasCol($pdo,$tblIng,'fecha_ingreso'),
    'descarga_inicio_at'   => hasCol($pdo,$tblIng,'descarga_inicio_at'),
    'operarios_cant'       => hasCol($pdo,$tblIng,'operarios_cant'),
    'observacion'          => hasCol($pdo,$tblIng,'observacion'),
    'movil_id'             => hasCol($pdo,$tblIng,'movil_id'),
    'chofer_id'            => hasCol($pdo,$tblIng,'chofer_id'),
    'deposito_id'          => hasCol($pdo,$tblIng,'deposito_id'),
    'packinglist_id'       => hasCol($pdo,$tblIng,'packinglist_id'),
    'doc_tipo'             => hasCol($pdo,$tblIng,'doc_tipo'),
    'doc_numero'           => hasCol($pdo,$tblIng,'doc_numero'),
    'doc_fecha'            => hasCol($pdo,$tblIng,'doc_fecha'),
  ];

  $hasMoviles  = hasTbl($pdo,'para_moviles');
  $movilLabel  = $hasMoviles && hasCol($pdo,'para_moviles','chapa') ? 'chapa' : ($hasMoviles && hasCol($pdo,'para_moviles','codigo') ? 'codigo' : null);
  $hasChoferes = hasTbl($pdo,'para_choferes');
  $choferName  = $hasChoferes && hasCol($pdo,'para_choferes','nombre') ? 'nombre' : null;
  $hasDepositos = hasTbl($pdo,'wh_deposito');
  $depositoName = $hasDepositos && hasCol($pdo,'wh_deposito','nombre') ? 'nombre' : null;
  $hasPL = hasTbl($pdo,'pl_packinglist');
  $plLabel = $hasPL && hasCol($pdo,'pl_packinglist','codigo') ? 'codigo' : null;

  $select = ["i.id"];
  if ($ci['fecha_ingreso'])        $select[] = "i.fecha_ingreso";
  if ($ci['descarga_inicio_at'])   $select[] = "i.descarga_inicio_at";
  if ($ci['operarios_cant'])       $select[] = "i.operarios_cant";
  if ($ci['observacion'])          $select[] = "i.observacion";
  if ($ci['movil_id'])             $select[] = "i.movil_id";
  if ($ci['chofer_id'])            $select[] = "i.chofer_id";
  if ($ci['deposito_id'])          $select[] = "i.deposito_id";
  if ($ci['packinglist_id'])       $select[] = "i.packinglist_id";
  if ($ci['doc_tipo'])             $select[] = "i.doc_tipo";
  if ($ci['doc_numero'])           $select[] = "i.doc_numero";
  if ($ci['doc_fecha'])            $select[] = "i.doc_fecha";

  $joins = [];
  if ($ci['movil_id'] && $hasMoviles && $movilLabel) {
    $joins[] = "LEFT JOIN para_moviles mv ON mv.id = i.movil_id";
    $select[] = "mv.$movilLabel AS movil";
  }
  if ($ci['chofer_id'] && $hasChoferes && $choferName) {
    $joins[] = "LEFT JOIN para_choferes ch ON ch.id = i.chofer_id";
    $select[] = "ch.$choferName AS chofer";
  }
  if ($ci['deposito_id'] && $hasDepositos && $depositoName) {
    $joins[] = "LEFT JOIN wh_deposito dep ON dep.id = i.deposito_id";
    $select[] = "dep.$depositoName AS deposito";
  }
  if ($ci['packinglist_id'] && $hasPL && $plLabel) {
    $joins[] = "LEFT JOIN pl_packinglist pl ON pl.id = i.packinglist_id";
    $select[] = "pl.$plLabel AS packing_list";
  }

  $sqlH = "SELECT ".implode(", ", $select)." FROM {$tblIng} i ".implode(" ", array_unique($joins))." WHERE i.id=? LIMIT 1";
  $sth = $pdo->prepare($sqlH);
  $sth->execute([$ingresoId]);
  $H = $sth->fetch(PDO::FETCH_ASSOC);

  if (!$H) {
    http_response_code(404);
    echo "Ingreso no encontrado";
    exit;
  }

  // Normalización cabecera
  $cab = [
    'packing_list' => $H['packing_list'] ?? (isset($H['packinglist_id']) ? '#'.$H['packinglist_id'] : '-'),
    'deposito'     => $H['deposito'] ?? (isset($H['deposito_id']) ? '#'.$H['deposito_id'] : '-'),
    'movil'        => $H['movil']   ?? (isset($H['movil_id']) ? '#'.$H['movil_id'] : '-'),
    'chofer'       => $H['chofer']  ?? (isset($H['chofer_id']) ? '#'.$H['chofer_id'] : '-'),
    'fecha'        => $H['fecha_ingreso'] ?? '-',
    'inicio'       => $H['descarga_inicio_at'] ?? '-',
    'operarios'    => isset($H['operarios_cant']) ? (int)$H['operarios_cant'] : null,
    'obs'          => $H['observacion'] ?? '',
    'doc_tipo'     => $H['doc_tipo'] ?? '',
    'doc_numero'   => $H['doc_numero'] ?? '',
    'doc_fecha'    => $H['doc_fecha'] ?? '',
  ];

  // ---------- PALLETS + TOTALES ----------
  // Detectar tablas según esquema (new-core vs legacy)
  $PAL  = hasTbl($pdo,'wh_pallets')       ? 'wh_pallets'       : (hasTbl($pdo,'wh_pallet') ? 'wh_pallet' : null);
  $PI   = hasTbl($pdo,'wh_pallet_items')  ? 'wh_pallet_items'  : (hasTbl($pdo,'wh_pallet_item') ? 'wh_pallet_item' : null);
  $POS  = hasTbl($pdo,'wh_positions')     ? 'wh_positions'     : (hasTbl($pdo,'wh_posicion') ? 'wh_posicion' : null);
  $hasOcc = hasTbl($pdo,'wh_position_occupancy') && hasCol($pdo,'wh_position_occupancy','pallet_id');

  if (!$PAL) { throw new RuntimeException('No se encontró tabla de pallets (wh_pallets/wh_pallet)'); }
  if (!$PI)  { throw new RuntimeException('No se encontró tabla de ítems de pallet (wh_pallet_items/wh_pallet_item)'); }
  if (!$POS) { throw new RuntimeException('No se encontró tabla de posiciones (wh_positions/wh_posicion)'); }

  // Etiqueta de posición compatible
  $posCodeExpr = hasCol($pdo,$POS,'codigo')
    ? 'pos.codigo'
    : (hasCol($pdo,$POS,'nombre')
      ? 'pos.nombre'
      : (hasCol($pdo,$POS,'code_full')
        ? 'pos.code_full'
        : (hasCol($pdo,$POS,'code') ? 'pos.code' : "CONCAT('POS ', pos.id)")));

  // Joins para posición según disponibilidad de occupancy
  if ($hasOcc) {
    $posJoin = "LEFT JOIN (\n        SELECT o1.*\n        FROM wh_position_occupancy o1\n        JOIN (\n          SELECT pallet_id, MAX(id) AS max_id\n          FROM wh_position_occupancy\n          GROUP BY pallet_id\n        ) last_o ON last_o.pallet_id = o1.pallet_id AND last_o.max_id = o1.id\n      ) o ON o.pallet_id = p.id\n      LEFT JOIN {$POS} pos ON pos.id = o.position_id";
  } else {
    $posJoin = hasCol($pdo, $PAL, 'posicion_id')
      ? "LEFT JOIN {$POS} pos ON pos.id = p.posicion_id"
      : "LEFT JOIN {$POS} pos ON 1=0";
  }

  // Pallet código compatible
  $palletCodeExpr = hasCol($pdo, $PAL, 'codigo') ? 'p.codigo' : (hasCol($pdo, $PAL, 'code') ? 'p.code' : "CAST(p.id AS CHAR)");

  // Ítems: compatibilidad new-core vs legacy
  $isNewPI = ($PI === 'wh_pallet_items');
  $hasUcPorCaja  = hasCol($pdo, $PI, 'uc_por_caja');
  $hasUcSueltas  = hasCol($pdo, $PI, 'uc_sueltas');
  $hasUcUnidades = hasCol($pdo, $PI, 'uc_unidades');
  $ucTotalExpr = ($hasUcPorCaja || $hasUcSueltas)
    ? 'COALESCE(COALESCE(i.uc_por_caja,0)*COALESCE(i.uv_cajas,0) + COALESCE(i.uc_sueltas,0),0)'
    : ($hasUcUnidades ? 'COALESCE(i.uc_unidades,0)' : '0');
  $itemsExtraCols = ($hasUcPorCaja || $hasUcSueltas)
    ? 'i.uc_por_caja, i.uc_sueltas'
    : 'NULL AS uc_por_caja, NULL AS uc_sueltas';

  // Lote/fechas por esquema
  $lotAlias = 'lt';
  $hasLoteTbl = hasTbl($pdo, 'wh_lote');
  $joinLote = '';
  if ($isNewPI) {
    $selectLote   = hasCol($pdo, 'wh_pallet_items', 'lote') ? 'i.lote AS lote' : 'NULL AS lote';
    $selectFProd  = hasCol($pdo, 'wh_pallet_items', 'fecha_produccion') ? 'i.fecha_produccion AS fecha_produccion' : 'NULL AS fecha_produccion';
    $selectFVenc  = hasCol($pdo, 'wh_pallet_items', 'fecha_vencimiento') ? 'i.fecha_vencimiento AS fecha_vencimiento' : 'NULL AS fecha_vencimiento';
  } elseif ($hasLoteTbl && hasCol($pdo, 'wh_pallet_item', 'lote_id')) {
    $joinLote = "LEFT JOIN wh_lote $lotAlias ON $lotAlias.id = i.lote_id";
    $lotCodeCol = hasCol($pdo, 'wh_lote', 'codigo') ? 'codigo' : (hasCol($pdo, 'wh_lote', 'code') ? 'code' : null);
    $selectLote = $lotCodeCol ? "$lotAlias.$lotCodeCol AS lote" : 'NULL AS lote';
    $loteProdCol = hasCol($pdo, 'wh_lote', 'fecha_produccion') ? 'fecha_produccion' : null;
    $loteVencCol = hasCol($pdo, 'wh_lote', 'fecha_vencimiento') ? 'fecha_vencimiento' : null;
    $selectFProd = $loteProdCol ? "$lotAlias.$loteProdCol AS fecha_produccion" : 'NULL AS fecha_produccion';
    $selectFVenc = $loteVencCol ? "$lotAlias.$loteVencCol AS fecha_vencimiento" : 'NULL AS fecha_vencimiento';
  } else {
    $selectLote  = 'NULL AS lote';
    $selectFProd = 'NULL AS fecha_produccion';
    $selectFVenc = 'NULL AS fecha_vencimiento';
  }

  // Producto: sku/nombre compatibles
  $prodSkuExpr  = hasCol($pdo,'para_productos','sku') ? 'pr.sku' : (hasCol($pdo,'para_productos','codigo') ? 'pr.codigo' : "''");
  $prodNameExpr = hasCol($pdo,'para_productos','denominacion') ? 'pr.denominacion' : (hasCol($pdo,'para_productos','nombre') ? 'pr.nombre' : "''");

  // Intentar con FK pl_ingreso_id (si existe) o ingreso_id apuntando a pl_ingreso
  $palletByFK    = hasCol($pdo,$PAL,'pl_ingreso_id') || hasCol($pdo,$PAL,'ingreso_id');
  $hasWhMoves    = hasTbl($pdo, 'wh_moves');
  $hasWhMove     = hasTbl($pdo, 'wh_move');
  $hasLink       = hasTbl($pdo, 'pl_rcv_link');
  $packinglistId = isset($H['packinglist_id']) ? (int)$H['packinglist_id'] : null;
  $moveTypeCol   = $hasWhMoves ? (hasCol($pdo,'wh_moves','move_type') ? 'move_type' : (hasCol($pdo,'wh_moves','tipo') ? 'tipo' : null)) : null;
  $moveCodeCol   = $hasWhMoves ? (hasCol($pdo,'wh_moves','move_code') ? 'move_code' : (hasCol($pdo,'wh_moves','codigo') ? 'codigo' : null)) : null;
  $moveStatusCol = $hasWhMoves ? (hasCol($pdo,'wh_moves','status') ? 'status' : (hasCol($pdo,'wh_moves','estado') ? 'estado' : null)) : null;
  $moveStartCol  = $hasWhMoves ? (hasCol($pdo,'wh_moves','iniciado_at') ? 'iniciado_at' : (hasCol($pdo,'wh_moves','created_at') ? 'created_at' : null)) : null;
  $moveEndCol    = $hasWhMoves ? (hasCol($pdo,'wh_moves','finalizado_at') ? 'finalizado_at' : (hasCol($pdo,'wh_moves','completed_at') ? 'completed_at' : (hasCol($pdo,'wh_moves','closed_at') ? 'closed_at' : null))) : null;

  // PALLETS
  $pallets = [];
  $palletSource = null;
  if ($palletByFK) {
    $fkField = hasCol($pdo,$PAL,'pl_ingreso_id') ? 'pl_ingreso_id' : 'ingreso_id';
    $sqlP = "
      SELECT
        p.id AS pallet_id,
        {$palletCodeExpr} AS pallet_codigo,
        {$posCodeExpr} AS posicion,
        COUNT(i.id) AS items_count,
        COALESCE(SUM(i.uv_cajas),0) AS uv_total,
        COALESCE(SUM({$ucTotalExpr}),0) AS uc_total
      FROM {$PAL} p
      {$posJoin}
      LEFT JOIN {$PI} i ON i.pallet_id = p.id
      WHERE p.{$fkField} = ?
      GROUP BY p.id, {$palletCodeExpr}, {$posCodeExpr}
      ORDER BY p.id ASC
    ";
    $stp = $pdo->prepare($sqlP);
    $stp->execute([$ingresoId]);
    $pallets = $stp->fetchAll(PDO::FETCH_ASSOC) ?: [];
    if ($pallets) {
      $palletSource = 'fk';
    }
  }

  if (!$pallets && $hasLink && $packinglistId) {
    $sqlP = "
      SELECT
        p.id AS pallet_id,
        {$palletCodeExpr} AS pallet_codigo,
        {$posCodeExpr} AS posicion,
        COUNT(i.id) AS items_count,
        COALESCE(SUM(i.uv_cajas),0) AS uv_total,
        COALESCE(SUM({$ucTotalExpr}),0) AS uc_total
      FROM pl_rcv_link l
      JOIN {$PAL} p ON p.id = l.pallet_id
      {$posJoin}
      LEFT JOIN {$PI} i ON i.pallet_id = p.id
      WHERE l.packinglist_id = ?
      GROUP BY p.id, {$palletCodeExpr}, {$posCodeExpr}
      ORDER BY p.id ASC
    ";
    $stp = $pdo->prepare($sqlP);
    $stp->execute([$packinglistId]);
    $pallets = $stp->fetchAll(PDO::FETCH_ASSOC) ?: [];
    if ($pallets) {
      $palletSource = 'pl_link';
    }
  }

  if (!$pallets && $hasWhMoves && $moveCodeCol) {
    $sqlP = "
      SELECT
        p.id AS pallet_id,
        {$palletCodeExpr} AS pallet_codigo,
        {$posCodeExpr} AS posicion,
        COUNT(i.id) AS items_count,
        COALESCE(SUM(i.uv_cajas),0) AS uv_total,
        COALESCE(SUM({$ucTotalExpr}),0) AS uc_total
      FROM wh_moves m
      JOIN {$PAL} p ON p.id = m.pallet_id
      {$posJoin}
      LEFT JOIN {$PI} i ON i.pallet_id = p.id
      WHERE m.move_type='INBOUND' AND m.move_code LIKE ?
      GROUP BY p.id, {$palletCodeExpr}, {$posCodeExpr}
      ORDER BY p.id ASC
    ";
    $stp = $pdo->prepare($sqlP);
    $stp->execute(['IN-'.$ingresoId.'-%']);
    $pallets = $stp->fetchAll(PDO::FETCH_ASSOC) ?: [];
    if ($pallets) {
      $palletSource = 'moves';
    }
  }

  // Totales pallets
  $totUV = 0; $totUC = 0;
  foreach ($pallets as $r) {
    $totUV += (int)($r['uv_total'] ?? 0);
    $totUC += (int)($r['uc_total'] ?? 0);
  }

  // ÍTEMS
  $items = [];
  if ($palletSource === 'fk') {
    $fkField = hasCol($pdo,$PAL,'pl_ingreso_id') ? 'pl_ingreso_id' : 'ingreso_id';
    $sqlI = "
      SELECT
        {$palletCodeExpr} AS pallet_codigo,
        {$prodSkuExpr} AS sku,
        {$prodNameExpr} AS denominacion,
        {$selectLote},
        {$selectFProd},
        {$selectFVenc},
        i.uv_cajas,
        {$itemsExtraCols},
        ({$ucTotalExpr}) AS uc_total
      FROM {$PAL} p
      JOIN {$PI} i ON i.pallet_id = p.id
      {$joinLote}
      LEFT JOIN para_productos pr ON pr.id = i.producto_id
      WHERE p.{$fkField} = ?
      ORDER BY p.id, i.id
    ";
    $sti = $pdo->prepare($sqlI);
    $sti->execute([$ingresoId]);
    $items = $sti->fetchAll(PDO::FETCH_ASSOC) ?: [];
  } elseif ($palletSource === 'pl_link') {
    $sqlI = "
      SELECT
        {$palletCodeExpr} AS pallet_codigo,
        {$prodSkuExpr} AS sku,
        {$prodNameExpr} AS denominacion,
        {$selectLote},
        {$selectFProd},
        {$selectFVenc},
        i.uv_cajas,
        {$itemsExtraCols},
        ({$ucTotalExpr}) AS uc_total
      FROM pl_rcv_link l
      JOIN {$PAL} p ON p.id = l.pallet_id
      JOIN {$PI} i ON i.pallet_id = p.id
      {$joinLote}
      LEFT JOIN para_productos pr ON pr.id = i.producto_id
      WHERE l.packinglist_id = ?
      ORDER BY p.id, i.id
    ";
    $sti = $pdo->prepare($sqlI);
    $sti->execute([$packinglistId]);
    $items = $sti->fetchAll(PDO::FETCH_ASSOC) ?: [];
  } elseif ($palletSource === 'moves') {
    $sqlI = "
      SELECT
        {$palletCodeExpr} AS pallet_codigo,
        {$prodSkuExpr} AS sku,
        {$prodNameExpr} AS denominacion,
        {$selectLote},
        {$selectFProd},
        {$selectFVenc},
        i.uv_cajas,
        {$itemsExtraCols},
        ({$ucTotalExpr}) AS uc_total
      FROM wh_moves m
      JOIN {$PAL} p ON p.id = m.pallet_id
      JOIN {$PI} i ON i.pallet_id = p.id
      {$joinLote}
      LEFT JOIN para_productos pr ON pr.id = i.producto_id
      WHERE m.move_type='INBOUND' AND m.move_code LIKE ?
      ORDER BY p.id, i.id
    ";
    $sti = $pdo->prepare($sqlI);
    $sti->execute(['IN-'.$ingresoId.'-%']);
    $items = $sti->fetchAll(PDO::FETCH_ASSOC) ?: [];
  }

  $itTotUV = 0; $itTotUC = 0;
  foreach ($items as $it) {
    $itTotUV += (int)($it['uv_cajas'] ?? 0);
    $itTotUC += (int)($it['uc_total'] ?? 0);
  }

  // ---------- MOVIMIENTOS ----------
  $moves = [];
  if ($hasWhMoves && !empty($pallets)) {
    $palletIds = array_map(static fn($r) => (int)($r['pallet_id'] ?? 0), $pallets);
    $palletIds = array_values(array_filter($palletIds, static fn($id) => $id > 0));
    if ($palletIds) {
      $placeholders = implode(',', array_fill(0, count($palletIds), '?'));
      $posSelect = $hasOcc ? 'posTxt.posicion_codigo' : $posCodeExpr;
      $posJoinMov = $hasOcc
        ? "LEFT JOIN (\n        SELECT o1.pallet_id, {$posCodeExpr} AS posicion_codigo\n        FROM wh_position_occupancy o1\n        JOIN {$POS} pos ON pos.id = o1.position_id\n        JOIN (\n          SELECT pallet_id, MAX(id) AS max_id\n          FROM wh_position_occupancy\n          GROUP BY pallet_id\n        ) last_o ON last_o.pallet_id = o1.pallet_id AND last_o.max_id = o1.id\n      ) posTxt ON posTxt.pallet_id = p.id"
        : "LEFT JOIN {$POS} pos ON pos.id = p.posicion_id";

      $selectMoveCode   = $moveCodeCol   ? "m.{$moveCodeCol} AS move_code"         : "NULL AS move_code";
      $selectMoveStatus = $moveStatusCol ? "m.{$moveStatusCol} AS status"          : "NULL AS status";
      $selectMoveType   = $moveTypeCol   ? "m.{$moveTypeCol} AS tipo"              : "NULL AS tipo";
      $selectStart      = $moveStartCol  ? "m.{$moveStartCol} AS iniciado_at"      : "NULL AS iniciado_at";
      $selectEnd        = $moveEndCol    ? "m.{$moveEndCol} AS finalizado_at"      : "NULL AS finalizado_at";

      $where = [];
      $params = [];
      if ($moveTypeCol) {
        $where[] = "m.{$moveTypeCol} = ?";
        $params[] = ($moveTypeCol === 'tipo') ? 'IN' : 'INBOUND';
      }
      $where[] = "m.pallet_id IN ({$placeholders})";
      $params = array_merge($params, $palletIds);
      if ($moveCodeCol) {
        $where[] = "m.{$moveCodeCol} LIKE ?";
        $params[] = 'IN-'.$ingresoId.'-%';
      }

      $sqlM = "
        SELECT
          m.id,
          {$selectMoveCode},
          {$selectMoveStatus},
          {$selectMoveType},
          {$palletCodeExpr} AS pallet_codigo,
          {$posSelect} AS posicion_codigo,
          {$selectStart},
          {$selectEnd}
        FROM wh_moves m
        JOIN {$PAL} p ON p.id = m.pallet_id
        {$posJoinMov}
        WHERE " . implode(' AND ', $where) . "
        ORDER BY m.id ASC
      ";
      $stm = $pdo->prepare($sqlM);
      $stm->execute($params);
      $moves = $stm->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }
  } elseif ($hasWhMove && !empty($pallets)) {
    // Obtener movimientos IN del conjunto de pallets detectados
    $palletIds = array_map(fn($r) => (int)$r['pallet_id'], $pallets);
    $placeholders = implode(',', array_fill(0, count($palletIds), '?'));
    $posSelect = $hasOcc ? 'posTxt.posicion_codigo' : $posCodeExpr;
    $posJoinMov = $hasOcc
      ? "LEFT JOIN (\n        SELECT o1.pallet_id, {$posCodeExpr} AS posicion_codigo\n        FROM wh_position_occupancy o1\n        JOIN {$POS} pos ON pos.id = o1.position_id\n        JOIN (\n          SELECT pallet_id, MAX(id) AS max_id\n          FROM wh_position_occupancy\n          GROUP BY pallet_id\n        ) last_o ON last_o.pallet_id = o1.pallet_id AND last_o.max_id = o1.id\n      ) posTxt ON posTxt.pallet_id = p.id"
      : "LEFT JOIN {$POS} pos ON pos.id = p.posicion_id";
    $sqlM = "
      SELECT
        m.id,
        NULL AS move_code,
        NULL AS status,
        m.tipo,
        {$palletCodeExpr} AS pallet_codigo,
        {$posSelect} AS posicion_codigo,
        m.created_at AS iniciado_at,
        NULL AS finalizado_at
      FROM wh_move m
      JOIN {$PAL} p ON p.id = m.pallet_id
      {$posJoinMov}
      WHERE m.tipo='IN' AND m.pallet_id IN (".$placeholders.")
      ORDER BY m.id ASC
    ";
    $stm = $pdo->prepare($sqlM);
    $stm->execute($palletIds);
    $moves = $stm->fetchAll(PDO::FETCH_ASSOC) ?: [];
  }

  // ---------- RENDER PDF ----------
  $mpdf = new \Mpdf\Mpdf([
    'format' => 'A4',
    'margin_left'   => 10,
    'margin_right'  => 10,
    'margin_top'    => 14,
    'margin_bottom' => 12,
  ]);

  $title = "Detalle de ingreso #{$ingresoId}";
  $styles = '
    <style>
      body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 11px; }
      h1 { font-size: 18px; margin: 0 0 8px 0; }
      .muted { color:#666; }
      .grid { width:100%; border-collapse: collapse; margin-top:6px; }
      .grid th, .grid td { border: 0.6px solid #999; padding: 4px 6px; }
      .grid th { background: #f1f1f1; }
      .right { text-align:right; }
      .small { font-size: 10px; }
      .section { margin-top: 10px; }
      .badge { display:inline-block; padding:1px 6px; border:1px solid #999; border-radius:4px; }
      .kv { display:flex; flex-wrap:wrap; gap:12px; }
      .kv div { min-width: 160px; }
    </style>
  ';

  ob_start();
  ?>
  <?= $styles ?>
  <h1><?= htmlspecialchars($title) ?></h1>
  <div class="muted small">Generado: <?= date('Y-m-d H:i') ?></div>

  <div class="section">
    <strong>Meta datos</strong>
    <div class="kv">
      <table width="100%">
        <tr>
          <td width="12%"><b>Packing List:</b></td>
          <td><?= htmlspecialchars((string)$cab['packing_list']) ?></td>
          <td width="12%"><b>Depósito:</b></td>
          <td><?= htmlspecialchars((string)$cab['deposito']) ?></td>
        </tr>
        <tr>
          <td><b>Móvil:</b></td>
          <td><?= htmlspecialchars((string)$cab['movil']) ?></td>
          <td><b>Chofer:</b></td>
          <td><?= htmlspecialchars((string)$cab['chofer']) ?></td>
        </tr>
        <tr>
          <td><b>Fecha ingreso:</b></td>
          <td><?= htmlspecialchars((string)$cab['fecha']) ?></td>
          <td><b>Hora inicio:</b></td>
          <td><?= htmlspecialchars((string)$cab['inicio']) ?></td>
        </tr>
        <tr>
          <td><b>Operarios:</b></td>
          <td><?= htmlspecialchars((string)($cab['operarios'] ?? '-')) ?></td>
          <td><b>Documento:</b></td>
          <td><?= htmlspecialchars((string)$cab['doc_tipo']) ?> <?= htmlspecialchars((string)$cab['doc_numero']) ?></td>
        </tr>
        <tr>
          <td><b>Fecha doc.:</b></td>
          <td><?= htmlspecialchars((string)$cab['doc_fecha']) ?></td>
          <td><b>Observación:</b></td>
          <td><?= htmlspecialchars((string)$cab['obs']) ?></td>
        </tr>
      </table>
    </div>
  </div>

  <div class="section">
    <strong>Pallets del ingreso</strong>
    <table class="grid">
      <thead>
        <tr>
          <th>ID</th>
          <th>Código pallet</th>
          <th>Posición</th>
          <th>Ítems</th>
          <th class="right">UV total</th>
          <th class="right">UC totales</th>
        </tr>
      </thead>
      <tbody>
      <?php foreach ($pallets as $r): ?>
        <tr>
          <td><?= (int)$r['pallet_id'] ?></td>
          <td><?= htmlspecialchars((string)$r['pallet_codigo']) ?></td>
          <td><?= htmlspecialchars((string)$r['posicion']) ?></td>
          <td class="right"><?= nf($r['items_count']) ?></td>
          <td class="right"><?= nf($r['uv_total']) ?></td>
          <td class="right"><?= nf($r['uc_total']) ?></td>
        </tr>
      <?php endforeach; ?>
      </tbody>
      <tfoot>
        <tr>
          <th colspan="4" class="right">Totales</th>
          <th class="right"><?= nf($totUV) ?></th>
          <th class="right"><?= nf($totUC) ?></th>
        </tr>
      </tfoot>
    </table>
  </div>

  <div class="section">
    <strong>Ítems</strong>
    <table class="grid">
      <thead>
        <tr>
          <th>Pallet</th>
          <th>SKU</th>
          <th>Producto</th>
          <th>Lote</th>
          <th>F.Prod</th>
          <th>F.Vto</th>
          <th class="right">UV cajas</th>
          <th class="right">UC x caja</th>
          <th class="right">UC sueltas</th>
          <th class="right">UC totales</th>
        </tr>
      </thead>
      <tbody>
      <?php foreach ($items as $it): ?>
        <tr>
          <td><?= htmlspecialchars((string)$it['pallet_codigo']) ?></td>
          <td><?= htmlspecialchars((string)($it['sku'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($it['denominacion'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($it['lote'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($it['fecha_produccion'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($it['fecha_vencimiento'] ?? '')) ?></td>
          <td class="right"><?= nf($it['uv_cajas']) ?></td>
          <td class="right"><?= nf($it['uc_por_caja']) ?></td>
          <td class="right"><?= nf($it['uc_sueltas']) ?></td>
          <td class="right"><?= nf($it['uc_total']) ?></td>
        </tr>
      <?php endforeach; ?>
      </tbody>
      <tfoot>
        <tr>
          <th colspan="6" class="right">Totales</th>
          <th class="right"><?= nf($itTotUV) ?></th>
          <th></th>
          <th></th>
          <th class="right"><?= nf($itTotUC) ?></th>
        </tr>
      </tfoot>
    </table>
  </div>

  <div class="section">
    <strong>Movimientos de INGRESO</strong>
    <table class="grid">
      <thead>
        <tr>
          <th>ID</th>
          <th>Move code</th>
          <th>Tipo/Status</th>
          <th>Pallet</th>
          <th>Posición</th>
          <th>Iniciado</th>
          <th>Finalizado</th>
        </tr>
      </thead>
      <tbody>
      <?php foreach ($moves as $mv): ?>
        <tr>
          <td><?= (int)$mv['id'] ?></td>
          <td><?= htmlspecialchars((string)$mv['move_code']) ?></td>
          <td><?= htmlspecialchars((string)$mv['tipo']) ?> / <?= htmlspecialchars((string)$mv['status']) ?></td>
          <td><?= htmlspecialchars((string)$mv['pallet_codigo']) ?></td>
          <td><?= htmlspecialchars((string)($mv['posicion_codigo'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($mv['iniciado_at'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($mv['finalizado_at'] ?? '')) ?></td>
        </tr>
      <?php endforeach; ?>
      </tbody>
    </table>
  </div>
  <?php
  $html = ob_get_clean();

  $mpdf->SetTitle($title);
  $mpdf->SetAuthor((string)env('APP_NAME','SOL'));
  $mpdf->WriteHTML($html);
  $mpdf->Output("ingreso_{$ingresoId}.pdf", \Mpdf\Output\Destination::INLINE);

} catch (Throwable $e) {
  http_response_code(500);
  header('Content-Type: text/plain; charset=utf-8');
  echo (env('APP_ENV')==='local') ? ("Error generando PDF: ".$e->getMessage()) : "No se pudo generar el PDF";
}
