<?php
// api/operaciones/ingresos.php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

// En prod: 0; en dev: 1 (sin E_DEPRECATED)
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

// Raíz del proyecto (estás en /api/operaciones → subir 2 niveles)
$ROOT = dirname(__DIR__, 2);

// ---- Carga DB y Editor ----
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';
require_once $ROOT . '/app/Support/Auth.php';

// Forzar a DataTables Editor a usar nuestra conexión PDO (.env) en lugar del vendor/config.php por defecto
// Definimos $sql_details ANTES de incluir DataTables.php, así Bootstrap.php no cargará su config propia.
try {
  $pdo = getPDO();
  // Suficiente con pasar el PDO y el tipo
  $sql_details = [
    'type' => 'Mysql',
    'pdo'  => $pdo,
  ];
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['error' => 'No se pudo inicializar la DB', 'message' => $e->getMessage()], JSON_UNESCAPED_UNICODE);
  exit;
}

require_once $ROOT . '/vendor/Editor/lib/DataTables.php';

use
DataTables\Editor,
DataTables\Editor\Field,
DataTables\Editor\Format,
DataTables\Editor\Mjoin,
DataTables\Editor\Options,
DataTables\Editor\Validate,
DataTables\Editor\ValidateOptions;

try {
  // Sesión (para auditoría)
  if (session_status() === PHP_SESSION_NONE)
    session_start();
  $uid = current_user_id();
  // $uid = isset($_SESSION['usuario_id']) ? (int)$_SESSION['usuario_id'] : null;

  /**
   * MODELO USADO:
   *SELECT `id`, `packinglist_id`, `deposito_id`, `movil_id`, 
   *`chofer_id`, `fecha_ingreso`, `llegada_at`, `descarga_inicio_at`, 
   *`descarga_fin_at`, `operarios_cant`, `doc_tipo`, `doc_numero`, `doc_fecha`, 
   *`observacion`, `created_at`, `updated_at` FROM `pl_ingreso`
   * )
   */

  $editor = Editor::inst($db, 'pl_ingreso', 'pl_ingreso.id')
    ->fields(
      Field::inst('pl_ingreso.id')->set(false),

      // packinglist_id - required field in new table
      Field::inst('pl_ingreso.packinglist_id')
        ->validator(Validate::notEmpty(ValidateOptions::inst()->message('El packing list es requerido.')))
        ->options(
          Options::inst()
            ->table('pl_packinglist')
            ->value('id')
            ->label(['codigo'])
        ),

      // deposito_id - required field in new table  
      Field::inst('pl_ingreso.deposito_id')
        ->validator(Validate::notEmpty(ValidateOptions::inst()->message('El depósito es requerido.')))
        ->options(
          Options::inst()
            ->table('wh_deposito')
            ->value('id')
            ->label(['nombre'])
            ->where(fn($q) => $q->where('deleted_at', null))
        ),

      Field::inst('pl_ingreso.fecha_ingreso')
        ->validator(Validate::notEmpty(ValidateOptions::inst()->message('La fecha de ingreso es requerida.')))
        ->getFormatter(Format::datetime('Y-m-d', 'Y-m-d'))
        ->setFormatter(Format::datetime('Y-m-d', 'Y-m-d')),

      Field::inst('pl_ingreso.llegada_at')
        ->getFormatter(Format::datetime('H:i:s', 'H:i:s'))
        ->setFormatter(Format::datetime('H:i:s', 'H:i:s')),

      Field::inst('pl_ingreso.descarga_inicio_at')
        ->getFormatter(Format::datetime('H:i:s', 'H:i:s'))
        ->setFormatter(Format::datetime('H:i:s', 'H:i:s')),

      Field::inst('pl_ingreso.descarga_fin_at')
        ->getFormatter(Format::datetime('H:i:s', 'H:i:s'))
        ->setFormatter(Format::datetime('H:i:s', 'H:i:s')),

      Field::inst('pl_ingreso.doc_tipo')
        ->validator(function ($val) {
          return in_array($val, ['FACTURA', 'REMITO', 'OTRO'], true) ? true : 'Tipo de documento inválido.';
        }),
      
      Field::inst('pl_ingreso.doc_numero'),
      
      Field::inst('pl_ingreso.doc_fecha')
        ->getFormatter(Format::datetime('Y-m-d', 'Y-m-d'))
        ->setFormatter(Format::datetime('Y-m-d', 'Y-m-d')),

      // ---- movil_id ----
      Field::inst('pl_ingreso.movil_id')
        ->options(
          Options::inst()
            ->table('para_moviles')
            ->value('id')
            ->label(['chapa'])
            ->where(fn($q) => $q->where('deleted_at', null))
        )
        ->validator(
          Validate::values(
            ValidateOptions::inst()->message('El móvil seleccionado no existe o no está activo.')
          )
        ),

      // ---- chofer_id ----
      Field::inst('pl_ingreso.chofer_id')
        ->options(
          Options::inst()
            ->table('para_choferes')
            ->value('id')
            ->label(['nombre'])
        )
        ->validator(
          Validate::values(
            ValidateOptions::inst()->message('El chofer seleccionado no existe o no está activo.')
          )
        ),

      Field::inst('pl_ingreso.operarios_cant')
        ->set(Field::SET_BOTH) // permite setear en create/edit
        ->validator(Validate::numeric(
          ValidateOptions::inst()->message('Debe ser un número válido.')
        ))
        ->validator(function ($val) {
          return ((int) $val >= 1) ? true : 'Debe ser al menos 1.';
        }),
        
      Field::inst('pl_ingreso.observacion'),

      // --- Auditoría: permitir setear en CREATE/EDIT desde el hook ---
      Field::inst('pl_ingreso.created_at')->set(false),
      Field::inst('pl_ingreso.updated_at')->set(false),

      // Labels de joins
      Field::inst('para_moviles.chapa')->set(false),
      Field::inst('para_choferes.nombre')->set(false),
      Field::inst('wh_deposito.nombre')->set(false),
      Field::inst('pl_packinglist.codigo')->set(false)
    )
    ->leftJoin('para_moviles', 'para_moviles.id', '=', 'pl_ingreso.movil_id')
    ->leftJoin('para_choferes', 'para_choferes.id', '=', 'pl_ingreso.chofer_id')
    ->leftJoin('wh_deposito', 'wh_deposito.id', '=', 'pl_ingreso.deposito_id')
    ->leftJoin('pl_packinglist', 'pl_packinglist.id', '=', 'pl_ingreso.packinglist_id')
    ->debug(false);

  $editor->process($_POST)->json();
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['error' => 'Error del servidor', 'message' => $e->getMessage()], JSON_UNESCAPED_UNICODE);
}
