<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

// Helpers de introspección
function hasTable(PDO $pdo, string $table): bool {
  static $cache = [];
  if (array_key_exists($table, $cache)) return $cache[$table];
  $stmt = $pdo->prepare("
    SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES
    WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?
  ");
  $stmt->execute([$table]);
  return $cache[$table] = ((int)$stmt->fetchColumn() > 0);
}

function hasColumn(PDO $pdo, string $table, string $col): bool {
  static $cache = [];
  $key = $table.'|'.$col;
  if (array_key_exists($key, $cache)) return $cache[$key];
  $stmt = $pdo->prepare("
    SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?
  ");
  $stmt->execute([$table, $col]);
  return $cache[$key] = ((int)$stmt->fetchColumn() > 0);
}

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  $ingresoId = (int)($_GET['ingreso_id'] ?? 0);
  if ($ingresoId <= 0) {
    http_response_code(400);
    echo json_encode(['ok'=>false,'error'=>'ingreso_id requerido']);
    exit;
  }

  // --------- Descubrir columnas disponibles en pl_ingreso ----------
  $tblIng  = 'pl_ingreso';
  $cols = [
    'fecha_ingreso'        => hasColumn($pdo, $tblIng, 'fecha_ingreso'),
    'descarga_inicio_at'   => hasColumn($pdo, $tblIng, 'descarga_inicio_at'),
    'operarios_cant'       => hasColumn($pdo, $tblIng, 'operarios_cant'),
    'observacion'          => hasColumn($pdo, $tblIng, 'observacion'),
    'movil_id'             => hasColumn($pdo, $tblIng, 'movil_id'),
    'chofer_id'            => hasColumn($pdo, $tblIng, 'chofer_id'),
    'deposito_id'          => hasColumn($pdo, $tblIng, 'deposito_id'),
    'packinglist_id'       => hasColumn($pdo, $tblIng, 'packinglist_id'),
    'doc_tipo'             => hasColumn($pdo, $tblIng, 'doc_tipo'),
    'doc_numero'           => hasColumn($pdo, $tblIng, 'doc_numero'),
    'doc_fecha'            => hasColumn($pdo, $tblIng, 'doc_fecha'),
  ];

  // --------- Descubrir tablas/cols relacionadas ----------
  $hasMoviles   = hasTable($pdo, 'para_moviles');
  $movilLabel   = $hasMoviles && hasColumn($pdo, 'para_moviles', 'chapa') ? 'chapa'
                : ($hasMoviles && hasColumn($pdo, 'para_moviles', 'codigo') ? 'codigo' : null);

  $hasChoferes  = hasTable($pdo, 'para_choferes');
  $choferName   = $hasChoferes && hasColumn($pdo, 'para_choferes', 'nombre') ? 'nombre' : null;

  $hasDepositos = hasTable($pdo, 'wh_deposito');
  $depositoName = $hasDepositos && hasColumn($pdo, 'wh_deposito', 'nombre') ? 'nombre' : null;

  $hasPL = hasTable($pdo, 'pl_packinglist');
  $plLabel = $hasPL && hasColumn($pdo, 'pl_packinglist', 'codigo') ? 'codigo' : null;

  // --------- Construir SELECT seguro ----------
  $select = ["i.id"];
  if ($cols['fecha_ingreso'])       $select[] = "i.fecha_ingreso";
  if ($cols['descarga_inicio_at'])  $select[] = "i.descarga_inicio_at";
  if ($cols['operarios_cant'])      $select[] = "i.operarios_cant";
  if ($cols['observacion'])         $select[] = "i.observacion";
  if ($cols['movil_id'])            $select[] = "i.movil_id";
  if ($cols['chofer_id'])           $select[] = "i.chofer_id";
  if ($cols['deposito_id'])         $select[] = "i.deposito_id";
  if ($cols['packinglist_id'])      $select[] = "i.packinglist_id";
  if ($cols['doc_tipo'])            $select[] = "i.doc_tipo";
  if ($cols['doc_numero'])          $select[] = "i.doc_numero";
  if ($cols['doc_fecha'])           $select[] = "i.doc_fecha";

  $joins = [];
  if ($cols['movil_id'] && $hasMoviles && $movilLabel) {
    $joins[] = "LEFT JOIN para_moviles mv ON mv.id = i.movil_id";
    $select[] = "mv.$movilLabel AS movil";
  }
  if ($cols['chofer_id'] && $hasChoferes && $choferName) {
    $joins[] = "LEFT JOIN para_choferes ch ON ch.id = i.chofer_id";
    $select[] = "ch.$choferName AS chofer";
  }
  if ($cols['deposito_id'] && $hasDepositos && $depositoName) {
    $joins[] = "LEFT JOIN wh_deposito dep ON dep.id = i.deposito_id";
    $select[] = "dep.$depositoName AS deposito";
  }
  if ($cols['packinglist_id'] && $hasPL && $plLabel) {
    $joins[] = "LEFT JOIN pl_packinglist pl ON pl.id = i.packinglist_id";
    $select[] = "pl.$plLabel AS packing_list";
  }

  $sql = "SELECT ".implode(",\n       ", $select)."
          FROM {$tblIng} i
          ".implode("\n", array_unique($joins))."
          WHERE i.id = ?
          LIMIT 1";

  $st = $pdo->prepare($sql);
  $st->execute([$ingresoId]);
  $row = $st->fetch(PDO::FETCH_ASSOC);

  if (!$row) {
    echo json_encode(['ok'=>false, 'error'=>'Ingreso no encontrado']);
    exit;
  }

  // --------- Normalizar payload para la vista ----------
  $data = [
    'fecha_ingreso'      => $row['fecha_ingreso']      ?? null,
    'descarga_inicio_at' => $row['descarga_inicio_at'] ?? null,
    'operarios_cant'     => isset($row['operarios_cant']) ? (int)$row['operarios_cant'] : null,
    'observacion'        => $row['observacion']        ?? null,
    'movil'              => $row['movil']              ?? (isset($row['movil_id']) ? ('#'.$row['movil_id']) : null),
    'chofer'             => $row['chofer']             ?? (isset($row['chofer_id']) ? ('#'.$row['chofer_id']) : null),
    'deposito'           => $row['deposito']           ?? (isset($row['deposito_id']) ? ('#'.$row['deposito_id']) : null),
    'packing_list'       => $row['packing_list']       ?? (isset($row['packinglist_id']) ? ('#'.$row['packinglist_id']) : null),
    'doc_tipo'           => $row['doc_tipo']           ?? null,
    'doc_numero'         => $row['doc_numero']         ?? null,
    'doc_fecha'          => $row['doc_fecha']          ?? null,
  ];

  echo json_encode(['ok'=>true, 'data'=>$data]);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'ok'=>false,
    'error'=>'Error cargando cabecera',
    'msg' => (env('APP_ENV') === 'local') ? $e->getMessage() : ''
  ], JSON_UNESCAPED_UNICODE);
}
