<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function hasColumn(PDO $pdo, string $t, string $c): bool {
  static $cache = [];
  $k = $t.'|'.$c;
  if (array_key_exists($k, $cache)) return $cache[$k];
  $stmt = $pdo->prepare("\n    SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS\n    WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?\n  ");
  $stmt->execute([$t,$c]);
  $cache[$k] = ((int)$stmt->fetchColumn() > 0);
  return $cache[$k];
}

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  // Admite ids=1,2,3 (GET) o JSON body { ids: [1,2,3] }
  $ids = [];
  if (!empty($_GET['ids'])) {
    $ids = array_filter(array_map('intval', explode(',', (string)$_GET['ids'])), fn($v) => $v > 0);
  } else {
    $raw = file_get_contents('php://input');
    if ($raw) {
      $json = json_decode($raw, true);
      if (json_last_error() === JSON_ERROR_NONE && isset($json['ids']) && is_array($json['ids'])) {
        $ids = array_filter(array_map('intval', $json['ids']), fn($v) => $v > 0);
      }
    }
  }

  if (empty($ids)) {
    http_response_code(400);
    echo json_encode(['ok'=>false,'error'=>'Parámetro ids requerido (csv o JSON).']);
    exit;
  }

  // Determinar columna para identificar nombres de posición (para detectar CUARENTENA por nombre)
  $posColExpr = null;
  if (hasColumn($pdo, 'wh_posicion', 'code')) {
    $posColExpr = 'pos.code';
  } elseif (hasColumn($pdo, 'wh_posicion', 'pos_code')) {
    $posColExpr = 'pos.pos_code';
  } elseif (hasColumn($pdo, 'wh_posicion', 'title')) {
    $posColExpr = 'pos.title';
  } else {
    // Fallback que no permitirá detectar CUARENTENA por nombre, sólo por null
    $posColExpr = "CONCAT('POS ', pos.id)";
  }

  // Construir placeholders para IN
  $ph = implode(',', array_fill(0, count($ids), '?'));

  // Nota: Usamos DISTINCT en pallet.id por si pl_rcv_link tiene múltiples filas por pallet
  // Obtener id de estado CUARENTENA para contar pallets en ese estado
  $cuaId = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='CUARENTENA' LIMIT 1")->fetchColumn() ?: 0);

  $sql = "
    SELECT
      ing.id AS ingreso_id,
      COUNT(DISTINCT p.id) AS pallets_total,
      COUNT(DISTINCT CASE WHEN p.posicion_id IS NOT NULL AND NOT ( $posColExpr LIKE '%CUARENTENA%' OR $posColExpr LIKE '%CUAR%' OR $posColExpr LIKE '%QUA%' ) THEN p.id END) AS pallets_posicionados,
      COUNT(DISTINCT CASE WHEN p.estado_id = {$cuaId} THEN p.id END) AS pallets_cuarentena
    FROM pl_ingreso ing
    JOIN pl_rcv_link link ON link.packinglist_id = ing.packinglist_id
    JOIN wh_pallet p ON p.id = link.pallet_id AND p.deleted_at IS NULL
    LEFT JOIN wh_posicion pos ON pos.id = p.posicion_id
    WHERE ing.id IN ($ph)
    GROUP BY ing.id
  ";

  $st = $pdo->prepare($sql);
  foreach ($ids as $i => $val) {
    $st->bindValue($i + 1, $val, PDO::PARAM_INT);
  }
  $st->execute();
  $rows = $st->fetchAll(PDO::FETCH_ASSOC);

  // Normalizar a mapa ingreso_id => { total, posicionados, estado }
  $map = [];
  foreach ($rows as $r) {
    $tot = (int)$r['pallets_total'];
    $pos = (int)$r['pallets_posicionados'];
    $cua = (int)($r['pallets_cuarentena'] ?? 0);
    $estado = $tot > 0 && $pos >= $tot ? 'completo' : ($pos > 0 ? 'parcial' : 'ninguno');
    $map[(int)$r['ingreso_id']] = [
      'total' => $tot,
      'posicionados' => $pos,
      'estado' => $estado,
      'pallets_cuarentena' => $cua
    ];
  }

  echo json_encode(['ok'=>true,'data'=>$map]);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'ok'=>false,
    'error'=>'Error calculando estado de posicionamiento',
    'msg' => (env('APP_ENV') === 'local') ? $e->getMessage() : ''
  ], JSON_UNESCAPED_UNICODE);
}

?>
