<?php
/**
 * SOL · API · PDF Nota de Móviles disponibles por fecha
 * Ruta: api/operaciones/moviles_disponibles_pdf.php
 */
declare(strict_types=1);

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';

if (empty($_SESSION['usuario_id'])) {
    http_response_code(401);
    echo json_encode(['ok' => false, 'error' => 'No autorizado']);
    exit;
}

require_once $BASE . '/vendor/autoload.php';

use Mpdf\Mpdf;

try {
    $pdo = getPDO();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Fecha filtro
    $date = trim($_GET['date'] ?? '');
    if ($date === '' || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
        $date = date('Y-m-d');
    }

    // Query base
  $sql = "
        SELECT 
            md.id,
            md.fecha,
            md.disponible,
            md.ventana_ini,
            md.ventana_fin,
      md.llegada_time,
      md.no_asistio,
      md.no_utilizado,
            md.notas,
            m.chapa AS movil,
            t.nombre AS transportadora
        FROM oper_movil_disponible md
        LEFT JOIN para_moviles m ON m.id = md.movil_id
        LEFT JOIN para_transportadoras t ON t.id = m.transportadora_id
        WHERE md.fecha = ?
        ORDER BY m.chapa ASC, md.id ASC
    ";

    $st = $pdo->prepare($sql);
    $st->execute([$date]);
    $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

    // Calcular estadísticas para los porcentajes
    $totalMoviles = count($rows);
    $movilesDisponibles = 0;
    $movilesAsistieron = 0;
    $movilesUtilizados = 0;
    
    foreach ($rows as $r) {
        if ((string)($r['disponible'] ?? '') === '1') $movilesDisponibles++;
        if ((string)($r['no_asistio'] ?? '') === '1') $movilesAsistieron++; // Asistió = 1
        if ((string)($r['no_utilizado'] ?? '') === '1') $movilesUtilizados++; // Utilizado = 1
    }
    
    $pctAsistencia = $totalMoviles > 0 ? round(($movilesAsistieron / $totalMoviles) * 100, 1) : 0;
    $pctUtilizacion = $totalMoviles > 0 ? round(($movilesUtilizados / $totalMoviles) * 100, 1) : 0;

    // Helpers
    $h = fn($v) => htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8');
    $fmtDate = function(string $d): string { return $d ? date('d/m/Y', strtotime($d)) : '-'; };

    $mpdf = new Mpdf([
        'mode' => 'utf-8',
        'format' => 'A4-L', // Landscape (horizontal)
        'default_font' => 'sans-serif',
        'margin_left' => 10,
        'margin_right' => 10,
        'margin_top' => 30,
        'margin_bottom' => 15,
        'margin_header' => 5,
        'margin_footer' => 8,
    ]);

    $mpdf->SetTitle('Móviles disponibles · ' . $date);
    $mpdf->SetAuthor('SOL System');

    ob_start();
    ?>
<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8" />
  <style>
    body { font-family: sans-serif; font-size: 9pt; color: #333; }
    .header { text-align: center; margin-bottom: 15px; }
    .header h1 { font-size: 16pt; margin: 0; color: #2c5aa0; }
    .header .sub { font-size: 9pt; color: #666; margin-top: 4px; }
    table { width: 100%; border-collapse: collapse; font-size: 8pt; }
    th, td { border: 1px solid #dee2e6; padding: 4px 3px; vertical-align: top; }
    th { background: #f8f9fa; color: #495057; text-align: left; font-weight: bold; }
    .text-center { text-align: center; }
    .badge { display: inline-block; padding: 2px 5px; border-radius: 3px; font-size: 7pt; font-weight: bold; white-space: nowrap; }
    .b-ok { background: #d4edda; color: #155724; }
    .b-no { background: #f8d7da; color: #721c24; }
    .b-warn { background: #fff3cd; color: #856404; }
    .b-muted { background: #e2e3e5; color: #41464b; }
    .footer { margin-top: 15px; font-size: 8pt; color: #666; text-align: center; }
    .stats-box { margin: 15px 0; padding: 15px; background: #f8f9fa; border: 1px solid #dee2e6; border-radius: 5px; }
    .stats-row { width: 100%; border-collapse: collapse; }
    .stats-col { width: 33.33%; text-align: center; padding: 10px; border-right: 1px solid #dee2e6; }
    .stats-col:last-child { border-right: none; }
    .stats-title { font-size: 11pt; font-weight: bold; color: #2c5aa0; margin-bottom: 8px; }
    .stats-value { font-size: 20pt; font-weight: bold; margin: 5px 0; }
    .stats-detail { font-size: 8pt; color: #666; margin-top: 3px; }
  </style>
</head>
<body>
  <div class="header">
    <h1>Nota · Móviles disponibles</h1>
    <div class="sub">Fecha: <?= $h($fmtDate($date)) ?> · Generado: <?= $h(date('d/m/Y H:i')) ?> · Usuario: <?= $h($_SESSION['usuario_nombre'] ?? 'Sistema') ?></div>
    
    <!-- Estadísticas de cumplimiento y ocupación -->
    <div class="stats-box">
      <table class="stats-row">
        <tr>
          <td class="stats-col">
            <div class="stats-title">% Asistencia</div>
            <div class="stats-value" style="color: <?= $pctAsistencia >= 80 ? '#28a745' : ($pctAsistencia >= 60 ? '#ffc107' : '#dc3545') ?>;">
              <?= $pctAsistencia ?>%
            </div>
            <div class="stats-detail"><?= $movilesAsistieron ?> de <?= $totalMoviles ?> asistieron</div>
          </td>
          <td class="stats-col">
            <div class="stats-title">% Utilización</div>
            <div class="stats-value" style="color: <?= $pctUtilizacion >= 70 ? '#28a745' : ($pctUtilizacion >= 50 ? '#ffc107' : '#dc3545') ?>;">
              <?= $pctUtilizacion ?>%
            </div>
            <div class="stats-detail"><?= $movilesUtilizados ?> de <?= $totalMoviles ?> utilizados</div>
          </td>
          <td class="stats-col">
            <div class="stats-title">Total Móviles</div>
            <div class="stats-value" style="color: #6c757d;">
              <?= $totalMoviles ?>
            </div>
            <div class="stats-detail"><?= $movilesDisponibles ?> disponibles</div>
          </td>
        </tr>
      </table>
    </div>
  </div>

  <table>
    <thead>
      <tr>
        <th style="width:15%">Móvil</th>
        <th style="width:20%">Transportadora</th>
        <th style="width:10%">Disponible</th>
        <th style="width:10%">Llegada</th>
        <th style="width:10%">Asistió</th>
        <th style="width:10%">Utilizado</th>
        <th style="width:12%">Ventana ini</th>
        <th style="width:12%">Ventana fin</th>
        <th style="width:11%">Notas</th>
      </tr>
    </thead>
    <tbody>
      <?php if (!$rows): ?>
  <tr><td colspan="9" class="text-center">Sin registros para la fecha</td></tr>
      <?php else: ?>
        <?php foreach ($rows as $r): ?>
          <tr>
            <td><?= $h($r['movil'] ?? '-') ?></td>
            <td><?= $h($r['transportadora'] ?? '-') ?></td>
            <td class="text-center">
              <?php $ok = (string)($r['disponible'] ?? '') === '1'; ?>
              <span class="badge <?= $ok ? 'b-ok' : 'b-no' ?>"><?= $ok ? 'Sí' : 'No' ?></span>
            </td>
            <td class="text-center"><?= $h($r['llegada_time'] ?? '') ?></td>
            <td class="text-center">
              <?php $asistio = (string)($r['no_asistio'] ?? '') === '1'; ?>
              <span class="badge <?= $asistio ? 'b-ok' : 'b-no' ?>"><?= $asistio ? 'Sí' : 'No' ?></span>
            </td>
            <td class="text-center">
              <?php $utilizado = (string)($r['no_utilizado'] ?? '') === '1'; ?>
              <span class="badge <?= $utilizado ? 'b-ok' : 'b-no' ?>"><?= $utilizado ? 'Sí' : 'No' ?></span>
            </td>
            <td class="text-center"><?= $h($r['ventana_ini'] ?? '') ?></td>
            <td class="text-center"><?= $h($r['ventana_fin'] ?? '') ?></td>
            <td><?= $h($r['notas'] ?? '') ?></td>
          </tr>
        <?php endforeach; ?>
      <?php endif; ?>
    </tbody>
  </table>

  <div class="footer">SOL · <?= $h(date('Y')) ?></div>
</body>
</html>
<?php
    $html = ob_get_clean();
    $mpdf->WriteHTML($html);
    $mpdf->Output('moviles-disponibles-' . $date . '.pdf', 'I');

} catch (Throwable $e) {
    error_log('PDF moviles disponibles error: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['ok' => false, 'error' => 'Error generando PDF'], JSON_UNESCAPED_UNICODE);
}
