<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';
require_once $ROOT . '/app/Support/ApiHelpers.php';

session_start();
if (empty($_SESSION['usuario_id'])) {
    json_err('No autenticado', null, 401);
}

try {
    $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
    if ($method !== 'POST') {
        json_err('Método no permitido', null, 405);
    }

    $payload = getJsonInput();
    if (!$payload) {
        $payload = $_POST;
    }

    $fecha = isset($payload['fecha']) ? trim((string)$payload['fecha']) : '';
    if ($fecha === '') {
        $fecha = date('Y-m-d');
    }
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $fecha)) {
        json_err('Fecha inválida (usar formato YYYY-MM-DD)', null, 422);
    }

    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');
    $pdo->beginTransaction();

    $conditions = [];
    if (columnExists($pdo, 'para_moviles', 'disponibles')) {
        $conditions[] = 'COALESCE(pm.disponibles, 0) = 1';
    }
    if (columnExists($pdo, 'para_moviles', 'activo')) {
        $conditions[] = 'COALESCE(pm.activo, 0) = 1';
    }
    if (columnExists($pdo, 'para_moviles', 'deleted_at')) {
        $conditions[] = 'pm.deleted_at IS NULL';
    }

    $whereCatalog = $conditions ? ('WHERE ' . implode(' AND ', $conditions)) : '';

    $sqlExistingCatalog = 'SELECT COUNT(*)
                             FROM oper_movil_disponible od
                             JOIN para_moviles pm ON pm.id = od.movil_id ' .
                           ($conditions ? ('WHERE ' . implode(' AND ', array_merge($conditions, ['od.fecha = :fecha']))) : 'WHERE od.fecha = :fecha');
    $stmtExistingCatalog = $pdo->prepare($sqlExistingCatalog);
    $stmtExistingCatalog->execute([':fecha' => $fecha]);
    $existingCatalog = (int)$stmtExistingCatalog->fetchColumn();

    $sqlInsert = "INSERT IGNORE INTO oper_movil_disponible (fecha, movil_id, disponible)\n                   SELECT :fecha, pm.id, 0\n                     FROM para_moviles pm\n                     $whereCatalog";
    $stmtInsert = $pdo->prepare($sqlInsert);
    $stmtInsert->execute([':fecha' => $fecha]);
    $inserted = (int)$stmtInsert->rowCount();

    $sqlCountCandidates = "SELECT COUNT(*) FROM para_moviles pm $whereCatalog";
    $stmtCandidates = $pdo->prepare($sqlCountCandidates);
    $stmtCandidates->execute();
    $candidatos = (int)$stmtCandidates->fetchColumn();

    $stmtTotalFecha = $pdo->prepare('SELECT COUNT(*) FROM oper_movil_disponible WHERE fecha = :fecha');
    $stmtTotalFecha->execute([':fecha' => $fecha]);
    $totalFecha = (int)$stmtTotalFecha->fetchColumn();

    $pdo->commit();

    json_ok([
        'fecha' => $fecha,
        'inserted' => $inserted,
        'candidatos' => $candidatos,
        'ya_existentes' => $existingCatalog,
        'total_registros_fecha' => $totalFecha
    ]);
} catch (Throwable $e) {
    if (isset($pdo) && $pdo instanceof PDO && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    json_err('No se pudo cargar el catálogo del día', $e->getMessage(), 500);
}
