<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/db.php';
require_once $ROOT . '/app/Support/ApiHelpers.php';

function sendResponse(array $payload, int $status = 200): void
{
    http_response_code($status);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

try {
    $pdo = getPDO();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');

    if (!tableExists($pdo, 'para_moviles')) {
        sendResponse(['ok' => true, 'data' => []]);
    }

    $hasDisponibles = tableExists($pdo, 'oper_movil_disponible');
    $hasTransportadora = tableExists($pdo, 'para_transportadoras') && columnExists($pdo, 'para_moviles', 'transportadora_id');
    $hasOcupado = columnExists($pdo, 'para_moviles', 'ocupado');
    $hasLlegadaTime = $hasDisponibles && columnExists($pdo, 'oper_movil_disponible', 'llegada_time');
    $hasNoAsistio = $hasDisponibles && columnExists($pdo, 'oper_movil_disponible', 'no_asistio');
    $hasNoUtilizado = $hasDisponibles && columnExists($pdo, 'oper_movil_disponible', 'no_utilizado');

    $fechaParam = isset($_GET['fecha']) ? trim((string) $_GET['fecha']) : '';
    $fecha = date('Y-m-d');
    if ($fechaParam !== '') {
        $dt = DateTimeImmutable::createFromFormat('Y-m-d', $fechaParam);
        if ($dt instanceof DateTimeImmutable) {
            $fecha = $dt->format('Y-m-d');
        }
    }

    $q = isset($_GET['q']) ? trim((string) $_GET['q']) : '';

    $includeIds = [];
    if (isset($_GET['include_id'])) {
        $includeIds[] = (int) $_GET['include_id'];
    }
    if (!empty($_GET['include_ids'])) {
        $chunks = explode(',', (string) $_GET['include_ids']);
        foreach ($chunks as $chunk) {
            $id = (int) trim($chunk);
            if ($id > 0) {
                $includeIds[] = $id;
            }
        }
    }
    $includeIds = array_values(array_unique(array_filter($includeIds, static fn(int $id): bool => $id > 0)));

    // Si no existe tabla de disponibilidad, usar listado básico de móviles activos
    if (!$hasDisponibles) {
        $sql = 'SELECT m.id, m.chapa, m.activo' . ($hasOcupado ? ', m.ocupado' : ', 0 AS ocupado');
        if ($hasTransportadora) {
            $sql .= ', t.nombre AS transportadora';
        }
        $sql .= ' FROM para_moviles m';
        if ($hasTransportadora) {
            $sql .= ' LEFT JOIN para_transportadoras t ON t.id = m.transportadora_id';
        }
        $sql .= ' WHERE m.activo = 1';
        $params = [];
        if ($q !== '') {
            $sql .= ' AND m.chapa LIKE :q';
            $params[':q'] = '%' . $q . '%';
        }
        $sql .= ' ORDER BY m.chapa LIMIT 200';

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $map = [];
        foreach ($rows as $row) {
            $id = (int) $row['id'];
            $labelParts = [trim((string) ($row['chapa'] ?? ''))];
            if ($hasTransportadora && !empty($row['transportadora'])) {
                $labelParts[] = trim((string) $row['transportadora']);
            }
            $label = trim(implode(' · ', array_filter($labelParts)));
            if ($label === '') {
                $label = '#' . $id;
            }
            $map[$id] = [
                'id' => $id,
                'chapa' => (string) ($row['chapa'] ?? ''),
                'label' => $label,
                'activo' => isset($row['activo']) ? (int) $row['activo'] : 1,
                'ocupado' => $hasOcupado ? (int) ($row['ocupado'] ?? 0) : null,
            ] + ($hasTransportadora ? ['transportadora' => $row['transportadora']] : []);
        }

        if ($includeIds) {
            $placeholders = implode(',', array_fill(0, count($includeIds), '?'));
            $sqlIncl = 'SELECT m.id, m.chapa, m.activo' . ($hasOcupado ? ', m.ocupado' : ', 0 AS ocupado');
            if ($hasTransportadora) {
                $sqlIncl .= ', t.nombre AS transportadora';
            }
            $sqlIncl .= ' FROM para_moviles m';
            if ($hasTransportadora) {
                $sqlIncl .= ' LEFT JOIN para_transportadoras t ON t.id = m.transportadora_id';
            }
            $sqlIncl .= " WHERE m.id IN ($placeholders)";

            $stmtIncl = $pdo->prepare($sqlIncl);
            $stmtIncl->execute($includeIds);
            $rowsIncl = $stmtIncl->fetchAll(PDO::FETCH_ASSOC) ?: [];
            foreach ($rowsIncl as $row) {
                $id = (int) $row['id'];
                $labelParts = [trim((string) ($row['chapa'] ?? ''))];
                if ($hasTransportadora && !empty($row['transportadora'])) {
                    $labelParts[] = trim((string) $row['transportadora']);
                }
                $label = trim(implode(' · ', array_filter($labelParts)));
                if ($label === '') {
                    $label = '#' . $id;
                }
                $map[$id] = [
                    'id' => $id,
                    'chapa' => (string) ($row['chapa'] ?? ''),
                    'label' => $label,
                    'activo' => isset($row['activo']) ? (int) $row['activo'] : 1,
                    'ocupado' => $hasOcupado ? (int) ($row['ocupado'] ?? 0) : null,
                    'fuera_catalogo' => true,
                ] + ($hasTransportadora ? ['transportadora' => $row['transportadora']] : []);
            }
        }

        $payload = array_values($map);
        usort($payload, static function (array $a, array $b): int {
            return strcmp((string) ($a['chapa'] ?? ''), (string) ($b['chapa'] ?? ''));
        });

        sendResponse(['ok' => true, 'data' => $payload, 'fecha' => $fecha, 'fallback' => true]);
    }

    // Construir SELECT principal con disponibilidad del día
    $select = [
        'm.id',
        'm.chapa',
        'm.activo',
        $hasOcupado ? 'm.ocupado' : '0 AS ocupado',
        'omd.fecha',
        'omd.disponible',
        'omd.ventana_ini',
        'omd.ventana_fin',
    ];
    if ($hasTransportadora) {
        $select[] = 't.nombre AS transportadora';
    }
    if ($hasLlegadaTime) {
        $select[] = 'omd.llegada_time';
    }
    if ($hasNoAsistio) {
        $select[] = 'omd.no_asistio';
    }
    if ($hasNoUtilizado) {
        $select[] = 'omd.no_utilizado';
    }

    $sql = 'SELECT ' . implode(', ', $select)
         . ' FROM oper_movil_disponible omd'
         . ' JOIN para_moviles m ON m.id = omd.movil_id';
    if ($hasTransportadora) {
        $sql .= ' LEFT JOIN para_transportadoras t ON t.id = m.transportadora_id';
    }

    $conditions = [
        'omd.fecha = :fecha',
        'omd.disponible = 1',
        'm.activo = 1',
    ];
    if ($hasOcupado) {
        $conditions[] = 'm.ocupado = 0';
    }
    if ($hasNoAsistio) {
        $conditions[] = '(omd.no_asistio IS NULL OR omd.no_asistio = 0)';
    }
    if ($hasNoUtilizado) {
        $conditions[] = '(omd.no_utilizado IS NULL OR omd.no_utilizado = 0)';
    }
    $params = [':fecha' => $fecha];
    if ($q !== '') {
        $conditions[] = 'm.chapa LIKE :q';
        $params[':q'] = '%' . $q . '%';
    }

    $sql .= ' WHERE ' . implode(' AND ', $conditions)
        . ' ORDER BY COALESCE(omd.ventana_ini, "23:59:59"), m.chapa';

    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $map = [];
    foreach ($rows as $row) {
        $id = (int) $row['id'];
        $labelParts = [trim((string) ($row['chapa'] ?? ''))];
        $window = '';
        $ini = isset($row['ventana_ini']) ? trim((string) $row['ventana_ini']) : '';
        $fin = isset($row['ventana_fin']) ? trim((string) $row['ventana_fin']) : '';
        if ($ini !== '' || $fin !== '') {
            $window = trim($ini . ' - ' . $fin);
        }
        if ($window !== '' && $window !== '-') {
            $labelParts[] = $window;
        }
        if ($hasTransportadora && !empty($row['transportadora'])) {
            $labelParts[] = trim((string) $row['transportadora']);
        }
        $label = trim(implode(' · ', array_filter($labelParts)));
        if ($label === '') {
            $label = '#'.$id;
        }

        $map[$id] = [
            'id' => $id,
            'chapa' => (string) ($row['chapa'] ?? ''),
            'label' => $label,
            'fecha' => $row['fecha'],
            'ventana_ini' => $row['ventana_ini'],
            'ventana_fin' => $row['ventana_fin'],
            'disponible' => isset($row['disponible']) ? (int) $row['disponible'] : null,
            'activo' => isset($row['activo']) ? (int) $row['activo'] : null,
            'ocupado' => $hasOcupado ? (int) ($row['ocupado'] ?? 0) : null,
        ]
        + ($hasTransportadora ? ['transportadora' => $row['transportadora']] : [])
        + ($hasLlegadaTime ? ['llegada_time' => $row['llegada_time']] : [])
        + ($hasNoAsistio ? ['no_asistio' => $row['no_asistio']] : [])
        + ($hasNoUtilizado ? ['no_utilizado' => $row['no_utilizado']] : []);
    }

    if ($includeIds) {
        $existingIds = array_map(static fn(array $row): int => (int) $row['id'], $rows);
        $missing = array_values(array_diff($includeIds, $existingIds));
        if ($missing) {
            $placeholders = implode(',', array_fill(0, count($missing), '?'));
            $selectMissing = [
                'm.id',
                'm.chapa',
                'm.activo',
                $hasOcupado ? 'm.ocupado' : '0 AS ocupado',
                'omd.fecha',
                'omd.disponible',
                'omd.ventana_ini',
                'omd.ventana_fin',
            ];
            if ($hasTransportadora) {
                $selectMissing[] = 't.nombre AS transportadora';
            }
            if ($hasLlegadaTime) {
                $selectMissing[] = 'omd.llegada_time';
            }
            if ($hasNoAsistio) {
                $selectMissing[] = 'omd.no_asistio';
            }
            if ($hasNoUtilizado) {
                $selectMissing[] = 'omd.no_utilizado';
            }

            $sqlMissing = 'SELECT ' . implode(', ', $selectMissing)
                        . ' FROM para_moviles m'
                        . ' LEFT JOIN oper_movil_disponible omd ON omd.movil_id = m.id AND omd.fecha = ?';
            if ($hasTransportadora) {
                $sqlMissing .= ' LEFT JOIN para_transportadoras t ON t.id = m.transportadora_id';
            }
            $sqlMissing .= " WHERE m.id IN ($placeholders)";

            $stmtMissing = $pdo->prepare($sqlMissing);
            $stmtMissing->execute(array_merge([$fecha], $missing));
            $rowsMissing = $stmtMissing->fetchAll(PDO::FETCH_ASSOC) ?: [];

            foreach ($rowsMissing as $row) {
                $id = (int) $row['id'];
                $labelParts = [trim((string) ($row['chapa'] ?? ''))];
                $window = '';
                $ini = isset($row['ventana_ini']) ? trim((string) $row['ventana_ini']) : '';
                $fin = isset($row['ventana_fin']) ? trim((string) $row['ventana_fin']) : '';
                if ($ini !== '' || $fin !== '') {
                    $window = trim($ini . ' - ' . $fin);
                }
                if ($window !== '' && $window !== '-') {
                    $labelParts[] = $window;
                }
                if ($hasTransportadora && !empty($row['transportadora'])) {
                    $labelParts[] = trim((string) $row['transportadora']);
                }
                $label = trim(implode(' · ', array_filter($labelParts)));
                if ($label === '') {
                    $label = '#'.$id;
                }

                $map[$id] = [
                    'id' => $id,
                    'chapa' => (string) ($row['chapa'] ?? ''),
                    'label' => $label,
                    'fecha' => $row['fecha'],
                    'ventana_ini' => $row['ventana_ini'],
                    'ventana_fin' => $row['ventana_fin'],
                    'disponible' => isset($row['disponible']) ? (int) $row['disponible'] : null,
                    'activo' => isset($row['activo']) ? (int) $row['activo'] : null,
                    'ocupado' => $hasOcupado ? (int) ($row['ocupado'] ?? 0) : null,
                    'fuera_catalogo' => true,
                ]
                + ($hasTransportadora ? ['transportadora' => $row['transportadora']] : [])
                + ($hasLlegadaTime ? ['llegada_time' => $row['llegada_time']] : [])
                + ($hasNoAsistio ? ['no_asistio' => $row['no_asistio']] : [])
                + ($hasNoUtilizado ? ['no_utilizado' => $row['no_utilizado']] : []);
            }
        }
    }

    $data = array_values($map);
    usort($data, static function (array $a, array $b): int {
        $aWin = trim((string) ($a['ventana_ini'] ?? ''));
        $bWin = trim((string) ($b['ventana_ini'] ?? ''));
        if ($aWin === '' && $bWin === '') {
            return strcmp((string) ($a['chapa'] ?? ''), (string) ($b['chapa'] ?? ''));
        }
        if ($aWin === '') {
            return 1;
        }
        if ($bWin === '') {
            return -1;
        }
        $cmp = strcmp($aWin, $bWin);
        if ($cmp !== 0) {
            return $cmp;
        }
        return strcmp((string) ($a['chapa'] ?? ''), (string) ($b['chapa'] ?? ''));
    });

    sendResponse(['ok' => true, 'data' => $data, 'fecha' => $fecha]);
} catch (Throwable $e) {
    sendResponse([
        'ok' => false,
        'error' => 'No se pudo listar los móviles disponibles',
        'message' => $e->getMessage(),
    ], 500);
}
