<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $ingresoId = (int)($_GET['ingreso_id'] ?? 0);
    if ($ingresoId <= 0) {
        http_response_code(400);
        echo json_encode(['error'=>'ingreso_id requerido']);
        exit;
    }

    // Función helper para verificar columnas (igual que en moves_por_ingreso.php)
    $cache = [];
    function hasColumn($pdo, $table, $col) {
        global $cache;
        $key = "{$table}.{$col}";
        if (isset($cache[$key])) return $cache[$key];
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM information_schema.columns WHERE table_name=? AND column_name=? AND table_schema=DATABASE()");
        $stmt->execute([$table, $col]);
        $cache[$key] = ((int)$stmt->fetchColumn() > 0);
        return $cache[$key];
    }

    if (!function_exists('pallets_en_cuarentena_dedupe')) {
        function pallets_en_cuarentena_dedupe(array $rows): array {
            if (!$rows) {
                return [];
            }
            $byKey = [];
            foreach ($rows as $row) {
                $id = $row['id'] ?? null;
                $code = $row['codigo_pallet'] ?? null;
                $key = $id !== null ? 'id:' . $id : ($code !== null ? 'code:' . $code : spl_object_id((object)$row));

                if (!isset($byKey[$key])) {
                    $byKey[$key] = $row;
                    continue;
                }

                // merge: keep max cantidad_recibida and prefer non-empty descriptive fields
                $byKey[$key]['cantidad_recibida'] = max(
                    (int)($byKey[$key]['cantidad_recibida'] ?? 0),
                    (int)($row['cantidad_recibida'] ?? 0)
                );

                foreach (['posicion_codigo', 'producto_nombre', 'lote', 'fecha_vencimiento', 'ambiente_id'] as $col) {
                    if (empty($byKey[$key][$col]) && !empty($row[$col])) {
                        $byKey[$key][$col] = $row[$col];
                    }
                }
            }

            return array_values($byKey);
        }
    }

    if (!function_exists('pallets_en_cuarentena_should_include')) {
        function pallets_en_cuarentena_should_include(array $row, int $ambCuaId): bool {
            $posId = isset($row['posicion_id']) ? (int)$row['posicion_id'] : 0;
            if ($posId <= 0) {
                return true;
            }
            $ambId = isset($row['ambiente_id']) ? (int)$row['ambiente_id'] : 0;
            if ($ambCuaId > 0 && $ambId === $ambCuaId) {
                return true;
            }
            $code = strtoupper((string)($row['posicion_codigo'] ?? ''));
            if ($code === '' || $code === 'SIN POSICIÓN' || $code === 'N/A') {
                return true;
            }
            if (strpos($code, 'CUAR') !== false || strpos($code, 'QUAR') !== false || strpos($code, 'CUA') !== false) {
                return true;
            }
            return false;
        }
    }

    // Determinar qué columna usar para posición
    $posTable = 'wh_posicion';
    $displayExpr = null;
    if (hasColumn($pdo, $posTable, 'code')) {
        $displayExpr = "pos.code";
    } elseif (hasColumn($pdo, $posTable, 'pos_code')) {
        $displayExpr = "pos.pos_code";
    } elseif (hasColumn($pdo, $posTable, 'title')) {
        $displayExpr = "pos.title";
    } else {
        $displayExpr = "CONCAT('POS ', pos.id)";
    }

    // Obtener id estado CUARENTENA y ambiente CUARENTENA
    $cuaId = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='CUARENTENA' LIMIT 1")->fetchColumn() ?: 0);
    $ambCuaId = (int)($pdo->query("SELECT id FROM wh_ambiente WHERE code='CUARENTENA' LIMIT 1")->fetchColumn() ?: 0);

    // Obtener pallets en CUARENTENA relacionados con el ingreso (por estado_id) — estrategia 1: via pl_rcv_link
    $sql1 = "
        SELECT 
            p.id,
            p.codigo AS codigo_pallet,
            COALESCE(MIN(l.codigo), 'N/A') AS lote,
            COALESCE(DATE_FORMAT(MIN(l.fecha_vencimiento), '%Y-%m-%d'), 'N/A') AS fecha_vencimiento,
            COUNT(DISTINCT link.id) AS cantidad_recibida,
            p.posicion_id,
            CASE WHEN COUNT(DISTINCT prod.id) > 1 THEN 'Mixto' ELSE COALESCE(MAX(prod.denominacion), 'Producto') END AS producto_nombre,
            COALESCE({$displayExpr}, 'Sin posición') AS posicion_codigo,
            amb.id AS ambiente_id
        FROM pl_ingreso ing
        JOIN pl_rcv_link link ON link.packinglist_id = ing.packinglist_id
        JOIN wh_pallet p ON p.id = link.pallet_id AND p.deleted_at IS NULL
        LEFT JOIN wh_posicion pos ON pos.id = p.posicion_id
        LEFT JOIN wh_ambiente amb ON amb.id = pos.ambiente_id
        LEFT JOIN wh_lote l ON l.id = link.lote_id
        LEFT JOIN para_productos prod ON prod.id = link.producto_id
        WHERE ing.id = ?
          AND (
                p.estado_id = {$cuaId}
             OR (amb.id IS NOT NULL AND amb.id = {$ambCuaId})
             OR {$displayExpr} LIKE '%CUARENTENA%' OR {$displayExpr} LIKE '%CUAR%' OR {$displayExpr} LIKE '%QUA%'
          )
    GROUP BY p.id, p.codigo, p.posicion_id, {$displayExpr}, amb.id
        ORDER BY p.codigo
    ";

    $stmt = $pdo->prepare($sql1);
    $stmt->execute([$ingresoId]);
    $pallets = pallets_en_cuarentena_dedupe($stmt->fetchAll(PDO::FETCH_ASSOC));
    $pallets = array_values(array_filter($pallets, static function(array $row) use ($ambCuaId): bool {
        return pallets_en_cuarentena_should_include($row, $ambCuaId);
    }));

    // Estrategia 2: si no hay filas, intentar FK directo en wh_pallet (pl_ingreso_id o ingreso_id)
    if (empty($pallets)) {
        $hasPalIng = hasColumn($pdo, 'wh_pallet', 'pl_ingreso_id');
        $hasIng = hasColumn($pdo, 'wh_pallet', 'ingreso_id');
                if ($hasPalIng || $hasIng) {
            $fkField = $hasPalIng ? 'p.pl_ingreso_id' : 'p.ingreso_id';
            $sql2 = "
                SELECT 
                    p.id,
                    p.codigo AS codigo_pallet,
                    COALESCE(MIN(l.codigo), 'N/A') AS lote,
                    COALESCE(DATE_FORMAT(MIN(l.fecha_vencimiento), '%Y-%m-%d'), 'N/A') AS fecha_vencimiento,
                    COUNT(DISTINCT pi.id) AS cantidad_recibida,
                    p.posicion_id,
                    CASE WHEN COUNT(DISTINCT prod.id) > 1 THEN 'Mixto' ELSE COALESCE(MAX(prod.denominacion), 'Producto') END AS producto_nombre,
                    COALESCE({$displayExpr}, 'Sin posición') AS posicion_codigo,
                    amb.id AS ambiente_id
                FROM wh_pallet p
                LEFT JOIN wh_posicion pos ON pos.id = p.posicion_id
                                LEFT JOIN wh_ambiente amb ON amb.id = pos.ambiente_id
                                LEFT JOIN wh_pallet_item pi ON pi.pallet_id = p.id
                                LEFT JOIN wh_lote l ON l.id = pi.lote_id
                                LEFT JOIN para_productos prod ON prod.id = pi.producto_id
                WHERE {$fkField} = ?
                  AND p.deleted_at IS NULL
                                    AND (
                                                p.estado_id = {$cuaId}
                                         OR (amb.id IS NOT NULL AND amb.id = {$ambCuaId})
                                         OR {$displayExpr} LIKE '%CUARENTENA%' OR {$displayExpr} LIKE '%CUAR%' OR {$displayExpr} LIKE '%QUA%'
                                    )
                                GROUP BY p.id, p.codigo, p.posicion_id, {$displayExpr}, amb.id
                ORDER BY p.codigo
            ";
            $st2 = $pdo->prepare($sql2);
            $st2->execute([$ingresoId]);
            $pallets = pallets_en_cuarentena_dedupe($st2->fetchAll(PDO::FETCH_ASSOC));
            $pallets = array_values(array_filter($pallets, static function(array $row) use ($ambCuaId): bool {
                return pallets_en_cuarentena_should_include($row, $ambCuaId);
            }));
        }
    }

    // Estrategia 3 (fallback): si no tenemos cta. del estado, identificar por posición de CUARENTENA
    if (empty($pallets)) {
        $sql3 = "
        SELECT 
            p.id,
            p.codigo AS codigo_pallet,
            COALESCE(MIN(l.codigo), 'N/A') AS lote,
            COALESCE(DATE_FORMAT(MIN(l.fecha_vencimiento), '%Y-%m-%d'), 'N/A') AS fecha_vencimiento,
            COUNT(DISTINCT link.id) AS cantidad_recibida,
            p.posicion_id,
            CASE WHEN COUNT(DISTINCT prod.id) > 1 THEN 'Mixto' ELSE COALESCE(MAX(prod.denominacion), 'Producto') END AS producto_nombre,
            COALESCE({$displayExpr}, 'Sin posición') AS posicion_codigo,
            amb.id AS ambiente_id
            FROM pl_ingreso ing
            JOIN pl_rcv_link link ON link.packinglist_id = ing.packinglist_id
            JOIN wh_pallet p ON p.id = link.pallet_id AND p.deleted_at IS NULL
            LEFT JOIN wh_posicion pos ON pos.id = p.posicion_id
            LEFT JOIN wh_ambiente amb ON amb.id = pos.ambiente_id
            LEFT JOIN wh_lote l ON l.id = link.lote_id
            LEFT JOIN para_productos prod ON prod.id = link.producto_id
            WHERE ing.id = ?
              AND (
                    p.posicion_id IS NULL OR 
                    {$displayExpr} LIKE '%CUARENTENA%' OR 
                    {$displayExpr} LIKE '%CUAR%' OR 
                    {$displayExpr} LIKE '%QUA%'
                  )
            GROUP BY p.id, p.codigo, p.posicion_id, {$displayExpr}, amb.id
            ORDER BY p.codigo
        ";
        $st3 = $pdo->prepare($sql3);
        $st3->execute([$ingresoId]);
        $pallets = pallets_en_cuarentena_dedupe($st3->fetchAll(PDO::FETCH_ASSOC));
        $pallets = array_values(array_filter($pallets, static function(array $row) use ($ambCuaId): bool {
            return pallets_en_cuarentena_should_include($row, $ambCuaId);
        }));
    }

    echo json_encode(['success' => true, 'data'=>$pallets]);

} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode([
        'error'=>'Error listando pallets en cuarentena',
        'details'=>$e->getMessage()
    ]);
}
?>