<?php
declare(strict_types=1);

/**
 * API REAL (sin SP): Confirmar Packing List materializando ítems desde el staging (pl_import_row)
 * Ruta: api/operaciones/pl_confirm_commit.php
 * Método: POST
 *
 * Cambios:
 * - Se resuelve producto por SKU en para_productos (SELECT id WHERE sku = ?).
 * - Si no existe, INSERT en para_productos (sku, denominacion) para evitar "Field 'sku' doesn't have a default value".
 * - Instrumentación de debug detallado.
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';
// Helper local para detectar columnas (cache simple)
function hasColumn(PDO $pdo, string $table, string $col): bool {
    static $cache = [];
    $key = $table.'|'.$col;
    if (array_key_exists($key, $cache)) return $cache[$key];
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?");
    $stmt->execute([$table, $col]);
    return $cache[$key] = ((int)$stmt->fetchColumn() > 0);
}

function respond(array $payload, int $code = 200): void {
    http_response_code($code);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE);
    exit;
}

/** Construye un cliente_ref legible (CLI-XXX) desde razón social y/o RUC */
function build_cliente_ref_from_razon(string $razon, ?string $ruc = null): string {
    $s = $razon;
    if (function_exists('iconv')) {
        $tmp = @iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $s);
        if ($tmp !== false) $s = $tmp;
    }
    $s = strtoupper($s);
    $s = preg_replace('/[^A-Z0-9 ]+/', ' ', $s);
    $tokens = preg_split('/\s+/', trim($s)) ?: [];
    $stop = ['SA','SAS','SRL','SAC','LTDA','LTD','CORP','CORPORATION','COMPANIA','CIA','DE','LA','EL','LOS','LAS','Y'];
    $alias = '';
    foreach ($tokens as $t) { if ($t === '' || in_array($t, $stop, true)) continue; $alias = $t; break; }
    if ($alias === '') {
        $ruc = (string)$ruc;
        $ruc = preg_replace('/[^0-9A-Z]+/','', strtoupper($ruc ?? 'GEN'));
        $alias = $ruc !== '' ? $ruc : 'GEN';
    }
    return 'CLI-' . $alias;
}

$debug = ['step' => 'init'];

try {
    if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
        respond(['ok' => false, 'error' => 'Método no permitido.', 'debug' => $debug], 405);
    }

    $batchId   = (int)($_POST['batch_id'] ?? 0);
    $clienteRef= trim((string)($_POST['cliente_ref'] ?? ''));
    $clienteId = (int)($_POST['cliente_id'] ?? 0);
    $plCodigo  = trim((string)($_POST['pl_codigo'] ?? ''));
    $plFecha   = trim((string)($_POST['pl_fecha']  ?? ''));
    $overwrite = (int)($_POST['overwrite'] ?? 0);
    $forceNew  = (int)($_POST['force_new_product'] ?? 0);

    $debug['params'] = [
        'batch_id' => $batchId,
        'cliente_ref' => $clienteRef,
        'cliente_id' => $clienteId,
        'pl_codigo' => $plCodigo,
        'pl_fecha' => $plFecha,
        'overwrite' => $overwrite,
        'force_new_product' => $forceNew,
    ];

    if ($batchId <= 0) respond(['ok'=>false,'error'=>'batch_id es requerido.','debug'=>$debug], 422);
    if ($plFecha === '' || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $plFecha)) $plFecha = date('Y-m-d');
    if ($plCodigo === '') $plCodigo = 'PL-' . date('Ymd-His');

    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec("SET NAMES utf8mb4");

    // Detectar si pl_producto_alias usa cliente_id (nueva convención)
    $aliasUsesClienteId = hasColumn($pdo, 'pl_producto_alias', 'cliente_id');

    $debug['step'] = 'resolve_cliente_ref';

    if ($clienteRef === '' && $clienteId > 0) {
        $stmt = $pdo->prepare("SELECT razon_social, ruc FROM para_clientes WHERE id = ? LIMIT 1");
        $stmt->execute([$clienteId]);
        if ($cli = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $clienteRef = build_cliente_ref_from_razon((string)$cli['razon_social'], (string)$cli['ruc']);
        }
    }
    if ($clienteRef === '') $clienteRef = 'CLI-GENERICO';
    $debug['cliente_ref'] = $clienteRef;
    // prefer null when cliente_id not provided
    $clienteIdParam = $clienteId > 0 ? $clienteId : null;

    // Validar staging
    $debug['step'] = 'check_staging';
    $q = $pdo->prepare("SELECT COUNT(*) FROM pl_import_row WHERE batch_id=?");
    $q->execute([$batchId]);
    $rowsStaging = (int)$q->fetchColumn();
    $debug['staging_count'] = $rowsStaging;

    if ($rowsStaging === 0) {
        $sampleBatch = $pdo->query("SELECT id, tipo, filename, imported_at, rows_total, rows_ok, rows_error, log FROM pl_import_batch WHERE id = " . (int)$batchId . " LIMIT 1")->fetch(PDO::FETCH_ASSOC) ?: null;
        respond(['ok'=>false,'error'=>"No hay filas en staging para batch_id=$batchId",'debug'=>['step'=>'check_staging','staging_count'=>0,'batch'=>$sampleBatch]], 404);
    }

    $debug['sample_row'] = $pdo->query("SELECT id, rownum, raw FROM pl_import_row WHERE batch_id = ".(int)$batchId." ORDER BY id ASC LIMIT 1")->fetch(PDO::FETCH_ASSOC);

    $pdo->beginTransaction();

    // 1) Crear/actualizar pl_packinglist
    $debug['step'] = 'upsert_pl';
    $plId = (int)$pdo->query("SELECT id FROM pl_packinglist WHERE codigo=" . $pdo->quote($plCodigo) . " LIMIT 1")->fetchColumn();
    if ($plId > 0) {
        if ($overwrite === 1) {
            $st = $pdo->prepare("DELETE FROM pl_packinglist_item WHERE packinglist_id=?");
            $st->execute([$plId]);
            $st = $pdo->prepare("UPDATE pl_packinglist
                                    SET cliente_ref=?, cliente_id=?, fecha=?, estado='IMPORTADO', import_batch_id=?, updated_at=NOW()
                                  WHERE id=?");
            $st->execute([$clienteRef, $clienteIdParam, $plFecha, $batchId, $plId]);
        } else {
            $pdo->rollBack();
            respond(['ok'=>false,'error'=>"El Packing List '$plCodigo' ya existe. Use overwrite=1 para reimportar.",'debug'=>$debug], 409);
        }
    } else {
        $st = $pdo->prepare("INSERT INTO pl_packinglist (codigo, cliente_ref, cliente_id, fecha, estado, import_batch_id)
                             VALUES (?,?,?,?,?,?)");
        $st->execute([$plCodigo, $clienteRef, $clienteIdParam, $plFecha, 'IMPORTADO', $batchId]);
        $plId = (int)$pdo->lastInsertId();
    }
    $debug['pl_id'] = $plId;

    // (Opcional) borrar alias existentes para forzar productos nuevos
    $debug['step'] = 'force_new_alias_cleanup';
    if ($forceNew === 1) {
        $st = $pdo->prepare("SELECT DISTINCT JSON_UNQUOTE(JSON_EXTRACT(raw,'$.sku_cliente')) AS sku
                               FROM pl_import_row
                              WHERE batch_id=? AND JSON_UNQUOTE(JSON_EXTRACT(raw,'$.sku_cliente')) IS NOT NULL
                                                AND JSON_UNQUOTE(JSON_EXTRACT(raw,'$.sku_cliente')) <> ''");
        $st->execute([$batchId]);
        $skus = $st->fetchAll(PDO::FETCH_COLUMN);
        $debug['skus_forced_new'] = $skus;

        if ($skus) {
            $in = implode(',', array_fill(0, count($skus), '?'));
            if ($aliasUsesClienteId && $clienteIdParam !== null) {
                $params = array_merge([$clienteIdParam], $skus);
                $del = $pdo->prepare("DELETE FROM pl_producto_alias WHERE cliente_id=? AND sku_cliente IN ($in)");
            } else {
                $params = array_merge([$clienteRef], $skus);
                $del = $pdo->prepare("DELETE FROM pl_producto_alias WHERE cliente_ref=? AND sku_cliente IN ($in)");
            }
            $del->execute($params);
            $debug['forced_new_deleted_aliases'] = count($skus);
        }
    }

    // 2) Consolidar staging: sku + lote
    $debug['step'] = 'aggregate_staging';
    $stAgg = $pdo->prepare("
        SELECT
          JSON_UNQUOTE(JSON_EXTRACT(raw,'$.sku_cliente')) AS sku,
          NULLIF(JSON_UNQUOTE(JSON_EXTRACT(raw,'$.lote')), '') AS lote,
          COALESCE(SUM(CAST(JSON_UNQUOTE(JSON_EXTRACT(raw,'$.expected_uv')) AS UNSIGNED)),0) AS sum_uv,
          COALESCE(SUM(CAST(JSON_UNQUOTE(JSON_EXTRACT(raw,'$.expected_uc')) AS UNSIGNED)),0) AS sum_uc,
          MAX(NULLIF(JSON_UNQUOTE(JSON_EXTRACT(raw,'$.fecha_produccion')),''))  AS fprod,
          MAX(NULLIF(JSON_UNQUOTE(JSON_EXTRACT(raw,'$.fecha_vencimiento')),'')) AS fvenc,
          MAX(NULLIF(JSON_UNQUOTE(JSON_EXTRACT(raw,'$.descripcion')),''))       AS descr
        FROM pl_import_row
        WHERE batch_id = :bid
        GROUP BY sku, lote
        HAVING sku IS NOT NULL AND sku <> ''
    ");
    $stAgg->execute([':bid' => $batchId]);
    $groups = $stAgg->fetchAll(PDO::FETCH_ASSOC) ?: [];
    $debug['groups_count'] = count($groups);

    if (count($groups) === 0) {
        $pdo->rollBack();
        respond([
            'ok' => false,
            'error' => 'No se formaron grupos (sku+lote) desde el staging. Verifica claves: sku_cliente, expected_uv, expected_uc.',
            'debug' => $debug
        ], 422);
    }

    // Preparar statements usados en loop
    $debug['step'] = 'prepare_statements';
    if ($aliasUsesClienteId) {
        $stFindAlias     = $pdo->prepare("SELECT producto_id FROM pl_producto_alias WHERE cliente_id=? AND sku_cliente=? LIMIT 1");
        // Detectar si la tabla tiene columna producto_desc
        $aliasHasProductoDesc = hasColumn($pdo, 'pl_producto_alias', 'producto_desc');
        if ($aliasHasProductoDesc) {
            $stUpsertAlias = $pdo->prepare("INSERT INTO pl_producto_alias (cliente_id, sku_cliente, producto_id, producto_desc) VALUES (?,?,?,?) ON DUPLICATE KEY UPDATE producto_id=VALUES(producto_id), producto_desc=VALUES(producto_desc)");
        } else {
            $stUpsertAlias = $pdo->prepare("INSERT INTO pl_producto_alias (cliente_id, sku_cliente, producto_id) VALUES (?,?,?) ON DUPLICATE KEY UPDATE producto_id=VALUES(producto_id)");
        }
    } else {
        $stFindAlias     = $pdo->prepare("SELECT producto_id FROM pl_producto_alias WHERE cliente_ref=? AND sku_cliente=? LIMIT 1");
        $aliasHasProductoDesc = hasColumn($pdo, 'pl_producto_alias', 'producto_desc');
        if ($aliasHasProductoDesc) {
            $stUpsertAlias = $pdo->prepare("INSERT INTO pl_producto_alias (cliente_ref, sku_cliente, producto_id, producto_desc) VALUES (?,?,?,?) ON DUPLICATE KEY UPDATE producto_id=VALUES(producto_id), producto_desc=VALUES(producto_desc)");
        } else {
            $stUpsertAlias = $pdo->prepare("INSERT INTO pl_producto_alias (cliente_ref, sku_cliente, producto_id) VALUES (?,?,?) ON DUPLICATE KEY UPDATE producto_id=VALUES(producto_id)");
        }
    }
    $stFindProdBySku  = $pdo->prepare("SELECT id FROM para_productos WHERE sku=? LIMIT 1");
    $stNewProd        = $pdo->prepare("INSERT INTO para_productos (sku, denominacion) VALUES (?, ?)");
    $stUpsertLote     = $pdo->prepare("
        INSERT INTO wh_lote (producto_id, codigo, fecha_produccion, fecha_vencimiento)
        VALUES (?,?,?,?)
        ON DUPLICATE KEY UPDATE
            fecha_produccion  = COALESCE(wh_lote.fecha_produccion, VALUES(fecha_produccion)),
            fecha_vencimiento = COALESCE(wh_lote.fecha_vencimiento, VALUES(fecha_vencimiento))
    ");
    $stFindItem = $pdo->prepare("
        SELECT id FROM pl_packinglist_item
         WHERE packinglist_id=? AND sku_cliente=? AND (producto_id <=> ?) AND (lote_codigo <=> ?)
         LIMIT 1
    ");
    $stInsItem  = $pdo->prepare("
        INSERT INTO pl_packinglist_item (
            packinglist_id, sku_cliente, producto_id, descripcion,
            lote_codigo, fecha_produccion, fecha_vencimiento,
            expected_uv, expected_uc
        ) VALUES (?,?,?,?,?,?,?,?,?)
    ");
    $stUpdItem  = $pdo->prepare("
        UPDATE pl_packinglist_item
           SET descripcion       = COALESCE(NULLIF(descripcion,''), ?),
               fecha_produccion  = COALESCE(fecha_produccion, ?),
               fecha_vencimiento = COALESCE(fecha_vencimiento, ?),
               expected_uv       = COALESCE(expected_uv,0) + ?,
               expected_uc       = COALESCE(expected_uc,0) + ?
         WHERE id = ?
    ");

    $debug['step'] = 'loop_groups';
    $loopCount = 0;
    $debug['product_resolution'] = []; // para ver qué pasó por cada SKU

    foreach ($groups as $g) {
        $loopCount++;
        $sku   = (string)$g['sku'];
        $lote  = $g['lote'] !== null ? (string)$g['lote'] : null;
        $sumUV = (int)$g['sum_uv'];
        $sumUC = (int)$g['sum_uc'];
        $desc1 = (string)($g['descr'] ?? '');
        $fprod = ($g['fprod'] ?? '') !== '' ? (string)$g['fprod'] : null;
        $fvenc = ($g['fvenc'] ?? '') !== '' ? (string)$g['fvenc'] : null;

        $prodId = 0;
        $how    = '';

        // 2.1 Resolver por alias (cliente_ref o cliente_id + sku_cliente)
        if ($aliasUsesClienteId && $clienteIdParam !== null) {
            $stFindAlias->execute([$clienteIdParam, $sku]);
            $prodId = (int)$stFindAlias->fetchColumn();
            if ($prodId > 0) {
                $how = 'alias';
                if ($desc1 !== '') {
                    if ($aliasHasProductoDesc) $stUpsertAlias->execute([$clienteIdParam, $sku, $prodId, $desc1]);
                    else $stUpsertAlias->execute([$clienteIdParam, $sku, $prodId]);
                }
            } else {
                // continuar a resolver por sku
                $prodId = 0;
            }
        } else {
            $stFindAlias->execute([$clienteRef, $sku]);
            $prodId = (int)$stFindAlias->fetchColumn();
            if ($prodId > 0) {
                $how = 'alias';
                if ($desc1 !== '') {
                    if ($aliasHasProductoDesc) $stUpsertAlias->execute([$clienteRef, $sku, $prodId, $desc1]);
                    else $stUpsertAlias->execute([$clienteRef, $sku, $prodId]);
                }
            } else {
                // continuar a resolver por sku
                $prodId = 0;
            }
        }
        if ($prodId <= 0) {
            // 2.1.b Resolver por SKU directo en para_productos (si ya existe)
            $stFindProdBySku->execute([$sku]);
            $prodId = (int)$stFindProdBySku->fetchColumn();
            if ($prodId > 0) {
                $how = 'producto_por_sku';
                // asegurar alias
                if ($aliasUsesClienteId && $clienteIdParam !== null) {
                    if ($aliasHasProductoDesc) $stUpsertAlias->execute([$clienteIdParam, $sku, $prodId, $desc1]);
                    else $stUpsertAlias->execute([$clienteIdParam, $sku, $prodId]);
                } else {
                    if ($aliasHasProductoDesc) $stUpsertAlias->execute([$clienteRef, $sku, $prodId, $desc1]);
                    else $stUpsertAlias->execute([$clienteRef, $sku, $prodId]);
                }
            } else {
                // 2.1.c Crear producto nuevo (sku, denominacion)
                $denom = $desc1 !== '' ? $desc1 : ('SKU ' . $sku);
                $stNewProd->execute([$sku, $denom]);  // <-- clave: insertar SKU
                $prodId = (int)$pdo->lastInsertId();
                $how = 'producto_creado';
                // alias
                if ($aliasUsesClienteId && $clienteIdParam !== null) {
                    if ($aliasHasProductoDesc) $stUpsertAlias->execute([$clienteIdParam, $sku, $prodId, $desc1]);
                    else $stUpsertAlias->execute([$clienteIdParam, $sku, $prodId]);
                } else {
                    if ($aliasHasProductoDesc) $stUpsertAlias->execute([$clienteRef, $sku, $prodId, $desc1]);
                    else $stUpsertAlias->execute([$clienteRef, $sku, $prodId]);
                }
            }
        }

        $debug['product_resolution'][] = [
            'sku' => $sku, 'prod_id' => $prodId, 'how' => $how,
            'lote' => $lote, 'sumUV' => $sumUV, 'sumUC' => $sumUC
        ];

        // 2.2 Upsert lote si vino
        if (($lote ?? '') !== '') {
            $stUpsertLote->execute([$prodId, $lote, $fprod, $fvenc]);
        }

        // 2.3 Upsert ítem del PL
        $stFindItem->execute([$plId, $sku, $prodId, $lote]);
        $itemId = (int)$stFindItem->fetchColumn();
        if ($itemId > 0) {
            $stUpdItem->execute([$desc1 !== '' ? $desc1 : null, $fprod, $fvenc, $sumUV, $sumUC, $itemId]);
        } else {
            $stInsItem->execute([$plId, $sku, $prodId, $desc1 !== '' ? $desc1 : null, $lote, $fprod, $fvenc, $sumUV, $sumUC]);
            $itemId = (int)$pdo->lastInsertId();
        }

        // 2.4 Vincular filas del staging y marcar OK
        $stLink = $pdo->prepare("
            UPDATE pl_import_row
               SET packinglist_id = :plid,
                   item_id        = :iid,
                   status         = 'OK',
                   error_msg      = NULL
             WHERE batch_id = :bid
               AND JSON_UNQUOTE(JSON_EXTRACT(raw,'$.sku_cliente')) = :sku
               AND (NULLIF(JSON_UNQUOTE(JSON_EXTRACT(raw,'$.lote')), '') <=> :lote)
        ");
        $stLink->execute([
            ':plid' => $plId,
            ':iid'  => $itemId,
            ':bid'  => $batchId,
            ':sku'  => $sku,
            ':lote' => $lote,
        ]);
    }

    $debug['loop_processed_groups'] = $loopCount;

    // 3) Totales y hints
    $debug['step'] = 'summary';
    $stAgg2 = $pdo->prepare("
        SELECT
          COUNT(*)                                                AS items_importados,
          COUNT(DISTINCT i.sku_cliente)                           AS skus_importados,
          COUNT(DISTINCT NULLIF(i.lote_codigo,''))                AS lotes_importados,
          COALESCE(SUM(i.expected_uv), 0)                         AS cajas,
          COALESCE(SUM(i.expected_uc), 0)                         AS uc_totales
        FROM pl_packinglist_item i
        WHERE i.packinglist_id = :plid
    ");
    $stAgg2->execute([':plid' => $plId]);
    $agg = $stAgg2->fetch(PDO::FETCH_ASSOC) ?: [
        'items_importados'=>0,'skus_importados'=>0,'lotes_importados'=>0,'cajas'=>0,'uc_totales'=>0
    ];

    // sueltas estimadas
    $sueltas = 0;
    $stRows = $pdo->prepare("SELECT expected_uv AS uv, expected_uc AS uc FROM pl_packinglist_item WHERE packinglist_id=?");
    $stRows->execute([$plId]);
    while ($r = $stRows->fetch(PDO::FETCH_ASSOC)) {
        $uv = (int)($r['uv'] ?? 0);
        $uc = (int)($r['uc'] ?? 0);
        if ($uv > 0 && $uc > 0) {
            $uxc = intdiv($uc, max(1,$uv)); // safe guard
            $sueltas += max(0, $uc - ($uv * $uxc));
        }
    }

    // Pallets/posiciones desde staging
    $debug['step'] = 'hints';
    $stHints = $pdo->prepare("
        SELECT
          COUNT(DISTINCT NULLIF(JSON_UNQUOTE(JSON_EXTRACT(raw,'$.pallet_hint')),''))   AS pallets_count,
          COUNT(DISTINCT NULLIF(JSON_UNQUOTE(JSON_EXTRACT(raw,'$.position_hint')),'')) AS positions_count
        FROM pl_import_row
        WHERE batch_id = :bid
    ");
    $stHints->execute([':bid'=>$batchId]);
    $h = $stHints->fetch(PDO::FETCH_ASSOC) ?: [];
    $palletsCount   = (int)($h['pallets_count']   ?? 0);
    $positionsCount = (int)($h['positions_count'] ?? 0);

    $stTopPal = $pdo->prepare("
        SELECT JSON_UNQUOTE(JSON_EXTRACT(raw,'$.pallet_hint')) AS hint, COUNT(*) AS c
        FROM pl_import_row
        WHERE batch_id = :bid
          AND JSON_UNQUOTE(JSON_EXTRACT(raw,'$.pallet_hint')) IS NOT NULL
          AND JSON_UNQUOTE(JSON_EXTRACT(raw,'$.pallet_hint')) <> ''
        GROUP BY hint
        ORDER BY c DESC
        LIMIT 10
    ");
    $stTopPal->execute([':bid'=>$batchId]);
    $palletsTop = $stTopPal->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $stTopPos = $pdo->prepare("
        SELECT JSON_UNQUOTE(JSON_EXTRACT(raw,'$.position_hint')) AS hint, COUNT(*) AS c
        FROM pl_import_row
        WHERE batch_id = :bid
          AND JSON_UNQUOTE(JSON_EXTRACT(raw,'$.position_hint')) IS NOT NULL
          AND JSON_UNQUOTE(JSON_EXTRACT(raw,'$.position_hint')) <> ''
        GROUP BY hint
        ORDER BY c DESC
        LIMIT 10
    ");
    $stTopPos->execute([':bid'=>$batchId]);
    $positionsTop = $stTopPos->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $pdo->commit();
    $debug['step'] = 'done';

    respond([
        'ok'                 => true,
        'packinglist_id'     => $plId,
        'packinglist_codigo' => $plCodigo,
    'cliente_ref'        => $clienteRef,
    'cliente_id'         => $clienteIdParam,
        'fecha'              => $plFecha,
        'overwrite'          => (int)$overwrite,

        'summary' => [
            'skus_importados'  => (int)$agg['skus_importados'],
            'skus_nuevos'      => null,
            'items_importados' => (int)$agg['items_importados'],
            'lotes_importados' => (int)$agg['lotes_importados'],
            'lotes_nuevos'     => null,
            'cajas'            => (int)$agg['cajas'],
            'uc_totales'       => (int)$agg['uc_totales'],
            'uc_sueltas_est'   => (int)$sueltas,
        ],

        'hints' => [
            'pallets_count'   => $palletsCount,
            'positions_count' => $positionsCount,
            'pallets_top'     => $palletsTop,
            'positions_top'   => $positionsTop,
        ],

        'message' => 'Packing List confirmado materializando desde staging (sin SP).',
        'debug'   => $debug
    ]);

} catch (Throwable $e) {
    if (!empty($pdo) && $pdo->inTransaction()) $pdo->rollBack();
    $err = ['ok'=>false, 'error'=>'No se pudo confirmar el PL'];
    $err['debug'] = $debug + [
        'exception' => get_class($e),
        'message'   => $e->getMessage(),
    ];
    if ($e instanceof PDOException) {
        $err['debug']['sqlstate'] = $e->getCode();
    }
    respond($err, 500);
}
