<?php
// api/operaciones/pl_diag_confirm.php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function jexit($payload, int $code = 200): void {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}

try {
  $pdo = get_pdo();

  // Puedes pasar: pl_id, pl_codigo o batch_id (en ese orden de prioridad)
  $plId      = (int)($_GET['pl_id'] ?? $_POST['pl_id'] ?? 0);
  $plCodigo  = trim((string)($_GET['pl_codigo'] ?? $_POST['pl_codigo'] ?? ''));
  $batchId   = (int)($_GET['batch_id'] ?? $_POST['batch_id'] ?? 0);

  // Resolver por pl_codigo si no vino pl_id
  if ($plId <= 0 && $plCodigo !== '') {
    $st = $pdo->prepare("SELECT id FROM pl_packinglist WHERE codigo = ? LIMIT 1");
    $st->execute([$plCodigo]);
    $plId = (int)$st->fetchColumn();
  }

  // Si no vino nada, tomamos el último PL importado (para que siempre haya algo que mostrar)
  if ($plId <= 0 && $batchId <= 0) {
    $st = $pdo->query("SELECT id, import_batch_id FROM pl_packinglist ORDER BY id DESC LIMIT 1");
    if ($row = $st->fetch(PDO::FETCH_ASSOC)) {
      $plId = (int)$row['id'];
      $batchId = (int)$row['import_batch_id'];
    }
  }

  // Resolver batch a partir del PL si hace falta
  if ($plId > 0 && $batchId <= 0) {
    $st = $pdo->prepare("SELECT import_batch_id FROM pl_packinglist WHERE id = ? LIMIT 1");
    $st->execute([$plId]);
    $batchId = (int)$st->fetchColumn();
  }

  // Resolver PL a partir del batch si hace falta
  if ($batchId > 0 && $plId <= 0) {
    $st = $pdo->prepare("SELECT id FROM pl_packinglist WHERE import_batch_id = ? ORDER BY id DESC LIMIT 1");
    $st->execute([$batchId]);
    $plId = (int)$st->fetchColumn();
  }

  // Datos del PL y del Batch
  $pl = null; $batch = null;
  if ($plId > 0) {
    $st = $pdo->prepare("SELECT id, codigo, cliente_ref, fecha, import_batch_id FROM pl_packinglist WHERE id = ? LIMIT 1");
    $st->execute([$plId]);
    $pl = $st->fetch(PDO::FETCH_ASSOC) ?: null;
    if ($pl && $batchId <= 0) $batchId = (int)$pl['import_batch_id'];
  }
  if ($batchId > 0) {
    $st = $pdo->prepare("SELECT id, tipo, filename, imported_at, rows_total, rows_ok, rows_error, log FROM pl_import_batch WHERE id = ? LIMIT 1");
    $st->execute([$batchId]);
    $batch = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  }

  // Conteos en pl_packinglist_item
  $pli = ['items'=>0,'skus'=>0,'lotes'=>0,'sum_expected_uv'=>0,'sum_expected_uc'=>0];
  if ($plId > 0) {
    $st = $pdo->prepare("
      SELECT
        COUNT(*)                                           AS items,
        COUNT(DISTINCT sku_cliente)                        AS skus,
        COUNT(DISTINCT NULLIF(lote_codigo,''))             AS lotes,
        COALESCE(SUM(expected_uv),0)                       AS sum_expected_uv,
        COALESCE(SUM(expected_uc),0)                       AS sum_expected_uc
      FROM pl_packinglist_item
      WHERE packinglist_id = ?
    ");
    $st->execute([$plId]);
    $tmp = $st->fetch(PDO::FETCH_ASSOC) ?: [];
    $pli = array_map('intval', $tmp);
  }

  // Staging: estados y primeros errores
  $statusAgg = []; $errors = [];
  if ($batchId > 0) {
    $st = $pdo->prepare("SELECT status, COUNT(*) c FROM pl_import_row WHERE batch_id = ? GROUP BY status");
    $st->execute([$batchId]);
    $statusAgg = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $st = $pdo->prepare("
      SELECT rownum,
             JSON_UNQUOTE(JSON_EXTRACT(raw,'$.sku_cliente')) AS sku,
             JSON_UNQUOTE(JSON_EXTRACT(raw,'$.lote'))        AS lote,
             error_msg
      FROM pl_import_row
      WHERE batch_id = ? AND status = 'ERROR'
      ORDER BY rownum
      LIMIT 50
    ");
    $st->execute([$batchId]);
    $errors = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  }

  // Peek rápido: últimos PL y Batches (te ayuda a elegir ids válidos si llamaste sin params)
  $recent = [
    'pl' => [],
    'batch' => []
  ];
  $q1 = $pdo->query("SELECT id, codigo, cliente_ref, fecha, import_batch_id FROM pl_packinglist ORDER BY id DESC LIMIT 10");
  $recent['pl'] = $q1->fetchAll(PDO::FETCH_ASSOC) ?: [];
  $q2 = $pdo->query("SELECT id, tipo, filename, imported_at, rows_total, rows_ok, rows_error FROM pl_import_batch ORDER BY id DESC LIMIT 10");
  $recent['batch'] = $q2->fetchAll(PDO::FETCH_ASSOC) ?: [];

  // Esquema requerido (por si falta alguna tabla/col)
  $schemaIssues = [];
  $reqCols = [
    'pl_packinglist_item' => ['packinglist_id','sku_cliente','producto_id','descripcion','lote_codigo','fecha_produccion','fecha_vencimiento','expected_uv','expected_uc'],
    'para_productos'      => ['id','denominacion'],
    'pl_producto_alias'   => ['id','cliente_ref','sku_cliente','producto_id','producto_desc'],
    'wh_lote'             => ['id','producto_id','codigo','fecha_produccion','fecha_vencimiento'],
  ];
  foreach ($reqCols as $tbl => $cols) {
    try {
      $have = [];
      $q = $pdo->query("SHOW COLUMNS FROM `$tbl`");
      foreach ($q->fetchAll(PDO::FETCH_ASSOC) as $row) $have[] = $row['Field'];
      foreach ($cols as $c) {
        if (!in_array($c, $have, true)) $schemaIssues[] = "$tbl falta columna: $c";
      }
    } catch (\Throwable $e) {
      $schemaIssues[] = "No existe la tabla: $tbl";
    }
  }

  jexit([
    'ok' => true,
    'params' => ['pl_id'=>$plId, 'pl_codigo'=>$plCodigo, 'batch_id'=>$batchId],
    'pl' => $pl,
    'batch' => $batch,
    'packinglist_item' => $pli,
    'staging' => [
      'status' => $statusAgg,  // [{status, c}]
      'errors' => $errors,     // filas con error
    ],
    'recent' => $recent,       // últimos ids por si llamaste sin params
    'schema_issues' => $schemaIssues,
    'hint' => 'Puedes llamar con ?pl_id=XX o ?pl_codigo=PL-YYYY... o ?batch_id=XX'
  ]);
} catch (\Throwable $e) {
  jexit(['ok' => false, 'error' => 'diag failed: '.$e->getMessage()], 500);
}
