<?php
declare(strict_types=1);

/**
 * API: Listar ítems de un Packing List (consolidado)
 * Ruta: api/operaciones/pl_items.php
 * Método: GET
 *
 * Parámetros (uno de los siguientes):
 * - pl_id       (int)    → id de pl_packinglist
 * - pl_codigo   (string) → código de pl_packinglist (PL-YYYYMMDD-...)
 *
 * Respuesta:
 * {
 *   ok: true,
 *   packinglist: { id, codigo, cliente_ref, fecha },
 *   columns: [{data,title,type?}, ...],
 *   data: [ { ... }, ... ]
 * }
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function json_out(array $payload, int $code = 200): void {
    http_response_code($code);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE);
    exit;
}

try {
    if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'GET') {
        json_out(['ok' => false, 'error' => 'Método no permitido.'], 405);
    }

    $plId     = isset($_GET['pl_id']) ? (int)$_GET['pl_id'] : 0;
    $plCodigo = isset($_GET['pl_codigo']) ? trim((string)$_GET['pl_codigo']) : '';

    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec("SET NAMES utf8mb4");

    // Resolver PL por id o por código
    if ($plId <= 0 && $plCodigo !== '') {
        $stmt = $pdo->prepare("SELECT id FROM pl_packinglist WHERE codigo = ? LIMIT 1");
        $stmt->execute([$plCodigo]);
        $plId = (int)$stmt->fetchColumn();
    }

    if ($plId <= 0) {
        json_out(['ok'=>false, 'error'=>'Debe indicar pl_id o pl_codigo.'], 422);
    }

    // Cabecera
    $stH = $pdo->prepare("SELECT id, codigo, cliente_ref, fecha FROM pl_packinglist WHERE id = ? LIMIT 1");
    $stH->execute([$plId]);
    $hdr = $stH->fetch(PDO::FETCH_ASSOC);
    if (!$hdr) {
        json_out(['ok'=>false, 'error'=>'Packing List no encontrado.'], 404);
    }

    // Columnas predefinidas (compatibles con DataTables)
    $columns = [
        ['data'=>'id',               'title'=>'#',               'type'=>'number'],
        ['data'=>'sku_cliente',      'title'=>'SKU'],
        ['data'=>'producto_id',      'title'=>'ProdID',          'type'=>'number'],
        ['data'=>'descripcion',      'title'=>'Denominación'],
        ['data'=>'lote_codigo',      'title'=>'Lote'],
        ['data'=>'fecha_produccion', 'title'=>'F.Prod'],
        ['data'=>'fecha_vencimiento','title'=>'F.Vto'],
        ['data'=>'expected_uv',      'title'=>'UV Cajas',        'type'=>'number'],
        ['data'=>'expected_uc',      'title'=>'UC totales',      'type'=>'number'],
        ['data'=>'received_uv',      'title'=>'Rec. UV',         'type'=>'number'],
        ['data'=>'received_uc',      'title'=>'Rec. UC',         'type'=>'number'],
        ['data'=>'diff_uv',          'title'=>'Δ UV',            'type'=>'number'],
        ['data'=>'diff_uc',          'title'=>'Δ UC',            'type'=>'number'],
        ['data'=>'checked_item',     'title'=>'Check',           'type'=>'number'],
        ['data'=>'obs',              'title'=>'Obs.'],
    ];

    // Datos (no hacemos JOINs costosos aquí; mantenemos lo ya consolidado)
    $st = $pdo->prepare("
        SELECT
          i.id,
          i.sku_cliente,
          i.producto_id,
          i.descripcion,
          i.lote_codigo,
          DATE_FORMAT(i.fecha_produccion,  '%Y-%m-%d') AS fecha_produccion,
          DATE_FORMAT(i.fecha_vencimiento, '%Y-%m-%d') AS fecha_vencimiento,
          i.expected_uv,
          i.expected_uc,
          i.received_uv,
          i.received_uc,
          i.diff_uv,
          i.diff_uc,
          i.checked_item,
          i.obs
        FROM pl_packinglist_item i
        WHERE i.packinglist_id = ?
        ORDER BY i.id ASC
    ");
    $st->execute([$plId]);
    $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

    json_out([
        'ok' => true,
        'packinglist' => [
            'id'          => (int)$hdr['id'],
            'codigo'      => (string)$hdr['codigo'],
            'cliente_ref' => (string)$hdr['cliente_ref'],
            'fecha'       => (string)$hdr['fecha'],
        ],
        'columns' => $columns,
        'data'    => $rows,
    ]);

} catch (Throwable $e) {
    json_out([
        'ok' => false,
        'error' => 'No se pudo cargar el detalle del Packing List.',
        'message' => $e->getMessage(),
    ], 500);
}
