<?php
// api/operaciones/pl_preview.php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

$ROOT = dirname(__DIR__, 2); // api -> operaciones -> raíz
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function jexit(array $payload, int $httpCode = 200): void {
  http_response_code($httpCode);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}

try {
  $pdo = get_pdo();

  $batchId = (int)($_GET['batch_id'] ?? 0);
  if ($batchId <= 0) {
    jexit(['ok' => false, 'error' => 'batch_id requerido'], 400);
  }

  // Resumen del batch (singular)
  $stmtB = $pdo->prepare("
    SELECT
      id,
      filename,
      rows_total,
      rows_ok,
      rows_error,
      imported_at
    FROM pl_import_batch
    WHERE id = :id
    LIMIT 1
  ");
  $stmtB->execute([':id' => $batchId]);
  $batch = $stmtB->fetch(PDO::FETCH_ASSOC);
  if (!$batch) {
    jexit(['ok' => false, 'error' => 'Batch no encontrado'], 404);
  }

  // Previsualización desde pl_import_row (JSON)
  // Notas:
  // - expected_uv => Cajas
  // - expected_uc => UC Total
  // - Derivamos uc_por_caja y uc_sueltas para la UI
  $stmt = $pdo->prepare("
    SELECT
      pir.id,
      pir.rownum                         AS fila_nro,
      JSON_UNQUOTE(JSON_EXTRACT(pir.raw,'$.sku_cliente'))        AS sku,
      JSON_UNQUOTE(JSON_EXTRACT(pir.raw,'$.descripcion'))        AS denominacion,
      JSON_UNQUOTE(JSON_EXTRACT(pir.raw,'$.lote'))               AS lote,
      JSON_UNQUOTE(JSON_EXTRACT(pir.raw,'$.fecha_produccion'))   AS fecha_produccion,
      JSON_UNQUOTE(JSON_EXTRACT(pir.raw,'$.fecha_vencimiento'))  AS fecha_vencimiento,
      CAST(JSON_UNQUOTE(JSON_EXTRACT(pir.raw,'$.expected_uv')) AS UNSIGNED) AS uv_cajas,
      CAST(JSON_UNQUOTE(JSON_EXTRACT(pir.raw,'$.expected_uc')) AS UNSIGNED) AS uc_total
    FROM pl_import_row pir
    WHERE pir.batch_id = :bid
    ORDER BY pir.rownum ASC, pir.id ASC
  ");
  $stmt->execute([':bid' => $batchId]);
  $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

  // Derivar uc_por_caja y uc_sueltas en PHP para no complicar el SQL
  foreach ($rows as &$r) {
    $uv  = isset($r['uv_cajas']) ? (int)$r['uv_cajas'] : 0;
    $uct = isset($r['uc_total']) ? (int)$r['uc_total'] : 0;
    if ($uv > 0 && $uct > 0) {
      $uXC = intdiv($uct, $uv);           // unidades por caja (aprox)
      $su  = $uct - ($uv * $uXC);         // sueltas
      $r['uc_por_caja'] = $uXC ?: null;
      $r['uc_sueltas']  = $su ?: null;
    } else {
      $r['uc_por_caja'] = null;
      $r['uc_sueltas']  = null;
    }
    // Normalizar fechas vacías a null
    $r['fecha_produccion']  = ($r['fecha_produccion']  ?? '') !== '' ? $r['fecha_produccion']  : null;
    $r['fecha_vencimiento'] = ($r['fecha_vencimiento'] ?? '') !== '' ? $r['fecha_vencimiento'] : null;
  }
  unset($r);

  // Definición de columnas para DataTables (compatibles con tu JS y sumatorias)
  $columns = [
    ['data' => 'fila_nro',          'title' => '#'],
    ['data' => 'sku',               'title' => 'SKU'],
    ['data' => 'denominacion',      'title' => 'Denominación'],
    ['data' => 'lote',              'title' => 'Lote'],
    ['data' => 'fecha_produccion',  'title' => 'F.Prod.'],
    ['data' => 'fecha_vencimiento', 'title' => 'F.Venc.'],
    ['data' => 'uv_cajas',          'title' => 'Cajas',      'type' => 'number'],
    ['data' => 'uc_por_caja',       'title' => 'UC x Caja',  'type' => 'number'],
    ['data' => 'uc_sueltas',        'title' => 'UC Sueltas', 'type' => 'number'],
    ['data' => 'uc_total',          'title' => 'UC Total',   'type' => 'number'],
  ];

  jexit([
    'ok'          => true,
    'batch_id'    => $batchId,
    'filename'    => $batch['filename'],
    'rows_total'  => $batch['rows_total'],
    'rows_ok'     => $batch['rows_ok'],
    'rows_error'  => $batch['rows_error'],
    'imported_at' => $batch['imported_at'],
    'columns'     => $columns,
    'data'        => $rows,
  ]);
} catch (Throwable $e) {
  jexit(['ok' => false, 'error' => 'Error listando preview'], 500);
}
