<?php
// api/operaciones/pl_upload.php
declare(strict_types=1);

\header('Content-Type: application/json; charset=utf-8');

// En prod: 0; en dev puedes subir a 1 evitando E_DEPRECATED
\error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
\ini_set('display_errors', '0');

$BASE = \dirname(__DIR__, 2); // api -> operaciones -> raíz del proyecto

// Importante: NO incluir ViewHelpers.php aquí para evitar choque con syslog()
require_once $BASE . '/config/config.php'; // env()
require_once $BASE . '/config/db.php';     // get_pdo()

/* -------------------------
 * Helpers
 * ------------------------- */
function jexit(array $payload, int $httpCode = 200): void {
  \http_response_code($httpCode);
  echo \json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}
function ensure_dir(string $dir): void {
  if (!\is_dir($dir)) { @\mkdir($dir, 0775, true); }
}
function random_suffix(int $len = 6): string {
  $chars = 'abcdefghjkmnpqrstuvwxyzABCDEFGHJKMNPQRSTUVWXYZ23456789';
  $out = '';
  for ($i = 0; $i < $len; $i++) { $out .= $chars[\random_int(0, \strlen($chars) - 1)]; }
  return $out;
}
function to_date_ymd($raw): ?string {
  if ($raw === null || $raw === '') return null;
  if (\is_numeric($raw)) {
    try {
      $dt = \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject((float)$raw);
      return $dt ? $dt->format('Y-m-d') : null;
    } catch (\Throwable $e) {
      return null;
    }
  }
  $ts = \strtotime((string)$raw);
  return $ts ? \date('Y-m-d', $ts) : null;
}

/* -------------------------
 * Validaciones
 * ------------------------- */
if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
  jexit(['ok' => false, 'error' => 'Método no permitido'], 405);
}

$clienteId   = isset($_POST['cliente_id']) ? \trim((string)$_POST['cliente_id']) : '';
$observacion = isset($_POST['observacion']) ? \trim((string)$_POST['observacion']) : '';
$sheetPreferred = isset($_POST['sheet_name']) ? \trim((string)$_POST['sheet_name']) : '';

if ($clienteId === '') {
  jexit(['ok' => false, 'error' => 'Falta seleccionar el cliente'], 400);
}
if (!isset($_FILES['pl_file']) || !\is_array($_FILES['pl_file']) || (($_FILES['pl_file']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK)) {
  jexit(['ok' => false, 'error' => 'No se recibió el archivo Excel'], 400);
}

$up = $_FILES['pl_file'];
$origName = $up['name'] ?? 'archivo.xlsx';
$ext = \strtolower(\pathinfo($origName, PATHINFO_EXTENSION));
if (!\in_array($ext, ['xlsx', 'xls'], true)) {
  jexit(['ok' => false, 'error' => 'Formato no permitido. Use XLSX/XLS según el modelo'], 400);
}

/* -------------------------
 * Guardado del archivo (auditoría)
 * ------------------------- */
$baseUrl = \rtrim((string)\env('BASE_URL', '/'), '/');
$publicStorageRel = '/assets/storage/uploads/pl/' . \date('Y') . '/' . \date('m');
$publicStorageAbs = \rtrim($BASE . '/public' . $publicStorageRel, '/');
ensure_dir($publicStorageAbs);

$fname   = 'pl_' . \date('Ymd_His') . '_' . random_suffix() . '.' . $ext;
$destAbs = $publicStorageAbs . '/' . $fname;
$destRel = $publicStorageRel . '/' . $fname;

if (!@\move_uploaded_file($up['tmp_name'], $destAbs)) {
  jexit(['ok' => false, 'error' => 'No se pudo guardar el archivo subido'], 500);
}

/* -------------------------
 * DB: pl_import_batch + pl_import_row(raw JSON)
 * ------------------------- */
try {
  $pdo = get_pdo();
  $pdo->beginTransaction();

  // 1) Crear batch (tabla CANÓNICA singular)
  $sqlInsBatch = "INSERT INTO pl_import_batch
    (tipo, filename, sheet_name, imported_at, user_id, rows_total, rows_ok, rows_error, log)
    VALUES
    ('PACKINGLIST', :filename, NULL, NOW(), NULL, NULL, NULL, NULL,
     CONCAT('cliente_id=', :cliente_id, '; path=', :path, '; obs=', :obs))";
  $stb = $pdo->prepare($sqlInsBatch);
  $stb->execute([
    ':filename'   => $origName,
    ':cliente_id' => $clienteId,
    ':path'       => $destRel,
    ':obs'        => $observacion,
  ]);
  $batchId = (int)$pdo->lastInsertId();

  // 2) Leer Excel → construir RAW JSON por fila → pl_import_row
  $vendor = $BASE . '/vendor/autoload.php';
  if (!\is_file($vendor)) {
    throw new \RuntimeException('No se encontró vendor/autoload.php (PhpSpreadsheet requerido).');
  }
  require_once $vendor;

  $tolower = function(string $value): string {
    if (\function_exists('mb_strtolower')) {
      return \mb_strtolower($value);
    }
    return \strtolower($value);
  };

  $reader = $ext === 'xlsx'
    ? new \PhpOffice\PhpSpreadsheet\Reader\Xlsx()
    : new \PhpOffice\PhpSpreadsheet\Reader\Xls();
  $reader->setReadDataOnly(true);
  $spreadsheet = $reader->load($destAbs);

  $selectSheetByName = function(array $names, string $preferred) use ($tolower) {
    if ($preferred === '') return null;
  $norm = function(string $value) use ($tolower): string {
      $value = \trim($value);
      if (function_exists('iconv')) {
        $tmp = @\iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $value);
        if ($tmp !== false) { $value = $tmp; }
      }
      $value = $tolower($value);
      return \preg_replace('/[^a-z0-9]+/', '', $value);
    };
    $target = $norm($preferred);
    foreach ($names as $idx => $name) {
      if ($norm($name) === $target) {
        return (int)$idx;
      }
    }
    return null;
  };

  $sheetNames = $spreadsheet->getSheetNames();
  $sheetIndex = 0;
  $matchedIndex = $selectSheetByName($sheetNames, $sheetPreferred);
  if ($matchedIndex === null) {
    $matchedIndex = $selectSheetByName($sheetNames, 'detalle');
  }
  if ($matchedIndex === null) {
    foreach ($sheetNames as $idx => $name) {
      $lowerName = $tolower($name);
      if (\strpos($lowerName, 'detalle') !== false || \strpos($lowerName, 'detail') !== false) {
        $matchedIndex = (int)$idx;
        break;
      }
    }
  }
  if ($matchedIndex !== null) {
    $sheetIndex = $matchedIndex;
  }

  $sheet = $spreadsheet->getSheet($sheetIndex);
  $selectedSheetName = $sheet->getTitle();

  $highestRow = $sheet->getHighestDataRow();
  $highestCol = $sheet->getHighestDataColumn();
  $highestColIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestCol);

  // Cabeceras normalizadas (fila 1)
  $headers = [];
  for ($c = 1; $c <= $highestColIndex; $c++) {
    $letter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($c);
    $val = (string)$sheet->getCell($letter . '1')->getValue();
    $key = \mb_strtolower(\trim($val));
    $key = \preg_replace('/\s+/', '_', $key);
    $headers[$c] = $key;
  }

  // Aliases tolerantes
  $mapWanted = [
    'sku'               => ['sku','sku_cliente','cod','codigo','código','producto','producto_codigo'],
    'descripcion'       => ['denominacion','denominación','descripcion','descripción','producto_nombre','description'],
    'lote'              => ['lote','lote_codigo','lote_code','batch','lot'],
    'fecha_produccion'  => ['fecha_produccion','f_prod','fp','fecha_elaboracion','elaboracion'],
    'fecha_vencimiento' => ['fecha_vencimiento','vencimiento','expiry','exp_date','fvto'],
    'uv_cajas'          => ['uv_cajas','cajas','caja','bultos','cx'],
    'uc_por_caja'       => ['uc_por_caja','uxc','unid_x_caja','unidades_por_caja','units_per_box'],
    'uc_sueltas'        => ['uc_sueltas','sueltas','unidades_sueltas','sueltos'],
    'unidades_total'    => ['unidades','und','uds','uv','unit','units_total','total_unidades'],
    'pallet_hint'       => ['pallet','pallet_hint','pallet_sugerido','pallet_id'],
    'position_hint'     => ['position','position_hint','posicion','posicion_sugerida','posicion_destino','rack','ubicacion','ubicacion_destino'],
  ];
  $findColByAliases = function(array $aliases) use ($headers): ?int {
    foreach ($headers as $colIdx => $name) {
      foreach ($aliases as $a) {
        if ($name === $a) return $colIdx;
      }
    }
    return null;
  };
  $colIndex = [];
  foreach ($mapWanted as $k => $aliases) {
    $colIndex[$k] = $findColByAliases($aliases);
  }
  $cell = function(?int $idx, int $row) use ($sheet) {
    if (empty($idx)) return null;
    $letter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($idx);
    return $sheet->getCell($letter . $row)->getCalculatedValue();
  };

  // 👇 Nota: NO insertamos la columna "status" para evitar problemas de NOT NULL/ENUM
  $sqlRow = "INSERT INTO pl_import_row (batch_id, rownum, raw) VALUES (:batch_id, :rownum, :raw)";
  $sti = $pdo->prepare($sqlRow);

  $rowsOk = 0;
  $rowsErr = 0;
  $rownumCounter = 0;
  $firstErrors = [];

  for ($r = 2; $r <= $highestRow; $r++) {
    try {
      // Extraer campos
      $sku   = \trim((string)($cell($colIndex['sku'], $r) ?? ''));
      $desc  = \trim((string)($cell($colIndex['descripcion'], $r) ?? ''));
      $lote  = \trim((string)($cell($colIndex['lote'], $r) ?? ''));
      $fprod = to_date_ymd($cell($colIndex['fecha_produccion'], $r));
      $fvto  = to_date_ymd($cell($colIndex['fecha_vencimiento'], $r));

      // Si la fila está totalmente vacía, saltar
      if ($sku === '' && $desc === '' && $lote === '' && $fprod === null && $fvto === null) {
        continue;
      }

      // Cantidades
      $uv_cajas    = (int)\max(0, (float)($cell($colIndex['uv_cajas'], $r) ?? 0));
      $uc_por_caja = (int)\max(0, (float)($cell($colIndex['uc_por_caja'], $r) ?? 0));
      $uc_sueltas  = (int)\max(0, (float)($cell($colIndex['uc_sueltas'], $r) ?? 0));

      // Si vino un total de unidades, derivar sueltas si no vinieron
      $total_uc_in = $cell($colIndex['unidades_total'], $r);
      if ($uc_sueltas === 0 && $total_uc_in !== null && ($uv_cajas > 0 || $uc_por_caja > 0)) {
        $total = (int)\max(0, (float)$total_uc_in);
        $calc  = $total - ($uv_cajas * $uc_por_caja);
        if ($calc > 0) $uc_sueltas = $calc;
      }

      $expected_uv = $uv_cajas; // cajas
      $expected_uc = ($uv_cajas * $uc_por_caja) + $uc_sueltas; // unidades totales

      $palletHint = \trim((string)($cell($colIndex['pallet_hint'], $r) ?? ''));
      $palletHint = $palletHint !== '' ? $palletHint : null;
      $positionHint = \trim((string)($cell($colIndex['position_hint'], $r) ?? ''));
      $positionHint = $positionHint !== '' ? $positionHint : null;

      // Armar RAW JSON como espera el SP
      $raw = [
        'sku_cliente'       => $sku,
        'descripcion'       => ($desc !== '' ? $desc : null),
        'lote'              => ($lote !== '' ? $lote : null),
        'fecha_produccion'  => $fprod,
        'fecha_vencimiento' => $fvto,
        'expected_uv'       => $expected_uv,
        'expected_uc'       => $expected_uc,
        'pallet_hint'       => $palletHint,
        'position_hint'     => $positionHint,
      ];

      $rownumCounter++;
      $sti->execute([
        ':batch_id' => $batchId,
        ':rownum'   => $rownumCounter,
        ':raw'      => \json_encode($raw, JSON_UNESCAPED_UNICODE),
      ]);

      $rowsOk++;
    } catch (\Throwable $eRow) {
      $rowsErr++;
      if (\count($firstErrors) < 3) {
        $firstErrors[] = "fila_excel={$r}: " . $eRow->getMessage();
      }
    }
  }

  // 3) Actualizar contadores y (si hubo) log de errores
  $logExtra = '';
  if ($rowsErr > 0 && !empty($firstErrors)) {
    $logExtra = ' · errs: ' . \implode(' | ', $firstErrors);
  }

  $upd = $pdo->prepare("UPDATE pl_import_batch
                          SET rows_total = :rtotal, rows_ok = :rok, rows_error = :rerr,
                              sheet_name = COALESCE(sheet_name, :sheetname),
                              log = CONCAT(COALESCE(log,''), :logextra)
                        WHERE id = :id");
  $upd->execute([
    ':rtotal'   => $rowsOk + $rowsErr,
    ':rok'      => $rowsOk,
    ':rerr'     => $rowsErr,
    ':sheetname'=> $selectedSheetName,
    ':logextra' => $logExtra,
    ':id'       => $batchId,
  ]);

  $pdo->commit();

  jexit([
    'ok'          => true,
    'batch_id'    => $batchId,
    'rows_ok'     => $rowsOk,
    'rows_error'  => $rowsErr,
    'file_url'    => $baseUrl . $destRel,
    'file_name'   => $origName,
  ]);
} catch (\Throwable $e) {
  if (isset($pdo) && $pdo instanceof \PDO && $pdo->inTransaction()) {
    $pdo->rollBack();
  }
  jexit([
    'ok'    => false,
    'error' => 'Error procesando el archivo: ' . $e->getMessage(),
  ], 500);
}
