<?php
declare(strict_types=1);

/**
 * SOL - Sistema de Operaciones Logísticas
 * API: Recepción (Entrante) — backend para recepción contra Packing List
 *
 * ## Objetivo
 *  - Contrastar lo esperado (pl_packinglist / pl_packinglist_item) con lo recibido (pl_rcv_link).
 *  - Registrar recepciones por plan detallado (pallets/ítems) o por ítem (autogenerando pallets).
 *  - Proveer listados/resúmenes para la UI (DataTables).
 *
 * ## Tablas involucradas (resumen funcional)
 *  - pl_packinglist (cabecera PL) / pl_packinglist_item (líneas normalizadas: expected_uv/uc, received_* y diff_*).
 *  - pl_rcv_link: vínculo de recepciones (por pallet y por ítem). Triggers actualizan received_* del item.
 *  - pl_import_batch / pl_import_row: staging de PL importado (Excel/CSV). En `row/raw` viene JSON con claves libres (e.g. "lote").
 *  - para_productos / para_producto_paletizado: maestro de producto y parámetros (p.ej. cajas_por_pallet).
 *  - wh_pallet / wh_posicion / wh_pallet_estado: palets, ubicaciones físicas, estados.
 *
 * ## Stored Procedures utilizados
 *  - sp_pl_recibir_desde_pl(pl_codigo, deposito_code, pos_code, plan_json, check_items, check_all_if_full)
 *      * plan_json: se expande con JSON_TABLE. **Claves esperadas**:
 *        - Nivel pallet:   "pallet_codigo", "pos_code", "items": []
 *        - Nivel item:     "pl_item_id", "producto_id" (o "sku_cliente"), "lote_codigo", "uv", "uc"
 *      * Crea/ubica pallet, inserta en pl_rcv_link y actualiza received_* vía triggers.
 *
 * ## Endpoints
 *  GET   ?meta=pl_pendientes                 → Listado de PL con agregados esperados y recibidos
 *  GET   ?meta=pl_resumen&pl_id=###          → Resumen global del PL
 *  GET   ?meta=recepcion_list&pl_id=###      → Lista base para recibir (si ya hay recepción: por pallet; si no: por items esperados)
 *  POST  ?meta=recepcion_guardar             → Registrar recepción (plan detallado o auto)
 *  POST  ?meta=recepcion_guardar_item        → Registrar recepción autogenerando pallets para **un ítem** (reglas A/B)
 *
 * ## Notas de implementación
 *  - Claves del plan JSON alineadas al SP: usar **pos_code** (no "ubicacion") y **lote_codigo** (no "lote").
 *  - No encapsular transacciones externas: los SPs manejan su propia transacción.
 *  - Se renombra el código de pallet a "PAL-<PL>-NNN" luego de recibir (orden visual y trazabilidad).
 */

header('Content-Type: application/json; charset=utf-8');

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/app/Support/ViewHelpers.php';
require_once $BASE . '/app/Support/ApiHelpers.php';  // ← helpers extraídos
require_once $BASE . '/config/db.php';

/* ------------------------------------------------------------------
 * Conexión PDO (estricta)
 * ------------------------------------------------------------------ */
try {
  $pdo = getPDO(); // usa tu factory
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'ok' => false,
    'error' => 'No se pudo abrir la conexión PDO',
    'debug' => $e->getMessage(),
  ]);
  exit;
}

/* ------------------------------------------------------------------
 * Descubrimiento de tablas (según existan en la BD actual)
 * ------------------------------------------------------------------ */
$T_PL_HDR = pickExistingTable($pdo, ['pl_packinglist']);
$T_PL_ITEM = pickExistingTable($pdo, ['pl_packinglist_item']);
$T_CLIENTE = pickExistingTable($pdo, ['para_clientes']);
$T_IMPORT_B = pickExistingTable($pdo, ['pl_import_batch']);
$T_IMPORT_R = pickExistingTable($pdo, ['pl_import_row']);
$RECV_TABLE = pickExistingTable($pdo, ['pl_rcv_link']);
$T_PALETIZ = pickExistingTable($pdo, ['para_producto_paletizado']);

if (!$T_PL_HDR)
  json_err('Falta tabla pl_packinglist en la BD.', null, 500);
if (!$T_PL_ITEM)
  json_err('Falta tabla pl_packinglist_item en la BD.', null, 500);

/* ------------------------------------------------------------------
 * Helpers de negocio locales (específicos de recepción)
 * ------------------------------------------------------------------ */

/**
 * Renombra los pallets asociados a un PL a formato PAL-<PL>-NNN
 * - No afecta la relación con pl_rcv_link ni movimientos; sólo orden visual/trazabilidad.
 */
function renamePalletCodesForPL(PDO $pdo, int $plId): array
{
  if (!tableExists($pdo, 'wh_pallet') || $plId <= 0) {
    return ['updated' => 0, 'total' => 0, 'mapping' => []];
  }

  $st = $pdo->prepare(
    "SELECT p.id AS pallet_id, p.codigo AS cur_code, MIN(l.id) AS first_link
       FROM pl_rcv_link l
       JOIN wh_pallet p ON p.id = l.pallet_id
      WHERE l.packinglist_id = ?
   GROUP BY p.id, p.codigo
   ORDER BY first_link ASC"
  );
  $st->execute([$plId]);
  $rows = $st->fetchAll();

  $updated = 0;
  $map = [];
  $seq = 1;

  foreach ($rows as $r) {
    $pid = (int) $r['pallet_id'];
    $cur = (string) $r['cur_code'];
    $new = 'PAL-' . $plId . '-' . str_pad((string) $seq, 3, '0', STR_PAD_LEFT);
    $seq++;

    if ($cur === $new) {
      $map[] = ['pallet_id' => $pid, 'from' => $cur, 'to' => $new, 'changed' => false];
      continue;
    }

    $chk = $pdo->prepare("SELECT id FROM wh_pallet WHERE codigo=? AND id<>? LIMIT 1");
    $chk->execute([$new, $pid]);
    if ($chk->fetchColumn()) {
      // garantizar unicidad si ya existe el código generado
      do {
        $new = 'PAL-' . $plId . '-' . str_pad((string) $seq, 3, '0', STR_PAD_LEFT);
        $seq++;
        $chk->execute([$new, $pid]);
      } while ($chk->fetchColumn());
    }

    $up = $pdo->prepare("UPDATE wh_pallet SET codigo=? WHERE id=?");
    $up->execute([$new, $pid]);
    $changed = (int) $up->rowCount() > 0;
    $updated++;
    $map[] = ['pallet_id' => $pid, 'from' => $cur, 'to' => $new, 'changed' => $changed];
  }

  return ['updated' => $updated, 'total' => count($rows), 'mapping' => $map];
}

/**
 * Builder de pallets para el SP, con claves alineadas al JSON_TABLE del SP.
 * - Usa "pos_code" (no "ubicacion")
 * - Usa "lote_codigo" (no "lote")
 */
function makePlanPalletForItem(array $it, int $itemId, int $uv, ?int $uc, bool $autoUb, string $posCode, bool $pickLast): array
{
  $uc = ($uc === null) ? 0 : (int) $uc;
  return [
    'pallet_codigo' => null,
    'pos_code' => ($autoUb && $posCode === '') ? 'CUARENTENA' : $posCode,
    'estado' => 'OK',            // extra: ignorado por el SP
    'pickeado' => $pickLast ? 1 : 0, // extra: ignorado por el SP
    'items' => [
      [
        'pl_item_id' => $itemId,
        'producto_id' => (int) ($it['producto_id'] ?? 0),
        'sku_cliente' => $it['sku_cliente'] ?? null,   // el SP lo acepta NULL
        'lote_codigo' => (string) $it['lote_codigo'],   // ← clave que el SP lee
        'fecha_produccion' => $it['fecha_produccion'] ?? null,  // ignorado por SP
        'fecha_vencimiento' => $it['fecha_vencimiento'] ?? null, // ignorado por SP
        'uv' => (int) $uv,
        'uc' => (int) $uc,
      ]
    ],
  ];
}

/**
 * Resolver un código de posición válido para wh_posicion.id=1.
 * Intenta code_full, pos_code_full, code, pos_code, title. Si no hay, devuelve 'CUARENTENA' como fallback.
 */
function resolvePosCodeForId(PDO $pdo, int $posId = 1): string
{
  if (!tableExists($pdo, 'wh_posicion')) return 'CUARENTENA';
  $cols = [];
  foreach (['code_full','pos_code_full','code','pos_code','title'] as $c) {
    if (columnExists($pdo, 'wh_posicion', $c)) $cols[] = $c;
  }
  if (!$cols) return 'CUARENTENA';
  $select = implode(', ', array_map(fn($c) => "`$c`", $cols));
  try {
    $st = $pdo->prepare("SELECT $select FROM wh_posicion WHERE id = ? LIMIT 1");
    $st->execute([$posId]);
    $row = $st->fetch(PDO::FETCH_ASSOC);
    if ($row) {
      foreach (['code_full','pos_code_full','code','pos_code','title'] as $c) {
        if (isset($row[$c]) && $row[$c] !== null && $row[$c] !== '') {
          return (string)$row[$c];
        }
      }
    }
  } catch (Throwable $e) { /* ignore */ }
  return 'CUARENTENA';
}

function resolveDefaultPosCodeForId1(PDO $pdo): string { return resolvePosCodeForId($pdo, 1); }

/**
 * Obtener el deposito_code correspondiente a la posición id=1.
 * Busca wh_posicion.deposito_id -> wh_deposito.code (o codigo/nombre como fallback).
 */
function resolveDepositoCodeForPosId(PDO $pdo, int $posId = 1): ?string
{
  if (!tableExists($pdo, 'wh_posicion')) return null;
  if (!columnExists($pdo, 'wh_posicion', 'deposito_id')) return null;
  $depId = null;
  try {
    $st = $pdo->prepare("SELECT deposito_id FROM wh_posicion WHERE id = ? LIMIT 1");
    $st->execute([$posId]);
    $depId = $st->fetchColumn();
  } catch (Throwable $e) { $depId = null; }
  if (!$depId) return null;

  if (!tableExists($pdo, 'wh_deposito')) return null;
  $depCols = [];
  foreach (['code','codigo','nombre'] as $c) {
    if (columnExists($pdo, 'wh_deposito', $c)) $depCols[] = $c;
  }
  if (!$depCols) return null;
  $select = implode(', ', array_map(fn($c) => "`$c`", $depCols));
  try {
    $q = $pdo->prepare("SELECT $select FROM wh_deposito WHERE id = ? LIMIT 1");
    $q->execute([$depId]);
    $row = $q->fetch(PDO::FETCH_ASSOC);
    if ($row) {
      foreach (['code','codigo','nombre'] as $c) {
        if (isset($row[$c]) && $row[$c] !== null && $row[$c] !== '') return (string)$row[$c];
      }
    }
  } catch (Throwable $e) { /* ignore */ }
  return null;
}

/**
 * Asegura que exista un registro en pl_movil con el mismo ID que el seleccionado
 * en para_moviles. Si aún no existe lo crea copiando los datos mínimos.
 */
function ensurePlMovilFromPara(PDO $pdo, int $paraMovilId): ?int
{
  if ($paraMovilId <= 0) return null;
  if (!tableExists($pdo, 'pl_movil')) return $paraMovilId;

  try {
    $chk = $pdo->prepare('SELECT id FROM pl_movil WHERE id = ? LIMIT 1');
    $chk->execute([$paraMovilId]);
    if ($chk->fetchColumn()) {
      return $paraMovilId;
    }
  } catch (Throwable $e) {
    return null;
  }

  if (!tableExists($pdo, 'para_moviles')) return null;

  try {
    $src = $pdo->prepare('SELECT id, chapa, activo FROM para_moviles WHERE id = ? LIMIT 1');
    $src->execute([$paraMovilId]);
    $row = $src->fetch(PDO::FETCH_ASSOC);
    if (!$row) return null;

    $patente = trim((string)($row['chapa'] ?? ''));
    if ($patente === '') {
      $patente = 'MOV-' . $paraMovilId;
    }
    $patente = mb_substr($patente, 0, 20);
    $activo = isset($row['activo']) ? (int)$row['activo'] : 1;

    $cols = ['id', 'patente', 'activo'];
    $place = ['?', '?', '?'];
    $params = [$paraMovilId, $patente, $activo];

    if (columnExists($pdo, 'pl_movil', 'empresa')) {
      $cols[] = 'empresa';
      $place[] = '?';
      $params[] = null; // no tenemos empresa en para_moviles
    }

    $sql = 'INSERT INTO pl_movil (' . implode(',', $cols) . ') VALUES (' . implode(',', $place) . ')
            ON DUPLICATE KEY UPDATE patente = VALUES(patente), activo = VALUES(activo)';
    if (columnExists($pdo, 'pl_movil', 'empresa')) {
      $sql .= ', empresa = VALUES(empresa)';
    }

    $pdo->prepare($sql)->execute($params);
    return $paraMovilId;
  } catch (Throwable $e) {
    return null;
  }
}

/**
 * Garantiza que exista un chofer compatible en pl_chofer a partir de para_choferes.
 */
function ensurePlChoferFromPara(PDO $pdo, int $paraChoferId): ?int
{
  if ($paraChoferId <= 0) return null;
  if (!tableExists($pdo, 'pl_chofer')) return $paraChoferId;

  try {
    $chk = $pdo->prepare('SELECT id FROM pl_chofer WHERE id = ? LIMIT 1');
    $chk->execute([$paraChoferId]);
    if ($chk->fetchColumn()) {
      return $paraChoferId;
    }
  } catch (Throwable $e) {
    return null;
  }

  if (!tableExists($pdo, 'para_choferes')) return null;

  try {
    $src = $pdo->prepare('SELECT id, nombre, documento, activo FROM para_choferes WHERE id = ? LIMIT 1');
    $src->execute([$paraChoferId]);
    $row = $src->fetch(PDO::FETCH_ASSOC);
    if (!$row) return null;

    $nombre = trim((string)($row['nombre'] ?? ''));
    if ($nombre === '') {
      $nombre = 'CHOFER ' . $paraChoferId;
    }
    $nombre = mb_substr($nombre, 0, 100);
    $doc = isset($row['documento']) ? mb_substr(trim((string)$row['documento']), 0, 50) : null;
    $activo = isset($row['activo']) ? (int)$row['activo'] : 1;

    $cols = ['id', 'nombre', 'activo'];
    $place = ['?', '?', '?'];
    $params = [$paraChoferId, $nombre, $activo];

    if (columnExists($pdo, 'pl_chofer', 'doc_numero')) {
      $cols[] = 'doc_numero';
      $place[] = '?';
      $params[] = $doc;
    }

    if (columnExists($pdo, 'pl_chofer', 'empresa')) {
      $cols[] = 'empresa';
      $place[] = '?';
      $params[] = null;
    }

    $sql = 'INSERT INTO pl_chofer (' . implode(',', $cols) . ') VALUES (' . implode(',', $place) . ')
            ON DUPLICATE KEY UPDATE nombre = VALUES(nombre), activo = VALUES(activo)';
    if (columnExists($pdo, 'pl_chofer', 'doc_numero')) {
      $sql .= ', doc_numero = VALUES(doc_numero)';
    }
    if (columnExists($pdo, 'pl_chofer', 'empresa')) {
      $sql .= ', empresa = VALUES(empresa)';
    }

    $pdo->prepare($sql)->execute($params);
    return $paraChoferId;
  } catch (Throwable $e) {
    return null;
  }
}

/**
 * Persistir/actualizar metadata en pl_ingreso si existe una fila para el packinglist
 * Recibe el payload $in (array) y aplica los campos opcionales si existen.
 */
function updatePlIngresoFromPayload(PDO $pdo, int $plId, array $in): void
{
  if ($plId <= 0) return;
  // Verificar existencia de la tabla y de la fila
  if (!tableExists($pdo, 'pl_ingreso')) return;
  $st = $pdo->prepare("SELECT id FROM pl_ingreso WHERE packinglist_id = ? LIMIT 1");
  $st->execute([$plId]);
  $rid = $st->fetchColumn();
  if (!$rid) {
    // No existe fila; crear una entrada mínima en pl_ingreso usando datos provistos en el payload.
    // Necesitamos deposito_id (no nulo). Intentamos resolverlo por deposito_id, deposito_code o usar la primera fila en wh_deposito.
    $depositoId = null;
    if (!empty($in['deposito_id'])) {
      $depositoId = (int)$in['deposito_id'];
    } elseif (!empty($in['deposito_code'])) {
      try {
        $q = $pdo->prepare("SELECT id FROM wh_deposito WHERE code = ? LIMIT 1");
        $q->execute([ (string)$in['deposito_code'] ]);
        $depositoId = (int)$q->fetchColumn();
      } catch (Throwable $e) { /* ignore */ }
    }
    if (empty($depositoId)) {
      // fallback: first deposito
      try {
        $q = $pdo->query("SELECT id FROM wh_deposito ORDER BY id LIMIT 1");
        $depositoId = (int)$q->fetchColumn();
      } catch (Throwable $e) { $depositoId = null; }
    }

    if (!empty($in['movil_id'])) {
      $movilResolved = ensurePlMovilFromPara($pdo, (int)$in['movil_id']);
      if ($movilResolved) {
        $in['movil_id'] = $movilResolved;
      } else {
        unset($in['movil_id']);
      }
    }

    if (!empty($in['chofer_id'])) {
      $choferResolved = ensurePlChoferFromPara($pdo, (int)$in['chofer_id']);
      if ($choferResolved) {
        $in['chofer_id'] = $choferResolved;
      } else {
        unset($in['chofer_id']);
      }
    }

    // Build insert columns with minimal required fields
    $insCols = ['packinglist_id'];
    $insVals = ['?'];
    $params = [$plId];

    if ($depositoId) { $insCols[] = 'deposito_id'; $insVals[] = '?'; $params[] = $depositoId; }
    if (!empty($in['movil_id'])) { $insCols[] = 'movil_id'; $insVals[] = '?'; $params[] = $in['movil_id']; }
    if (!empty($in['chofer_id'])) { $insCols[] = 'chofer_id'; $insVals[] = '?'; $params[] = $in['chofer_id']; }
    if (!empty($in['movil_desc'])) { $insCols[] = 'movil_desc'; $insVals[] = '?'; $params[] = $in['movil_desc']; }
    if (!empty($in['chofer_desc'])) { $insCols[] = 'chofer_desc'; $insVals[] = '?'; $params[] = $in['chofer_desc']; }
    if (!empty($in['fecha_ingreso'])) { $insCols[] = 'fecha_ingreso'; $insVals[] = '?'; $params[] = $in['fecha_ingreso']; }

    $sqlIns = "INSERT INTO pl_ingreso (" . implode(',', $insCols) . ") VALUES (" . implode(',', $insVals) . ")";
    try {
      $pdo->prepare($sqlIns)->execute($params);
      $rid = (int)$pdo->lastInsertId();
    } catch (Throwable $e) {
      // If insert fails, bail out silently to avoid breaking reception
      return;
    }
  }

  if (!empty($in['movil_id'])) {
    $movilResolved = ensurePlMovilFromPara($pdo, (int)$in['movil_id']);
    if ($movilResolved) {
      $in['movil_id'] = $movilResolved;
    } else {
      unset($in['movil_id']);
    }
  }

  if (!empty($in['chofer_id'])) {
    $choferResolved = ensurePlChoferFromPara($pdo, (int)$in['chofer_id']);
    if ($choferResolved) {
      $in['chofer_id'] = $choferResolved;
    } else {
      unset($in['chofer_id']);
    }
  }

  // Campos aceptados (opcionalmente presentes en el payload)
  $allowed = [
    'fecha_ingreso' => 'fecha_ingreso',
    'hora_arribo' => 'llegada_at',
    'movil_id' => 'movil_id',
    'chofer_id' => 'chofer_id',
    'operarios_cant' => 'operarios_cant',
    'tipo_documento' => 'doc_tipo',
    'documento_entrada' => 'doc_numero',
    'observacion' => 'observacion',
    'hora_inicio' => 'descarga_inicio_at',
    'hora_fin' => 'descarga_fin_at',
    'cliente_id' => 'cliente_id',
    'responsable_id' => 'responsable_id',
    'autorizado_id' => 'autorizado_id',
    'operario_id' => 'operario_id',
    // estado_id left for manual management if needed
  ];

  $sets = [];
  $params = [];
  foreach ($allowed as $src => $col) {
    if (array_key_exists($src, $in) && $in[$src] !== null && $in[$src] !== '') {
      $sets[] = "$col = ?";
      $params[] = $in[$src];
    }
  }

  if (!$sets) return; // nothing to update

  $params[] = $rid;
  $sql = "UPDATE pl_ingreso SET " . implode(', ', $sets) . " WHERE id = ? LIMIT 1";
  $upd = $pdo->prepare($sql);
  $upd->execute($params);
}

/**
 * Crea o actualiza un registro en recepcion_hdr para la PL y enlaza los pl_rcv_link
 * creados desde $opStart (timestamp) para asociarlos a la recepcion.
 * Devuelve el recepcion_id creado/actualizado o null si no se pudo.
 */
function createOrUpdateRecepcionHdrAndAttachLinks(PDO $pdo, int $plId, array $payload, string $opStart, ?int $uid = null): ?int
{
  if ($plId <= 0) return null;
  if (!tableExists($pdo, 'recepcion_hdr')) return null;

  // Intentar encontrar un header ya reciente para este PL (misma fecha/hora)
  $st = $pdo->prepare("SELECT id FROM recepcion_hdr WHERE packinglist_id = ? ORDER BY created_at DESC LIMIT 1");
  $st->execute([$plId]);
  $existing = $st->fetchColumn();

  $fields = [
    'fecha_ingreso' => $payload['fecha_ingreso'] ?? null,
    'llegada_at' => $payload['hora_arribo'] ?? null,
    'descarga_inicio_at' => $payload['hora_inicio'] ?? null,
    'descarga_fin_at' => $payload['hora_fin'] ?? null,
    'movil_id' => $payload['movil_id'] ?? null,
    'chofer_id' => $payload['chofer_id'] ?? null,
    'operarios_cant' => $payload['operarios_cant'] ?? null,
    'doc_tipo' => $payload['tipo_documento'] ?? null,
    'doc_numero' => $payload['documento_entrada'] ?? $payload['doc_numero'] ?? null,
    'observacion' => $payload['observacion'] ?? null,
    'cliente_id' => $payload['cliente_id'] ?? null,
    'responsable_id' => $payload['responsable_id'] ?? null,
    'autorizado_id' => $payload['autorizado_id'] ?? null,
    'operario_id' => $payload['operario_id'] ?? null,
    'estado_id' => $payload['estado_id'] ?? null,
  ];

  // Build insert or update
  if ($existing) {
    // UPDATE only the non-null fields
    $sets = [];
    $params = [];
    foreach ($fields as $k => $v) {
      if ($v !== null && $v !== '') { $sets[] = "$k = ?"; $params[] = $v; }
    }
    if ($sets) {
      $params[] = (int)$existing;
      $sql = "UPDATE recepcion_hdr SET " . implode(', ', $sets) . ", updated_at = NOW() WHERE id = ? LIMIT 1";
      $pdo->prepare($sql)->execute($params);
    }
    $rid = (int)$existing;
  } else {
    $cols = ['packinglist_id'];
    $vals = ['?'];
    $params = [$plId];
    foreach ($fields as $k => $v) {
      if ($v !== null && $v !== '') { $cols[] = $k; $vals[] = '?'; $params[] = $v; }
    }
    if ($uid !== null) { $cols[] = 'created_by'; $vals[] = '?'; $params[] = $uid; }
    $sql = "INSERT INTO recepcion_hdr (" . implode(',', $cols) . ") VALUES (" . implode(',', $vals) . ")";
    $pdo->prepare($sql)->execute($params);
    $rid = (int)$pdo->lastInsertId();
  }

  // Asociar pl_rcv_link creados desde $opStart para este PL
  if (tableExists($pdo, 'pl_rcv_link')) {
    // safe update: only rows of this packinglist and created_at >= opStart
    $u = $pdo->prepare("UPDATE pl_rcv_link SET recepcion_id = ? WHERE packinglist_id = ? AND created_at >= ?");
    $u->execute([$rid, $plId, $opStart]);
  }

  return $rid;
}


/**
 * Assign cliente_id from the PL to pallets created/linked for this PL.
 * Safe: only runs if `wh_pallet` has column `cliente_id` and pl_packinglist has `cliente_id`.
 */
function assignClienteToPalletsFromPl(PDO $pdo, int $plId): array
{
  if ($plId <= 0) return ['updated' => 0];
  if (!tableExists($pdo, 'wh_pallet')) return ['updated' => 0];
  if (!columnExists($pdo, 'wh_pallet', 'cliente_id')) return ['updated' => 0];
  if (!tableExists($pdo, 'pl_packinglist')) return ['updated' => 0];
  if (!columnExists($pdo, 'pl_packinglist', 'cliente_id')) return ['updated' => 0];

  // Update pallets linked via pl_rcv_link to take the packinglist's cliente_id.
  $sql = "UPDATE wh_pallet p
           JOIN pl_rcv_link l ON l.pallet_id = p.id
           JOIN pl_packinglist pl ON pl.id = l.packinglist_id
           SET p.cliente_id = pl.cliente_id
           WHERE pl.id = ?
             AND (p.cliente_id IS NULL OR p.cliente_id <> pl.cliente_id)";
  $st = $pdo->prepare($sql);
  $st->execute([$plId]);
  return ['updated' => (int)$st->rowCount()];
}

/**
 * Verifica que el payload contenga los metadatos obligatorios de recepción.
 * Lanza json_err con HTTP 422 si falta alguno.
 */
function assertRequiredReceptionMeta(array $payload): void
{
  $required = [
    'fecha_ingreso'     => 'Fecha de ingreso',
    'hora_arribo'       => 'Hora de arribo',
    'hora_inicio'       => 'Hora de inicio',
    'hora_fin'          => 'Hora de fin',
    'operarios_cant'    => 'Cantidad de operarios',
    'tipo_documento'    => 'Tipo de documento',
    'documento_entrada' => 'Documento',
    'movil_id'          => 'Móvil',
    'chofer_id'         => 'Chofer',
  ];

  $missing = [];

  foreach ($required as $key => $label) {
    if (!array_key_exists($key, $payload)) {
      $missing[$key] = $label;
      continue;
    }

    $value = $payload[$key];
    if ($key === 'operarios_cant') {
      if ($value === '' || $value === null || !is_numeric($value) || (int)$value <= 0) {
        $missing[$key] = $label;
      }
      continue;
    }

    if ($key === 'tipo_documento') {
      $valueStr = strtoupper(is_string($value) ? trim($value) : (string)$value);
      if ($valueStr === '' || !in_array($valueStr, ['FACTURA', 'REMITO'], true)) {
        $missing[$key] = $label;
      }
      continue;
    }

    $valueStr = is_string($value) ? trim($value) : (is_numeric($value) ? (string)$value : '');
    if ($valueStr === '') {
      $missing[$key] = $label;
    }
  }

  if ($missing) {
    json_err(
      'Faltan metadatos obligatorios para registrar la recepción: ' . implode(', ', array_values($missing)),
      ['missing_keys' => array_keys($missing)],
      422
    );
  }
}



/* ==================================================================
 * Endpoint: GET ?meta=pl_pendientes
 * ------------------------------------------------------------------
 * Propósito:
 *    Listar hasta 100 PL con agregados de esperados y, si existen,
 *    agregados de recibidos (pallets/uv/uc).
 * Parámetros:
 *    - q (opcional): string de búsqueda libre.
 * Respuesta:
 *    [{ pl_id, pl_codigo, pl_fecha, cliente, items_esp, uv_esp, uc_esp,
 *       productos_unicos, pallets_recv, uv_recv, uc_recv }]
 * Notas:
 *    - Si existe para_clientes y el header del PL tiene cliente_id, muestra su razón social.
 * ================================================================== */
function ep_pl_pendientes(PDO $pdo, string $T_PL_HDR, string $T_PL_ITEM, ?string $T_CLIENTE, ?string $RECV_TABLE): void
{
  if ($T_CLIENTE && columnExists($pdo, $T_PL_HDR, 'cliente_id')) {
    $sqlH = "SELECT h.id AS pl_id, h.codigo AS pl_codigo, h.fecha AS pl_fecha,
                    COALESCE(c.razon_social, h.cliente_ref) AS cliente
               FROM $T_PL_HDR h
          LEFT JOIN $T_CLIENTE c ON c.id = h.cliente_id";
  } else {
    $sqlH = "SELECT h.id AS pl_id, h.codigo AS pl_codigo, h.fecha AS pl_fecha,
                    h.cliente_ref AS cliente
               FROM $T_PL_HDR h";
  }

  $sqlAggExp = "SELECT i.packinglist_id AS pl_id,
                       COUNT(*) AS items_esp,
                       COALESCE(SUM(i.expected_uv),0) AS uv_esp,
                       COALESCE(SUM(i.expected_uc),0) AS uc_esp,
                       COUNT(DISTINCT i.producto_id) AS prod_uniq
                  FROM $T_PL_ITEM i
              GROUP BY i.packinglist_id";

  // Aggregate of pending items (those with remaining expected>received)
  $sqlAggPending = "SELECT i.packinglist_id AS pl_id,
                             SUM(CASE WHEN (GREATEST(COALESCE(i.expected_uv,0)-COALESCE(i.received_uv,0),0) > 0)
                                           OR (GREATEST(COALESCE(i.expected_uc,0)-COALESCE(i.received_uc,0),0) > 0)
                                      THEN 1 ELSE 0 END) AS items_pending
                        FROM $T_PL_ITEM i
                    GROUP BY i.packinglist_id";

  $joinRecv = "";
  $selRecv = ", 0 AS pallets_recv, 0 AS uv_recv, 0 AS uc_recv";
  if ($RECV_TABLE) {
    $joinRecv = "LEFT JOIN (
                   SELECT l.packinglist_id AS pl_id,
                          COUNT(DISTINCT l.pallet_id) AS pallets_recv,
                          COALESCE(SUM(l.uv_cajas),0) AS uv_recv,
                          COALESCE(SUM(l.uc_unidades),0) AS uc_recv
                     FROM $RECV_TABLE l
                 GROUP BY l.packinglist_id
                 ) rr ON rr.pl_id = h.pl_id";
    $selRecv = ", COALESCE(rr.pallets_recv,0) AS pallets_recv,
                 COALESCE(rr.uv_recv,0) AS uv_recv,
                 COALESCE(rr.uc_recv,0) AS uc_recv";
  }

  $q = trim((string) ($_GET['q'] ?? ''));
  $where = " WHERE 1=1 ";
  $params = [];
  if ($q !== '') {
    $where .= " AND (h.pl_codigo LIKE ? OR h.pl_fecha LIKE ? OR h.pl_id LIKE ? OR h.cliente LIKE ?)";
    $params = ["%$q%", "%$q%", "%$q%", "%$q%"];
  }

  // By default exclude PLs that have no pending items. To override and include
  // completed PLs set include_completed=1 in the query string.
  $includeCompleted = isset($_GET['include_completed']) && in_array(strval($_GET['include_completed']), ['1','true','yes'], true);


  $sql = "SELECT h.pl_id, h.pl_codigo, h.pl_fecha, h.cliente,
                 COALESCE(a.items_esp,0) AS items_esp,
                 COALESCE(a.uv_esp,0)    AS uv_esp,
                 COALESCE(a.uc_esp,0)    AS uc_esp,
                 COALESCE(a.prod_uniq,0) AS productos_unicos,
                 COALESCE(p.items_pending,0) AS items_pending
                 $selRecv
            FROM ($sqlH) h
       LEFT JOIN ($sqlAggExp) a ON a.pl_id = h.pl_id
       LEFT JOIN ($sqlAggPending) p ON p.pl_id = h.pl_id
       $joinRecv
           $where";

  // If not including completed, only return PLs with pending items
  if (! $includeCompleted) {
    $sql .= " AND COALESCE(p.items_pending,0) > 0";
  }

  // Append ordering and limit
  $sql .= "\n        ORDER BY h.pl_id DESC\n           LIMIT 100";

  $st = $pdo->prepare($sql);
  $st->execute($params);
  $rows = $st->fetchAll();
  // For convenience include items_pending in output when available
  foreach ($rows as &$r) {
    if (!array_key_exists('items_pending', $r)) {
      $r['items_pending'] = null;
    }
  }
  json_ok($rows);
}

/* ==================================================================
 * Endpoint: GET ?meta=pl_resumen&pl_id=###
 * ------------------------------------------------------------------
 * Propósito:
 *    Resumen global del PL: cabecera + agregados esperados + recibidos.
 * Parámetros:
 *    - pl_id: ID del PL (int, requerido)
 * Respuesta:
 *    { pl_id, pl_codigo, pl_fecha, cliente, items_esp, uv_esp, uc_esp,
 *      productos_unicos, pallets_recv, uv_recv, uc_recv, discrepancias }
 * ================================================================== */
function ep_pl_resumen(PDO $pdo, int $plId, string $T_PL_HDR, string $T_PL_ITEM, ?string $T_CLIENTE, ?string $RECV_TABLE): void
{
  if ($T_CLIENTE && columnExists($pdo, $T_PL_HDR, 'cliente_id')) {
    $st = $pdo->prepare("SELECT h.id AS pl_id, h.codigo AS pl_codigo, h.fecha AS pl_fecha,
                                COALESCE(c.razon_social, h.cliente_ref) AS cliente
                           FROM $T_PL_HDR h
                      LEFT JOIN $T_CLIENTE c ON c.id=h.cliente_id
                          WHERE h.id=? LIMIT 1");
  } else {
    $st = $pdo->prepare("SELECT h.id AS pl_id, h.codigo AS pl_codigo, h.fecha AS pl_fecha,
                                h.cliente_ref AS cliente
                           FROM $T_PL_HDR h
                          WHERE h.id=? LIMIT 1");
  }
  $st->execute([$plId]);
  $hdr = $st->fetch();
  if (!$hdr)
    json_err("PL no encontrado.");

  // Attempt to enrich resumen with movil/chofer data from pl_ingreso (latest)
  try {
    if (tableExists($pdo, 'pl_ingreso')) {
      // Build a safe list of columns that actually exist in pl_ingreso to avoid SQL errors
      $possible = [
        'movil_id','chofer_id','movil_desc','chofer_desc',
        'fecha_ingreso','llegada_at','descarga_inicio_at','descarga_fin_at',
        'operarios_cant','doc_tipo','doc_numero','observacion'
      ];
      $cols = [];
      foreach ($possible as $c) { if (columnExists($pdo, 'pl_ingreso', $c)) $cols[] = $c; }
      if ($cols) {
        $select = implode(', ', $cols);
        $st2 = $pdo->prepare("SELECT $select FROM pl_ingreso WHERE packinglist_id = ? ORDER BY created_at DESC LIMIT 1");
        $st2->execute([$plId]);
        $ing = $st2->fetch() ?: null;
        if ($ing) {
          if (array_key_exists('movil_id', $ing)) $hdr['movil_id'] = $ing['movil_id'] ?? null;
          if (array_key_exists('chofer_id', $ing)) $hdr['chofer_id'] = $ing['chofer_id'] ?? null;

          // textual fallbacks from descriptive columns if present
          if (array_key_exists('movil_desc', $ing)) $hdr['movil'] = $ing['movil_desc'] ?? null;
          if (array_key_exists('chofer_desc', $ing)) $hdr['chofer'] = $ing['chofer_desc'] ?? null;

          if (array_key_exists('fecha_ingreso', $ing)) $hdr['fecha_ingreso'] = $ing['fecha_ingreso'] ?? null;
          if (array_key_exists('llegada_at', $ing)) $hdr['hora_arribo'] = $ing['llegada_at'] ?? null;
          if (array_key_exists('descarga_inicio_at', $ing)) $hdr['hora_inicio'] = $ing['descarga_inicio_at'] ?? null;
          if (array_key_exists('descarga_fin_at', $ing)) $hdr['hora_fin'] = $ing['descarga_fin_at'] ?? null;
          if (array_key_exists('operarios_cant', $ing)) $hdr['operarios_cant'] = $ing['operarios_cant'] ?? null;
          if (array_key_exists('doc_tipo', $ing)) $hdr['tipo_documento'] = $ing['doc_tipo'] ?? null;
          if (array_key_exists('doc_numero', $ing)) $hdr['documento_entrada'] = $ing['doc_numero'] ?? null;
          if (array_key_exists('observacion', $ing)) $hdr['observacion'] = $ing['observacion'] ?? null;

          // Resolve names from master tables if needed (use movil_id/chofer_id)
          if (!empty($hdr['movil_id']) && tableExists($pdo, 'para_moviles') && columnExists($pdo, 'para_moviles', 'chapa')) {
            $q = $pdo->prepare('SELECT chapa FROM para_moviles WHERE id = ? LIMIT 1');
            $q->execute([(int)$hdr['movil_id']]);
            $mv = $q->fetchColumn(); if ($mv) $hdr['movil'] = $mv;
          }
          if (!empty($hdr['chofer_id']) && tableExists($pdo, 'para_choferes') && columnExists($pdo, 'para_choferes', 'nombre')) {
            $q = $pdo->prepare('SELECT nombre FROM para_choferes WHERE id = ? LIMIT 1');
            $q->execute([(int)$hdr['chofer_id']]);
            $ch = $q->fetchColumn(); if ($ch) $hdr['chofer'] = $ch;
          }
        }
      }
    }
  } catch (Throwable $e) { /* ignore enrichment errors */ }

  $st = $pdo->prepare("SELECT COUNT(*) AS items_esp,
                              COALESCE(SUM(expected_uv),0) AS uv_esp,
                              COALESCE(SUM(expected_uc),0) AS uc_esp,
                              COUNT(DISTINCT producto_id) AS productos_unicos
                         FROM $T_PL_ITEM
                        WHERE packinglist_id=?");
  $st->execute([$plId]);
  $agg = $st->fetch() ?: ['items_esp' => 0, 'uv_esp' => 0, 'uc_esp' => 0, 'productos_unicos' => 0];

  // Count items that still have pending quantities (expected > received)
  $st = $pdo->prepare("SELECT COALESCE(SUM(CASE WHEN (GREATEST(COALESCE(expected_uv,0)-COALESCE(received_uv,0),0) > 0)
                       OR (GREATEST(COALESCE(expected_uc,0)-COALESCE(received_uc,0),0) > 0)
                     THEN 1 ELSE 0 END),0) AS items_pending
          FROM $T_PL_ITEM
            WHERE packinglist_id = ?");
  $st->execute([$plId]);
  $pendingAgg = $st->fetch() ?: ['items_pending' => 0];

  $recv = ['pallets_recv' => 0, 'uv_recv' => 0, 'uc_recv' => 0];
  if ($RECV_TABLE) {
    $st = $pdo->prepare("SELECT COUNT(DISTINCT pallet_id) AS pallets_recv,
                                COALESCE(SUM(uv_cajas),0) AS uv_recv,
                                COALESCE(SUM(uc_unidades),0) AS uc_recv
                           FROM $RECV_TABLE
                          WHERE packinglist_id=?");
    $st->execute([$plId]);
    $recv = $st->fetch() ?: $recv;
  }

  json_ok(array_merge($hdr, $agg, $recv, ['items_pending' => (int)($pendingAgg['items_pending'] ?? 0), 'discrepancias' => 0]));
}

/* ==================================================================
 * Endpoint: GET ?meta=recepcion_list&pl_id=###
 * ------------------------------------------------------------------
 * Propósito:
 *    Devolver los datos base para la grilla de recepción.
 *    - Si ya hay recepciones: lista pallets recibidos (pallet, ubicacion, estado, uv/uc).
 *    - Si no hay recepciones: lista items esperados del PL (con pl_item_id y producto_id).
 * Parámetros:
 *    - pl_id: ID del PL (int, requerido)
 * Respuesta:
 *    Con recepción previa:
 *      [{ pallet_codigo, producto_txt?, lote, fecha_produccion, fecha_vencimiento,
 *         ubicacion, estado, items_count, uv, uc }]
 *    Sin recepción previa:
 *      [{ pl_item_id, producto_id, pallet_codigo:null, producto_txt, lote, fecha_produccion,
 *         fecha_vencimiento, uv, uc, peso:null, ubicacion:null, estado:'OK' }]
 * ================================================================== */
function ep_recepcion_list(PDO $pdo, int $plId, ?string $RECV_TABLE): void
{
  $T_PL_ITEM = $GLOBALS['T_PL_ITEM'] ?? null;
  if (!$T_PL_ITEM) json_err('Falta tabla pl_packinglist_item.', null, 500);

  $view = strtolower(trim((string)($_GET['view'] ?? 'all')));
  if (!in_array($view, ['all', 'pending', 'received'], true)) $view = 'all';

  // ---------- RECEIVED: pallets ya ingresados ----------
  $received = [];
  if ($RECV_TABLE) {
    // ¿existe FK al item?
    $itemFk = null;
    foreach (['pl_item_id', 'packinglist_item_id', 'item_id'] as $c) {
      if (columnExists($pdo, $RECV_TABLE, $c)) { $itemFk = $c; break; }
    }

    if ($itemFk) {
      $sqlRecv = "SELECT
                    pal.codigo AS pallet_codigo,
                    COALESCE(MIN(i.descripcion), MIN(i.sku_cliente)) AS producto_txt,
                    MIN(i.lote_codigo)            AS lote,
                    MIN(i.fecha_produccion)       AS fecha_produccion,
                    MIN(i.fecha_vencimiento)      AS fecha_vencimiento,
                    COALESCE(pos.code_full, pos.pos_code, pos.pos_code_full, pos.code) AS ubicacion,
                    COALESCE(est.code, 'OK') AS estado,
                    COUNT(*) AS items_count,
                    COALESCE(SUM(l.uv_cajas),0) AS uv,
                    COALESCE(SUM(l.uc_unidades),0) AS uc
                  FROM $RECV_TABLE l
                  LEFT JOIN $T_PL_ITEM i        ON i.id = l.`$itemFk`
                  LEFT JOIN wh_pallet pal       ON pal.id = l.pallet_id
                  LEFT JOIN wh_posicion pos     ON pos.id = pal.posicion_id
                  LEFT JOIN wh_pallet_estado est ON est.id = pal.estado_id
                  WHERE l.packinglist_id = ?
                  GROUP BY pal.id
                  ORDER BY pal.codigo";
    } else {
      // sin FK al item: igual listamos pallets
      $sqlRecv = "SELECT
                    pal.codigo AS pallet_codigo,
                    NULL AS producto_txt,
                    NULL AS lote,
                    NULL AS fecha_produccion,
                    NULL AS fecha_vencimiento,
                    COALESCE(pos.code_full, pos.pos_code, pos.pos_code_full, pos.code) AS ubicacion,
                    COALESCE(est.code, 'OK') AS estado,
                    COUNT(*) AS items_count,
                    COALESCE(SUM(l.uv_cajas),0) AS uv,
                    COALESCE(SUM(l.uc_unidades),0) AS uc
                  FROM $RECV_TABLE l
                  LEFT JOIN wh_pallet pal       ON pal.id = l.pallet_id
                  LEFT JOIN wh_posicion pos     ON pos.id = pal.posicion_id
                  LEFT JOIN wh_pallet_estado est ON est.id = pal.estado_id
                  WHERE l.packinglist_id = ?
                  GROUP BY pal.id
                  ORDER BY pal.codigo";
    }

    $st = $pdo->prepare($sqlRecv);
    $st->execute([$plId]);
    $received = $st->fetchAll() ?: [];
  }

  // ---------- PENDING: ítems con remanente ----------
  $pending = [];
  // Usamos las columnas expected_* / received_* del PL_ITEM (el SP/trigger las mantiene)
  $sqlPend = "SELECT
                i.id AS pl_item_id,
                i.producto_id,
                COALESCE(i.descripcion, i.sku_cliente) AS producto_txt,
                i.sku_cliente,
                i.lote_codigo AS lote,
                i.fecha_produccion,
                i.fecha_vencimiento,
                GREATEST(COALESCE(i.expected_uv,0) - COALESCE(i.received_uv,0), 0) AS rem_uv,
                GREATEST(COALESCE(i.expected_uc,0) - COALESCE(i.received_uc,0), 0) AS rem_uc
              FROM $T_PL_ITEM i
              WHERE i.packinglist_id = ?
                AND (GREATEST(COALESCE(i.expected_uv,0) - COALESCE(i.received_uv,0),0) > 0
                     OR GREATEST(COALESCE(i.expected_uc,0) - COALESCE(i.received_uc,0),0) > 0)
              ORDER BY i.id";
  $st = $pdo->prepare($sqlPend);
  $st->execute([$plId]);
  $pending = $st->fetchAll() ?: [];

  // ---------- Respuesta según view ----------
  if ($view === 'received') {
    json_ok($received);
    return;
  }
  if ($view === 'pending') {
    json_ok($pending);
    return;
  }
  // default: all
  json_ok(['received' => $received, 'pending' => $pending]);
}


/* ==================================================================
 * Endpoint: POST ?meta=recepcion_guardar
 * ------------------------------------------------------------------
 * Propósito:
 *    Registrar recepción a partir de un plan detallado (lista de pallets/ítems)
 *    o en modo AUTO (sin "rows", el SP hace el cálculo).
 * Parámetros (JSON):
 *    - pl_id (int) o pl_codigo (string) — uno de los dos es requerido
 *    - deposito_code (string)  — default "DEP1"
 *    - pos_code (string)       — posicion destino; si auto_ubicar y vacío → "CUARENTENA"
 *    - auto_ubicar (bool)      — si true y pos_code vacío, manda "CUARENTENA"
 *    - check_items (int)       — default 1 (validaciones del SP)
 *    - check_all_if_full (int) — default 1
 *    - rows (array opcional)   — cada fila: { pallet_codigo?, pos_code?, lote|lote_codigo, uv, uc, ... }
 *      * Normalizamos a: pos_code y lote_codigo
 * Respuesta:
 *    { mensaje, pl_codigo, summary, plan_mode, plan_count, pallet_codes }
 * ================================================================== */
function ep_recepcion_guardar(PDO $pdo, string $T_PL_HDR): void
{
  $T_PL_ITEM = $GLOBALS['T_PL_ITEM'] ?? null;
  if (!$T_PL_ITEM) json_err('No se detectó pl_packinglist_item en el entorno.', null, 500);

  $in             = getJsonInput();
  assertRequiredReceptionMeta($in);
  $plId           = (int)($in['pl_id'] ?? 0);
  $plCodigo       = trim((string)($in['pl_codigo'] ?? ''));
  $deposito       = trim((string)($in['deposito_code'] ?? 'DEP1'));
  $posCode        = trim((string)($in['pos_code'] ?? ''));
  // Forzar posición destino a la posición id=1 y trackear fallback
  $warnings = [];
  try {
    $forced = resolveDefaultPosCodeForId1($pdo);
    if ($forced !== '') {
      if ($forced === 'CUARENTENA') {
        $warnings[] = 'Posición id=1 no tiene code utilizable; usando CUARENTENA como destino.';
      }
      $posCode = $forced;
    }
  } catch (Throwable $e) { /* ignore and keep provided posCode */ }
  // Alinear deposito_code con la posición id=1
  try {
    $depCode = resolveDepositoCodeForPosId($pdo, 1);
    if ($depCode && $depCode !== $deposito) {
      $warnings[] = 'deposito_code sobreescrito para coincidir con la posición id=1: ' . $depCode;
      $deposito = $depCode;
    }
  } catch (Throwable $e) { /* ignore */ }
  $checkItems     = (int)($in['check_items'] ?? 1);
  $checkAllIfFull = (int)($in['check_all_if_full'] ?? 1);

  if ($plId <= 0 && $plCodigo === '') json_err('pl_id o pl_codigo requerido.');

  if ($plCodigo === '') {
    $st = $pdo->prepare("SELECT codigo FROM $T_PL_HDR WHERE id=? LIMIT 1");
    $st->execute([$plId]);
    $plCodigo = (string)$st->fetchColumn();
    if ($plCodigo === '' || $plCodigo === false) json_err('No se pudo resolver el código del PL.');
  }

  $rows       = is_array($in['rows'] ?? null) ? $in['rows'] : [];
  $autoUbicar = !empty($in['auto_ubicar']);

  // === MODO AUTO (sin rows) ===
  if (!$rows) {
    try {
      $opStart = date('Y-m-d H:i:s');
      $call = $pdo->prepare("CALL sp_pl_recibir_desde_pl(?, ?, ?, ?, ?, ?)");
      $call->execute([$plCodigo, $deposito, $posCode, null, $checkItems, $checkAllIfFull]);
      $summary = $call->fetchAll();
      while ($call->nextRowset()) { /* liberar */ }

      if ($plId <= 0) {
        $q = $pdo->prepare("SELECT id FROM $T_PL_HDR WHERE codigo = ? LIMIT 1");
        $q->execute([$plCodigo]);
        $plId = (int)($q->fetchColumn() ?: 0);
      }
      $rename = $plId > 0 ? renamePalletCodesForPL($pdo, (int)$plId) : ['updated' => 0, 'total' => 0, 'mapping' => []];

      // Crear/actualizar recepcion_hdr y asociar pl_rcv_link creados durante esta operación
      try {
        $uid = function_exists('current_user_id') ? current_user_id() : null;
        createOrUpdateRecepcionHdrAndAttachLinks($pdo, $plId, $in ?? [], $opStart, $uid);
      } catch (Throwable $e) { /* ignore */ }

      // After creating links/pallets, propagate cliente_id to pallets if the column exists
      try {
        $assign = assignClienteToPalletsFromPl($pdo, (int)$plId);
      } catch (Throwable $e) { /* ignore */ }

      json_ok([
        'mensaje'      => 'Recepción registrada (auto).',
        'pl_codigo'    => $plCodigo,
        'summary'      => $summary,
        'plan_mode'    => false,
        'plan_count'   => 0,
        'pallet_codes' => $rename,
        'warnings'     => $warnings,
      ]);
      return;
    } catch (Throwable $e) {
      json_err('Error guardando recepción1 (SP)', $e->getMessage(), 500);
    }
  }

  // === Con rows: construir plan con items[] por pallet ===
  $resolverPlItemId = function (PDO $pdo, string $T_PL_ITEM, int $plId, ?int $plItemId, ?string $loteCodigo): int {
    if ($plItemId && $plItemId > 0) return $plItemId;
    if ($loteCodigo === null || $loteCodigo === '') json_err('Falta pl_item_id o lote_codigo para resolver el ítem del PL.');
    $st = $pdo->prepare("SELECT id FROM $T_PL_ITEM WHERE packinglist_id = ? AND lote_codigo = ?");
    $st->execute([$plId, $loteCodigo]);
    $ids = $st->fetchAll(PDO::FETCH_COLUMN);
    if (!$ids)        json_err("No se encontró un ítem del PL con lote_codigo={$loteCodigo} en este PL.");
    if (count($ids)>1) json_err("Lote {$loteCodigo} es ambiguo en este PL. Envía pl_item_id explícito.");
    return (int)$ids[0];
  };

  $plan = [];

  // Pre-resolver códigos de posición para BUENAS (id=1) y DAÑADAS (id=2)
  $posCodeGood = resolvePosCodeForId($pdo, 1);
  $posCodeDam  = resolvePosCodeForId($pdo, 2);
  foreach ($rows as $r) {
    $loc = (string)($r['pos_code'] ?? $r['ubicacion'] ?? '');
    if ($autoUbicar && $loc === '') $loc = 'CUARENTENA';

  if (isset($r['items']) && is_array($r['items']) && count($r['items']) > 0) {
      $isDamaged = false;
      if (isset($r['meta_estado'])) {
        $es = strtoupper(trim((string)$r['meta_estado']));
        if ($es === 'DANADO' || $es === 'DAÑADO') $isDamaged = true;
      }
      $items = [];
      foreach ($r['items'] as $it) {
        $plItemId   = isset($it['pl_item_id']) ? (int)$it['pl_item_id'] : 0;
        $loteCodigo = (string)($it['lote_codigo'] ?? $it['lote'] ?? '');
        $uv         = array_key_exists('uv', $it) ? (($it['uv'] !== '' && $it['uv'] !== null) ? (int)$it['uv'] : null) : null;
        $uc         = array_key_exists('uc', $it) ? (($it['uc'] !== '' && $it['uc'] !== null) ? (int)$it['uc'] : null) : null;

        $plItemId = $resolverPlItemId($pdo, $T_PL_ITEM, $plId, $plItemId, $loteCodigo);
        if ($uv === null && $uc === null) json_err('Cada item debe incluir al menos uv o uc.');

        $items[] = [
          'pl_item_id'  => $plItemId,
          'lote_codigo' => $loteCodigo,
          'uv'          => $uv,
          'uc'          => $uc,
        ];
      }

      $plan[] = [
        'pallet_codigo' => $r['pallet_codigo'] ?? null,
        // Forzar posición: buenas -> id=1, dañadas -> id=2
        'pos_code'      => $isDamaged ? $posCodeDam : $posCodeGood,
        'items'         => $items,
      ];
      continue;
    }

    // Fila compacta → un item
    $loteCodigo = (string)($r['lote_codigo'] ?? $r['lote'] ?? '');
    $uv         = array_key_exists('uv', $r) ? (($r['uv'] !== '' && $r['uv'] !== null) ? (int)$r['uv'] : null) : null;
    $uc         = array_key_exists('uc', $r) ? (($r['uc'] !== '' && $r['uc'] !== null) ? (int)$r['uc'] : null) : null;
    $plItemId   = isset($r['pl_item_id']) ? (int)$r['pl_item_id'] : 0;

    $plItemId = $resolverPlItemId($pdo, $T_PL_ITEM, $plId, $plItemId, $loteCodigo);
    if ($uv === null && $uc === null) json_err('Cada fila debe incluir al menos uv o uc.');

    $isDamaged = false;
    if (isset($r['meta_estado'])) {
      $es = strtoupper(trim((string)$r['meta_estado']));
      if ($es === 'DANADO' || $es === 'DAÑADO') $isDamaged = true;
    }
    $plan[] = [
      'pallet_codigo' => $r['pallet_codigo'] ?? null,
      // Forzar posición: buenas -> id=1, dañadas -> id=2
      'pos_code'      => $isDamaged ? $posCodeDam : $posCodeGood,
      'items'         => [[
        'pl_item_id'  => $plItemId,
        'lote_codigo' => $loteCodigo,
        'uv'          => $uv,
        'uc'          => $uc,
      ]],
    ];
  }

  if (!$plan) json_err('Plan JSON vacío.');

  // ---- DIAGNÓSTICO: probamos ARRAY vs OBJETO {pallets:[...]} y elegimos el que MySQL “ve” ----
  $planJsonArray  = json_encode($plan, JSON_UNESCAPED_UNICODE);
  $planJsonObject = json_encode(['pallets' => $plan], JSON_UNESCAPED_UNICODE);

  $sqlDiagArray = "
    SELECT
      COUNT(*) AS pallets,
      SUM(JSON_LENGTH(jt.items)) AS items_total
    FROM JSON_TABLE(:j, '$[*]' COLUMNS(items JSON PATH '$.items' NULL ON EMPTY)) jt
  ";
  $sqlDiagObject = "
    SELECT
      COUNT(*) AS pallets,
      SUM(JSON_LENGTH(jt.items)) AS items_total
    FROM JSON_TABLE(JSON_EXTRACT(:j, '$.pallets'), '$[*]' COLUMNS(items JSON PATH '$.items' NULL ON EMPTY)) jt
  ";

  $itemsTotalArray  = 0;
  $itemsTotalObject = 0;

  try {
    $stA = $pdo->prepare($sqlDiagArray);
    $stA->execute([':j' => $planJsonArray]);
    $diagA = $stA->fetch();
    $itemsTotalArray = (int)($diagA['items_total'] ?? 0);
  } catch (Throwable $e) { /* ignorar diag */ }

  try {
    $stO = $pdo->prepare($sqlDiagObject);
    $stO->execute([':j' => $planJsonObject]);
    $diagO = $stO->fetch();
    $itemsTotalObject = (int)($diagO['items_total'] ?? 0);
  } catch (Throwable $e) { /* ignorar diag */ }

  // Elegimos el payload que MySQL entiende (prioriza el que tenga items)
  $planJson = null;
  if ($itemsTotalArray > 0) {
    $planJson = $planJsonArray;     // ARRAY puro
  } elseif ($itemsTotalObject > 0) {
    $planJson = $planJsonObject;    // OBJETO {pallets:[...]}
  } else {
    // Ninguna forma fue vista → devolvemos diagnóstico y dump
    @file_put_contents(__DIR__ . '/recep_plan_debug.json', $planJsonArray);
    json_err('Diagnóstico: MySQL no detecta items ni en ARRAY ni en OBJETO.', [
      'diag_array'  => $diagA ?? null,
      'diag_object' => $diagO ?? null,
      'sample_plan' => json_decode($planJsonArray, true),
    ]);
  }

  // ---- Llamado al SP con el payload que sí funciona en tu MySQL ----
  try {
    $opStart = date('Y-m-d H:i:s');
    $call = $pdo->prepare("CALL sp_pl_recibir_desde_pl(?, ?, ?, ?, ?, ?)");
    $call->execute([$plCodigo, $deposito, $posCode, $planJson, $checkItems, $checkAllIfFull]);
    $summary = $call->fetchAll();
    while ($call->nextRowset()) { /* liberar */ }

    if ($plId <= 0) {
      $q = $pdo->prepare("SELECT id FROM $T_PL_HDR WHERE codigo = ? LIMIT 1");
      $q->execute([$plCodigo]);
      $plId = (int)($q->fetchColumn() ?: 0);
    }
    $rename = $plId > 0 ? renamePalletCodesForPL($pdo, (int)$plId) : ['updated' => 0, 'total' => 0, 'mapping' => []];

    // Crear recepcion_hdr y asociar links creados
    try {
      $uid = function_exists('current_user_id') ? current_user_id() : null;
      createOrUpdateRecepcionHdrAndAttachLinks($pdo, $plId, $in ?? [], $opStart, $uid);
    } catch (Throwable $e) { /* ignore */ }

    // Persistir metadata opcional de ingreso en pl_ingreso si existe (backward compat)
    try {
      updatePlIngresoFromPayload($pdo, $plId, $in ?? []);
    } catch (Throwable $e) { /* ignore */ }

    // After creating links/pallets, propagate cliente_id to pallets if the column exists
    try {
      $assign = assignClienteToPalletsFromPl($pdo, (int)$plId);
    } catch (Throwable $e) { /* ignore */ }

    json_ok([
      'mensaje'      => 'Recepción registrada (plan detallado).',
      'pl_codigo'    => $plCodigo,
      'summary'      => $summary,
      'plan_mode'    => true,
      'plan_count'   => count($plan),
      'pallet_codes' => $rename,
      'warnings'     => $warnings,
    ]);
  } catch (Throwable $e) {
    json_err('Error guardando recepción2 (SP)', $e->getMessage(), 500);
  }
}



/* ==================================================================
 * Endpoint: POST ?meta=recepcion_guardar_item
 * ------------------------------------------------------------------
 * Propósito:
 *    Autogenerar un plan de pallets para **un ítem** del PL y registrar
 *    la recepción con el SP (reglas A/B).
 *
 * Reglas:
 *  A) Si existe para_producto_paletizado.cajas_por_pallet (>0):
 *     - dividir expected_uv en pallets de ese tamaño.
 *     - el resto (si hay) y las UC sueltas se asignan al último pallet.
 *  B) Si NO hay CPP:
 *     - usar filas de pl_import_row del mismo import_batch_id y del mismo producto/lote (desde el JSON libre).
 *     - N pallets = #rows; distribuir UV uniforme y el resto al último; UC = uv * (uc_por_caja modo) o proporcional.
 *
 * Parámetros (JSON):
 *    - pl_id (int, req)
 *    - pl_item_id (int, req)
 *    - auto_ubicar (bool) (opcional)
 *    - pos_code (string) (opcional; si auto_ubicar y vacío → "CUARENTENA")
 *    - (internos) deposito="DEP1", check_* = 1
 *
 * Respuesta:
 *    { ok, mensaje, pl_id, pl_codigo, pl_item_id, plan_count, plan[], pallet_codes, summary }
 * ================================================================== */
function ep_recepcion_guardar_item(PDO $pdo, string $T_PL_HDR, string $T_PL_ITEM, ?string $T_IMPORT_R, ?string $T_PALETIZ, ?string $RECV_TABLE): void
{
  // -------- Input --------
  $in = getJsonInput();
  assertRequiredReceptionMeta($in);
  $plId = (int) ($in['pl_id'] ?? 0);
  $plItemId = (int) ($in['pl_item_id'] ?? 0);
  $autoUbicar = !empty($in['auto_ubicar']);
  $posCode = trim((string) ($in['pos_code'] ?? ''));
  // Forzar posición destino a la posición id=1 y trackear fallback
  $warnings = [];
  try {
    $forced = resolveDefaultPosCodeForId1($pdo);
    if ($forced !== '') {
      if ($forced === 'CUARENTENA') {
        $warnings[] = 'Posición id=1 no tiene code utilizable; usando CUARENTENA como destino.';
      }
      $posCode = $forced;
    }
  } catch (Throwable $e) { /* ignore */ }
  // Alinear deposito_code con la posición id=1
  $deposito = 'DEP1';
  try {
    $depCode = resolveDepositoCodeForPosId($pdo, 1);
    if ($depCode) {
      $deposito = $depCode;
      $warnings[] = 'deposito_code sobreescrito para coincidir con la posición id=1: ' . $depCode;
    }
  } catch (Throwable $e) { /* ignore */ }
  $checkItems = 1;
  $checkAllIfFull = 1;

  if ($plId <= 0 || $plItemId <= 0)
    json_err('pl_id y pl_item_id requeridos.');

  // -------- Header: código + import_batch_id --------
  $st = $pdo->prepare("SELECT codigo, import_batch_id FROM $T_PL_HDR WHERE id = ? LIMIT 1");
  $st->execute([$plId]);
  $hdr = $st->fetch();
  if (!$hdr)
    json_err('PL no encontrado.');
  $plCodigo = (string) $hdr['codigo'];
  $importBatchId = (int) ($hdr['import_batch_id'] ?? 0);

  // -------- Ítem del PL (usamos expected_* como base a recibir) --------
  $sqlItem = "SELECT i.id, i.packinglist_id, i.producto_id,
                     COALESCE(i.descripcion, i.sku_cliente) AS producto_txt,
                     i.lote_codigo, i.fecha_produccion, i.fecha_vencimiento,
                     COALESCE(i.expected_uv,0) AS expected_uv,
                     COALESCE(i.expected_uc,0) AS expected_uc
                FROM $T_PL_ITEM i
               WHERE i.id = ? AND i.packinglist_id = ?
               LIMIT 1";
  $st = $pdo->prepare($sqlItem);
  $st->execute([$plItemId, $plId]);
  $it = $st->fetch();
  if (!$it)
    json_err('Ítem del PL no encontrado o no pertenece al PL.');

  $productoId = (int) ($it['producto_id'] ?? 0);
  $uvTot = max(0, (int) $it['expected_uv']);
  $ucTot = max(0, (int) $it['expected_uc']);

  if ($uvTot <= 0 && $ucTot <= 0)
    json_err('El ítem no tiene cantidades esperadas (UV/UC) para recibir.');

  // -------- Helper: parsear métricas desde JSON libre del Excel --------
  $parseRowMetrics = function (?string $json): array {
    $uv = null;
    $ucx = null;
    $sueltas = 0;
    if (!$json)
      return ['uv' => null, 'uc_por_caja' => null, 'uc_sueltas' => 0, 'lote' => null];
    $row = json_decode($json, true);
    if (!is_array($row))
      return ['uv' => null, 'uc_por_caja' => null, 'uc_sueltas' => 0, 'lote' => null];

    // UV (cajas)
    foreach (['uv_cajas', 'cajas', 'cantidad_cajas', 'uv', 'unidades_venta', 'cjas'] as $k) {
      if (isset($row[$k]) && is_numeric($row[$k])) {
        $uv = (int) $row[$k];
        break;
      }
    }
    // UC por caja
    foreach (['uc_por_caja', 'unidades_por_caja', 'u_x_caja', 'units_per_case', 'ucxuv'] as $k) {
      if (isset($row[$k]) && is_numeric($row[$k])) {
        $ucx = (int) $row[$k];
        break;
      }
    }
    // UC sueltas
    foreach (['uc_sueltas', 'sueltas', 'unidades_sueltas', 'units_loose'] as $k) {
      if (isset($row[$k]) && is_numeric($row[$k])) {
        $sueltas = (int) $row[$k];
        break;
      }
    }
    // Lote (puede venir con varios nombres)
    $lote = null;
    foreach (['lote', 'lote_codigo', 'lote_code'] as $k) {
      if (isset($row[$k]) && $row[$k] !== '') {
        $lote = (string) $row[$k];
        break;
      }
    }

    return ['uv' => $uv, 'uc_por_caja' => $ucx, 'uc_sueltas' => $sueltas, 'lote' => $lote];
  };

  // -------- Fuente Excel (pl_import_row) para scenario B y deducir uc_por_caja/sueltas --------
  $NrowsExcel = 0;
  $ucPorCaja = null;
  $ucSueltasTotal = 0;

  if ($T_IMPORT_R && $importBatchId > 0) {
    $conds = ["`batch_id` = ?"];
    $params = [$importBatchId];

    if (columnExists($pdo, $T_IMPORT_R, 'producto_id') && $productoId > 0) {
      $conds[] = "`producto_id` = ?";
      $params[] = $productoId;
    } else {
      $matched = false;
      foreach (['sku_cliente', 'sku', 'producto_codigo', 'producto'] as $pc) {
        if (columnExists($pdo, $T_IMPORT_R, $pc)) {
          $conds[] = "`$pc` = ?";
          $params[] = (string) $it['producto_txt'];
          $matched = true;
          break;
        }
      }
      if (!$matched && columnExists($pdo, $T_IMPORT_R, 'producto_txt')) {
        $conds[] = "`producto_txt` = ?";
        $params[] = (string) $it['producto_txt'];
      }
    }

    $colRow = columnExists($pdo, $T_IMPORT_R, 'row') ? "`row` AS row_json" : "NULL AS row_json";

    $sqlIR = "SELECT `id`, $colRow
                FROM `$T_IMPORT_R`
               WHERE " . implode(' AND ', $conds) . "
            ORDER BY `id`";
    $st = $pdo->prepare($sqlIR);
    $st->execute($params);
    $rowsIR = $st->fetchAll() ?: [];

    // Filtrado por LOTE (porque en Excel el lote está dentro del JSON)
    $lotPL = (string) ($it['lote_codigo'] ?? '');
    $lotPLn = trim(mb_strtolower($lotPL));

    $rowsIR = array_values(array_filter($rowsIR, function ($r) use ($parseRowMetrics, $lotPLn) {
      $m = $parseRowMetrics($r['row_json'] ?? null);
      $lr = trim(mb_strtolower((string) ($m['lote'] ?? '')));
      return ($lotPLn === '' || $lr === $lotPLn);
    }));

    // Deducciones de uc_por_caja y sueltas
    $candidatesUcx = [];
    $ucSueltasTotal = 0;
    foreach ($rowsIR as $r) {
      $m = $parseRowMetrics($r['row_json'] ?? null);
      if (is_numeric($m['uc_por_caja']) && (int) $m['uc_por_caja'] > 0)
        $candidatesUcx[] = (int) $m['uc_por_caja'];
      if (is_numeric($m['uc_sueltas']) && (int) $m['uc_sueltas'] > 0)
        $ucSueltasTotal += (int) $m['uc_sueltas'];
    }
    $ucPorCaja = mode_numeric($candidatesUcx);
    $NrowsExcel = count($rowsIR);
  }

  if ((!$ucPorCaja || $ucPorCaja <= 0) && $uvTot > 0 && $ucTot > 0) {
    $ucPorCaja = max(1, (int) round($ucTot / $uvTot));
  }

  // -------- ¿Existe CPP en para_producto_paletizado? --------
  $cpp = 0;
  if ($T_PALETIZ && $productoId > 0 && columnExists($pdo, $T_PALETIZ, 'cajas_por_pallet')) {
    $st = $pdo->prepare("SELECT cajas_por_pallet
                           FROM $T_PALETIZ
                          WHERE producto_id = ?
                       ORDER BY id DESC
                          LIMIT 1");
    $st->execute([$productoId]);
    $cpp = (int) ($st->fetchColumn() ?: 0);
  }

  // -------- Construcción de plan de pallets --------
  $plan = [];
  $autoUb = $autoUbicar ? true : false;
  $pickLast = false;

  if ($cpp > 0) {
    // === A) Con CPP ===
    $N = (int) floor($uvTot / $cpp) + (($uvTot % $cpp) > 0 ? 1 : 0);
    $N = max(1, $N);
    $rem = $uvTot % $cpp;

    for ($k = 1; $k <= $N; $k++) {
      $uv = ($k === $N && $rem > 0) ? $rem : min($cpp, max(0, $uvTot - ($k - 1) * $cpp));
      if ($N === 1 && $uv === 0)
        $uv = $uvTot;

      // UC por pallet
      $uc = ($ucPorCaja && $uv > 0) ? $uv * $ucPorCaja
        : ($uvTot > 0 ? (int) round(($uv / $uvTot) * $ucTot) : $ucTot);

      // Sueltas y resto al último pallet
      if ($k === $N && $ucSueltasTotal > 0) {
        $uc += $ucSueltasTotal;
        $pickLast = true;
      }
      if ($k === $N && $rem > 0) {
        $pickLast = true;
      }

      $plan[] = makePlanPalletForItem($it, $plItemId, (int) $uv, (int) $uc, $autoUb, $posCode, $k === $N ? $pickLast : false);
    }
  } else {
    // === B) Sin CPP ===
    $N = max(1, $NrowsExcel); // si no hubo filas importadas, al menos 1 pallet
    $base = (int) floor($uvTot / $N);
    $rest = $uvTot % $N;

    for ($k = 1; $k <= $N; $k++) {
      $uv = $base;
      if ($k === $N)
        $uv += $rest;

      $uc = ($ucPorCaja && $uv > 0) ? $uv * $ucPorCaja
        : ($uvTot > 0 ? (int) round(($uv / $uvTot) * $ucTot) : $ucTot);

      $pick = false;
      if ($k === $N) {
        $uc += $ucSueltasTotal;
        if ($rest > 0 || $ucSueltasTotal > 0)
          $pick = true;
      }

      $plan[] = makePlanPalletForItem($it, $plItemId, (int) $uv, (int) $uc, $autoUb, $posCode, $pick);
    }
  }

  if (!$plan)
    json_err('Plan JSON: pallet sin items.');

  $planJson = json_encode($plan, JSON_UNESCAPED_UNICODE);

  // -------- Guardar con SP --------
  try {
    $call = $pdo->prepare("CALL sp_pl_recibir_desde_pl(?, ?, ?, ?, ?, ?)");
    $call->execute([$plCodigo, $deposito, $posCode, $planJson, $checkItems, $checkAllIfFull]);
    $summary = $call->fetchAll();
    while ($call->nextRowset()) { /* liberar */
    }
  } catch (Throwable $e) {
    json_err('Error guardando recepcion por item (SP)', $e->getMessage(), 500);
  }

  // -------- Renombrar pallets creados a formato PAL-<PL>-NNN --------
  $rename = renamePalletCodesForPL($pdo, (int) $plId);

  json_ok([
    'ok' => true,
    'mensaje' => 'Ítem recibido (auto).',
    'pl_id' => $plId,
    'pl_codigo' => $plCodigo,
    'pl_item_id' => $plItemId,
    'plan_count' => count($plan),
    'plan' => $plan,     // útil para debug; quitar en producción si no se desea exponer
    'pallet_codes' => $rename,
    'summary' => $summary,
    'warnings' => $warnings,
  ]);
}

/* ------------------------------------------------------------------
 * Router
 * ------------------------------------------------------------------ */
$meta = $_GET['meta'] ?? ($_POST['meta'] ?? '');
try {
  switch ($meta) {
    case 'pl_pendientes':
      ep_pl_pendientes($pdo, $T_PL_HDR, $T_PL_ITEM, $T_CLIENTE, $RECV_TABLE);
      break;

    case 'pl_resumen':
      $plId = (int) ($_GET['pl_id'] ?? 0);
      if ($plId <= 0)
        json_err('pl_id requerido');
      ep_pl_resumen($pdo, $plId, $T_PL_HDR, $T_PL_ITEM, $T_CLIENTE, $RECV_TABLE);
      break;

    case 'recepcion_list':
      $plId = (int) ($_GET['pl_id'] ?? 0);
      if ($plId <= 0)
        json_err('pl_id requerido');
      ep_recepcion_list($pdo, $plId, $RECV_TABLE);
      break;

    case 'recepcion_guardar':
      ep_recepcion_guardar($pdo, $T_PL_HDR);
      break;

    case 'recepcion_guardar_hdr':
      // Persist only header metadata (pl_ingreso / recepcion_hdr) without calling the SP
      $in = getJsonInput();
      $plId = (int)($in['pl_id'] ?? 0);
      if ($plId <= 0) json_err('pl_id requerido');
      assertRequiredReceptionMeta($in);

      // Persist into pl_ingreso (create or update) using existing helper
      try {
        updatePlIngresoFromPayload($pdo, $plId, $in ?? []);
      } catch (Throwable $e) { /* ignore */ }

      // Create/Update recepcion_hdr (no links to attach because SP not run)
      try {
        $uid = function_exists('current_user_id') ? current_user_id() : null;
        $rid = createOrUpdateRecepcionHdrAndAttachLinks($pdo, $plId, $in ?? [], date('Y-m-d H:i:s'), $uid);
      } catch (Throwable $e) { $rid = null; }

      // Return refreshed resumen data
      try {
        ob_start();
        ep_pl_resumen($pdo, $plId, $T_PL_HDR, $T_PL_ITEM, $T_CLIENTE, $RECV_TABLE);
        $out = ob_get_clean();
        // ep_pl_resumen already echoes JSON; wrap success
        // If ep_pl_resumen emitted json_ok, it has already printed; simply exit
        // But to keep consistent, return the parsed resumen
        $st = $pdo->prepare("SELECT id FROM $T_PL_HDR WHERE id = ? LIMIT 1");
        $st->execute([$plId]);
        json_ok(['mensaje' => 'Cabecera guardada.', 'pl_id' => $plId, 'recepcion_id' => $rid]);
      } catch (Throwable $e) {
        json_ok(['mensaje' => 'Cabecera guardada (parcial).', 'pl_id' => $plId, 'recepcion_id' => $rid]);
      }
      break;

    case 'recepcion_guardar_item':
      ep_recepcion_guardar_item($pdo, $T_PL_HDR, $T_PL_ITEM, $T_IMPORT_R, $T_PALETIZ, $RECV_TABLE);
      break;

    case 'pos_default_codes':
      try {
        $goodPos = resolvePosCodeForId($pdo, 1);
        $damPos  = resolvePosCodeForId($pdo, 2);
        $goodDep = resolveDepositoCodeForPosId($pdo, 1);
        $damDep  = resolveDepositoCodeForPosId($pdo, 2);
        json_ok([
          'good_pos_code' => $goodPos,
          'damaged_pos_code' => $damPos,
          'good_deposito_code' => $goodDep,
          'damaged_deposito_code' => $damDep,
        ]);
      } catch (Throwable $e) {
        json_err('No se pudieron resolver códigos por defecto', $e->getMessage(), 500);
      }
      break;

    default:
      json_err('meta inválido o no especificado', [
        'allowed' => ['pl_pendientes', 'pl_resumen', 'recepcion_list', 'recepcion_guardar', 'recepcion_guardar_item']
      ], 404);
  }
} catch (Throwable $e) {
  json_err('Excepción no controlada', $e->getMessage(), 500);
}
