<?php
declare(strict_types=1);

header('Content-Type: application/pdf');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

$autoload = $ROOT . '/vendor/autoload.php';
if (!is_file($autoload)) {
  http_response_code(500);
  echo 'Falta vendor/autoload.php. Instala dependencias con Composer.';
  exit;
}
require_once $autoload;

if (!class_exists(\Mpdf\Mpdf::class)) {
  http_response_code(500);
  echo 'No está instalado mpdf/mpdf. Ejecuta: composer require mpdf/mpdf';
  exit;
}

function nf($n, int $dec = 0): string {
  if ($n === null || $n === '') {
    return $dec > 0 ? number_format(0, $dec, ',', '.') : '0';
  }
  return number_format((float)$n, $dec, ',', '.');
}

function hasTbl(PDO $pdo, string $table): bool {
  static $cache = [];
  if (array_key_exists($table, $cache)) {
    return $cache[$table];
  }
  $st = $pdo->prepare(
    'SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?'
  );
  $st->execute([$table]);
  return $cache[$table] = ((int)$st->fetchColumn() > 0);
}

function hasCol(PDO $pdo, string $table, string $column): bool {
  static $cache = [];
  $key = $table . '|' . $column;
  if (array_key_exists($key, $cache)) {
    return $cache[$key];
  }
  $st = $pdo->prepare(
    'SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?'
  );
  $st->execute([$table, $column]);
  return $cache[$key] = ((int)$st->fetchColumn() > 0);
}

function pickTable(PDO $pdo, array $candidates): ?string {
  foreach ($candidates as $tbl) {
    if ($tbl && hasTbl($pdo, $tbl)) {
      return $tbl;
    }
  }
  return null;
}

function pickColumn(PDO $pdo, string $table, array $candidates): ?string {
  foreach ($candidates as $col) {
    if ($col && hasCol($pdo, $table, $col)) {
      return $col;
    }
  }
  return null;
}

try {
  $pdo = get_pdo();
  $plId = (int)($_GET['pl_id'] ?? 0);
  if ($plId <= 0) {
    http_response_code(400);
    echo 'pl_id es requerido';
    exit;
  }

  $tblPL = 'pl_packinglist';
  if (!hasTbl($pdo, $tblPL)) {
    throw new RuntimeException('Falta tabla pl_packinglist en la base de datos.');
  }

  $codeCol = pickColumn($pdo, $tblPL, ['codigo', 'code', 'pl_codigo']);
  $dateCol = pickColumn($pdo, $tblPL, ['fecha', 'fecha_pl', 'created_at']);
  $clientTxtCol = pickColumn($pdo, $tblPL, ['cliente_ref', 'cliente', 'cliente_nombre']);
  $clientIdCol = pickColumn($pdo, $tblPL, ['cliente_id']);
  $movilCol = pickColumn($pdo, $tblPL, ['movil', 'movil_ref']);
  $choferCol = pickColumn($pdo, $tblPL, ['chofer', 'chofer_ref']);

  $select = ['pl.id AS pl_id'];
  if ($codeCol) {
    $select[] = "pl.`{$codeCol}` AS pl_codigo";
  }
  if ($dateCol) {
    $select[] = "pl.`{$dateCol}` AS pl_fecha";
  }
  if ($clientTxtCol) {
    $select[] = "pl.`{$clientTxtCol}` AS cliente_ref";
  }
  if ($movilCol) {
    $select[] = "pl.`{$movilCol}` AS movil_ref";
  }
  if ($choferCol) {
    $select[] = "pl.`{$choferCol}` AS chofer_ref";
  }

  $joins = [];
  if ($clientIdCol && hasTbl($pdo, 'para_clientes')) {
    $clientNameCol = pickColumn($pdo, 'para_clientes', ['razon_social', 'nombre', 'descripcion']);
    if ($clientNameCol) {
      $joins[] = "LEFT JOIN para_clientes cli ON cli.id = pl.`{$clientIdCol}`";
      $select[] = "cli.`{$clientNameCol}` AS cliente_nombre";
    }
  }

  $sqlHdr = 'SELECT ' . implode(', ', $select) . " FROM {$tblPL} pl " . implode(' ', $joins) . ' WHERE pl.id = ? LIMIT 1';
  $sth = $pdo->prepare($sqlHdr);
  $sth->execute([$plId]);
  $plHdr = $sth->fetch(PDO::FETCH_ASSOC);

  if (!$plHdr) {
    http_response_code(404);
    echo 'Packing List no encontrado';
    exit;
  }

  $cabecera = [
    'codigo'   => $plHdr['pl_codigo'] ?? ('PL #' . $plId),
    'fecha'    => $plHdr['pl_fecha'] ?? '',
    'cliente'  => $plHdr['cliente_nombre'] ?? $plHdr['cliente_ref'] ?? '-',
    'movil'    => $plHdr['movil_ref'] ?? '-',
    'chofer'   => $plHdr['chofer_ref'] ?? '-',
  ];

  $tblIngreso = pickTable($pdo, ['pl_ingreso']);
  if ($tblIngreso) {
    $ingCols = [];
    foreach ([
      'fecha_ingreso', 'llegada_at', 'descarga_inicio_at', 'descarga_fin_at',
      'operarios_cant', 'observacion', 'doc_tipo', 'doc_numero', 'doc_fecha',
      'movil_id', 'chofer_id', 'deposito_id', 'deposito_code', 'hora_arribo',
      'hora_inicio', 'hora_fin'
    ] as $col) {
      if (hasCol($pdo, $tblIngreso, $col)) {
        $ingCols[] = "i.`{$col}`";
      }
    }
    if ($ingCols) {
      $sqlIng = 'SELECT ' . implode(', ', $ingCols) . " FROM {$tblIngreso} i WHERE i.packinglist_id = ? ORDER BY i.id DESC LIMIT 1";
      $sti = $pdo->prepare($sqlIng);
      $sti->execute([$plId]);
      $ing = $sti->fetch(PDO::FETCH_ASSOC) ?: [];
      if ($ing) {
        $cabecera['fecha_ingreso'] = $ing['fecha_ingreso'] ?? '';
        $cabecera['hora_arribo'] = $ing['hora_arribo'] ?? $ing['llegada_at'] ?? '';
        $cabecera['hora_inicio'] = $ing['hora_inicio'] ?? $ing['descarga_inicio_at'] ?? '';
        $cabecera['hora_fin'] = $ing['hora_fin'] ?? $ing['descarga_fin_at'] ?? '';
        $cabecera['operarios'] = $ing['operarios_cant'] ?? '';
        $cabecera['observacion'] = $ing['observacion'] ?? '';
        $cabecera['doc_tipo'] = $ing['doc_tipo'] ?? '';
        $cabecera['doc_numero'] = $ing['doc_numero'] ?? '';
        $cabecera['doc_fecha'] = $ing['doc_fecha'] ?? '';
        if (!empty($ing['movil_id']) && hasTbl($pdo, 'para_moviles')) {
          $movilNameCol = pickColumn($pdo, 'para_moviles', ['chapa', 'codigo', 'nombre']);
          if ($movilNameCol) {
            $stmv = $pdo->prepare("SELECT `{$movilNameCol}` FROM para_moviles WHERE id = ? LIMIT 1");
            $stmv->execute([(int)$ing['movil_id']]);
            $mv = $stmv->fetchColumn();
            if ($mv) {
              $cabecera['movil'] = $mv;
            }
          }
        }
        if (!empty($ing['chofer_id']) && hasTbl($pdo, 'para_choferes')) {
          $choferNameCol = pickColumn($pdo, 'para_choferes', ['nombre', 'descripcion']);
          if ($choferNameCol) {
            $stch = $pdo->prepare("SELECT `{$choferNameCol}` FROM para_choferes WHERE id = ? LIMIT 1");
            $stch->execute([(int)$ing['chofer_id']]);
            $ch = $stch->fetchColumn();
            if ($ch) {
              $cabecera['chofer'] = $ch;
            }
          }
        }
        if (!empty($ing['deposito_id']) && hasTbl($pdo, 'wh_deposito')) {
          $depNameCol = pickColumn($pdo, 'wh_deposito', ['nombre', 'codigo', 'code']);
          if ($depNameCol) {
            $stdep = $pdo->prepare("SELECT `{$depNameCol}` FROM wh_deposito WHERE id = ? LIMIT 1");
            $stdep->execute([(int)$ing['deposito_id']]);
            $dep = $stdep->fetchColumn();
            if ($dep) {
              $cabecera['deposito'] = $dep;
            }
          }
        } elseif (!empty($ing['deposito_code'])) {
          $cabecera['deposito'] = $ing['deposito_code'];
        }
      }
    }
  }

  $cabecera['deposito'] = $cabecera['deposito'] ?? '-';

  $tblRecv = pickTable($pdo, ['pl_rcv_link']);
  if (!$tblRecv || !hasTbl($pdo, $tblRecv)) {
    throw new RuntimeException('No se encontró la tabla de recepciones (pl_rcv_link).');
  }

  $uvCol = pickColumn($pdo, $tblRecv, ['uv_cajas', 'uv', 'cantidad_uv']);
  $ucCol = pickColumn($pdo, $tblRecv, ['uc_unidades', 'uc', 'cantidad_uc']);
  $pesoCol = pickColumn($pdo, $tblRecv, ['peso_kg', 'peso', 'peso_total']);
  $loteCol = pickColumn($pdo, $tblRecv, ['lote_codigo', 'lote']);
  $fProdCol = pickColumn($pdo, $tblRecv, ['fecha_produccion']);
  $fVencCol = pickColumn($pdo, $tblRecv, ['fecha_vencimiento']);
  $estadoCol = pickColumn($pdo, $tblRecv, ['estado']);
  $itemFkCol = pickColumn($pdo, $tblRecv, ['pl_item_id', 'packinglist_item_id', 'item_id']);

  $tblPallet = pickTable($pdo, ['wh_pallet', 'wh_pallets']);
  $tblPos = pickTable($pdo, ['wh_posicion', 'wh_positions']);
  $tblEstado = pickTable($pdo, ['wh_pallet_estado']);
  $tblItem = 'pl_packinglist_item';
  $tblProd = pickTable($pdo, ['para_productos']);

  $joinPallet = '';
  $joinPos = '';
  $joinEstado = '';
  $joinItem = '';
  $joinProd = '';

  $palletCodeExpr = 'CONCAT("PAL-", l.pallet_id)';
  if ($tblPallet) {
    $joinPallet = "LEFT JOIN {$tblPallet} p ON p.id = l.pallet_id";
    $palCodeCol = pickColumn($pdo, $tblPallet, ['codigo', 'code']);
    if ($palCodeCol) {
      $palletCodeExpr = "p.`{$palCodeCol}`";
    }
    if ($tblPos && hasCol($pdo, $tblPallet, 'posicion_id')) {
      $joinPos = "LEFT JOIN {$tblPos} pos ON pos.id = p.posicion_id";
    }
    if ($tblEstado && hasCol($pdo, $tblPallet, 'estado_id')) {
      $joinEstado = "LEFT JOIN {$tblEstado} est ON est.id = p.estado_id";
    }
  }

  $posExprOptions = [];
  if ($tblPos) {
    foreach (['code_full', 'pos_code', 'pos_code_full', 'codigo', 'code', 'nombre', 'title'] as $c) {
      if (hasCol($pdo, $tblPos, $c)) {
        $posExprOptions[] = "pos.`{$c}`";
      }
    }
  }
  $posExpr = $posExprOptions ? 'COALESCE(' . implode(', ', $posExprOptions) . ')' : 'NULL';

  if ($itemFkCol && hasTbl($pdo, $tblItem)) {
    $joinItem = "LEFT JOIN {$tblItem} i ON i.id = l.`{$itemFkCol}`";
    if ($tblProd && hasCol($pdo, $tblItem, 'producto_id')) {
      $joinProd = "LEFT JOIN {$tblProd} pr ON pr.id = i.producto_id";
    }
  }

  $estadoExpr = 'NULL';
  if ($estadoCol) {
    $estadoExpr = "l.`{$estadoCol}`";
  } elseif ($tblEstado && hasCol($pdo, $tblEstado, 'code')) {
    $estadoExpr = 'est.`code`';
  }

  $uvExpr = $uvCol ? "l.`{$uvCol}`" : '0';
  $ucExpr = $ucCol ? "l.`{$ucCol}`" : '0';
  $pesoExpr = $pesoCol ? "l.`{$pesoCol}`" : 'NULL';
  $loteExpr = $loteCol ? "l.`{$loteCol}`" : ($itemFkCol ? 'i.lote_codigo' : 'NULL');
  $fProdExpr = $fProdCol ? "l.`{$fProdCol}`" : ($itemFkCol ? 'i.fecha_produccion' : 'NULL');
  $fVencExpr = $fVencCol ? "l.`{$fVencCol}`" : ($itemFkCol ? 'i.fecha_vencimiento' : 'NULL');

  $productoExprParts = [];
  if ($itemFkCol) {
    foreach (['descripcion', 'producto_txt', 'sku_cliente', 'sku'] as $cand) {
      if (hasCol($pdo, $tblItem, $cand)) {
        $productoExprParts[] = "i.`{$cand}`";
      }
    }
  }
  if ($tblProd) {
    foreach (['denominacion', 'nombre', 'descripcion', 'titulo'] as $cand) {
      if (hasCol($pdo, $tblProd, $cand)) {
        $productoExprParts[] = "pr.`{$cand}`";
      }
    }
  }
  if (!$productoExprParts) {
    $productoExprParts[] = 'NULL';
  }
  $productoExpr = 'COALESCE(' . implode(', ', $productoExprParts) . ')';

  $sqlPallets = "SELECT
      l.pallet_id AS pallet_id,
      COALESCE({$palletCodeExpr}, CONCAT('PAL-', l.pallet_id)) AS pallet_codigo,
      MAX({$posExpr}) AS posicion,
      COUNT(*) AS items_count,
      COALESCE(SUM({$uvExpr}), 0) AS uv_total,
      COALESCE(SUM({$ucExpr}), 0) AS uc_total
    FROM {$tblRecv} l
    {$joinPallet}
    {$joinPos}
    {$joinEstado}
    {$joinItem}
    {$joinProd}
    WHERE l.packinglist_id = ?
    GROUP BY l.pallet_id, COALESCE({$palletCodeExpr}, CONCAT('PAL-', l.pallet_id))
    ORDER BY COALESCE({$palletCodeExpr}, CONCAT('PAL-', l.pallet_id))";

  $stp = $pdo->prepare($sqlPallets);
  $stp->execute([$plId]);
  $pallets = $stp->fetchAll(PDO::FETCH_ASSOC) ?: [];

  $sqlItems = "SELECT
      COALESCE({$palletCodeExpr}, CONCAT('PAL-', l.pallet_id)) AS pallet_codigo,
      {$productoExpr} AS producto,
      {$loteExpr} AS lote,
      {$fProdExpr} AS fecha_produccion,
      {$fVencExpr} AS fecha_vencimiento,
      {$uvExpr} AS uv_cajas,
      {$ucExpr} AS uc_unidades,
      {$pesoExpr} AS peso,
      {$posExpr} AS posicion,
      {$estadoExpr} AS estado
    FROM {$tblRecv} l
    {$joinPallet}
    {$joinPos}
    {$joinEstado}
    {$joinItem}
    {$joinProd}
    WHERE l.packinglist_id = ?
    ORDER BY COALESCE({$palletCodeExpr}, CONCAT('PAL-', l.pallet_id)), producto, lote";

  $sti = $pdo->prepare($sqlItems);
  $sti->execute([$plId]);
  $items = $sti->fetchAll(PDO::FETCH_ASSOC) ?: [];

  $totPallets = count($pallets);
  $totUV = array_reduce($pallets, static function ($carry, $row) {
    return $carry + (int)($row['uv_total'] ?? 0);
  }, 0);
  $totUC = array_reduce($pallets, static function ($carry, $row) {
    return $carry + (int)($row['uc_total'] ?? 0);
  }, 0);

  $mpdf = new \Mpdf\Mpdf([
    'format' => 'A4',
    'margin_left' => 10,
    'margin_right' => 10,
    'margin_top' => 12,
    'margin_bottom' => 12,
  ]);

  $title = 'Recepción de ingreso - ' . ($cabecera['codigo'] ?? ('PL #' . $plId));
  $styles = '<style>
    body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 11px; }
    h1 { font-size: 18px; margin: 0 0 8px 0; }
    .muted { color: #666; }
    .section { margin-top: 10px; }
    .grid { width: 100%; border-collapse: collapse; margin-top: 6px; }
    .grid th, .grid td { border: 0.6px solid #999; padding: 4px 6px; }
    .grid th { background: #f3f3f3; }
    .kv { display: flex; flex-wrap: wrap; gap: 12px; }
    .kv div { min-width: 180px; }
    .right { text-align: right; }
    .small { font-size: 10px; }
    .sign-table { width: 100%; margin-top: 40px; border-spacing: 40px 0; }
    .sign-table td { height: 90px; vertical-align: bottom; }
    .sign-line { border-top: 1px solid #000; padding-top: 6px; }
  </style>';

  ob_start();
  ?>
  <?= $styles ?>
  <h1><?= htmlspecialchars($title) ?></h1>
  <div class="muted small">Generado: <?= date('Y-m-d H:i') ?></div>

  <div class="section">
    <strong>Cabecera</strong>
    <div class="kv">
      <div><b>Packing List:</b> <?= htmlspecialchars((string)($cabecera['codigo'] ?? '-')) ?></div>
      <div><b>Fecha PL:</b> <?= htmlspecialchars((string)($cabecera['fecha'] ?? '-')) ?></div>
      <div><b>Cliente:</b> <?= htmlspecialchars((string)($cabecera['cliente'] ?? '-')) ?></div>
      <div><b>Depósito:</b> <?= htmlspecialchars((string)($cabecera['deposito'] ?? '-')) ?></div>
      <div><b>Móvil:</b> <?= htmlspecialchars((string)($cabecera['movil'] ?? '-')) ?></div>
      <div><b>Chofer:</b> <?= htmlspecialchars((string)($cabecera['chofer'] ?? '-')) ?></div>
      <div><b>Fecha ingreso:</b> <?= htmlspecialchars((string)($cabecera['fecha_ingreso'] ?? '-')) ?></div>
      <div><b>Hora arribo:</b> <?= htmlspecialchars((string)($cabecera['hora_arribo'] ?? '-')) ?></div>
      <div><b>Hora inicio:</b> <?= htmlspecialchars((string)($cabecera['hora_inicio'] ?? '-')) ?></div>
      <div><b>Hora fin:</b> <?= htmlspecialchars((string)($cabecera['hora_fin'] ?? '-')) ?></div>
      <div><b>Operarios:</b> <?= htmlspecialchars((string)($cabecera['operarios'] ?? '-')) ?></div>
      <div><b>Documento:</b> <?= htmlspecialchars((string)($cabecera['doc_tipo'] ?? '')) ?> <?= htmlspecialchars((string)($cabecera['doc_numero'] ?? '')) ?></div>
      <div><b>Fecha doc.:</b> <?= htmlspecialchars((string)($cabecera['doc_fecha'] ?? '-')) ?></div>
      <div><b>Observación:</b> <?= htmlspecialchars((string)($cabecera['observacion'] ?? '')) ?></div>
    </div>
  </div>

  <div class="section">
    <strong>Resumen de recepción</strong>
    <div class="kv">
      <div><b>Pallets recibidos:</b> <?= nf($totPallets) ?></div>
      <div><b>UV totales:</b> <?= nf($totUV) ?></div>
      <div><b>UC totales:</b> <?= nf($totUC) ?></div>
    </div>
  </div>

  <div class="section">
    <strong>Detalle por pallet</strong>
    <table class="grid">
      <thead>
        <tr>
          <th>Pallet</th>
          <th>Ubicación</th>
          <th class="right">Ítems</th>
          <th class="right">UV</th>
          <th class="right">UC</th>
        </tr>
      </thead>
      <tbody>
      <?php foreach ($pallets as $row): ?>
        <tr>
          <td><?= htmlspecialchars((string)($row['pallet_codigo'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($row['posicion'] ?? '')) ?></td>
          <td class="right"><?= nf($row['items_count'] ?? 0) ?></td>
          <td class="right"><?= nf($row['uv_total'] ?? 0) ?></td>
          <td class="right"><?= nf($row['uc_total'] ?? 0) ?></td>
        </tr>
      <?php endforeach; ?>
      </tbody>
      <tfoot>
        <tr>
          <th colspan="3" class="right">Totales</th>
          <th class="right"><?= nf($totUV) ?></th>
          <th class="right"><?= nf($totUC) ?></th>
        </tr>
      </tfoot>
    </table>
  </div>

  <div class="section">
    <strong>Detalle de ítems recibidos</strong>
    <table class="grid">
      <thead>
        <tr>
          <th>Pallet</th>
          <th>Producto</th>
          <th>Lote</th>
          <th>F. Producción</th>
          <th>F. Vencimiento</th>
          <th class="right">UV</th>
          <th class="right">UC</th>
          <th class="right">Peso (kg)</th>
          <th>Ubicación</th>
          <th>Estado</th>
        </tr>
      </thead>
      <tbody>
      <?php foreach ($items as $it): ?>
        <tr>
          <td><?= htmlspecialchars((string)($it['pallet_codigo'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($it['producto'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($it['lote'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($it['fecha_produccion'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($it['fecha_vencimiento'] ?? '')) ?></td>
          <td class="right"><?= nf($it['uv_cajas'] ?? 0) ?></td>
          <td class="right"><?= nf($it['uc_unidades'] ?? 0) ?></td>
          <td class="right"><?= $it['peso'] !== null ? nf($it['peso'], 2) : '' ?></td>
          <td><?= htmlspecialchars((string)($it['posicion'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($it['estado'] ?? '')) ?></td>
        </tr>
      <?php endforeach; ?>
      </tbody>
    </table>
  </div>

  <div class="section">
    <strong>Firmas</strong>
    <table class="sign-table">
      <tr>
        <td>
          <div class="sign-line"></div>
          <div class="small">Firma Chofer</div>
        </td>
        <td>
          <div class="sign-line"></div>
          <div class="small">Firma Encargado de Recepción</div>
        </td>
      </tr>
    </table>
  </div>
  <?php
  $html = ob_get_clean();

  $mpdf->SetTitle($title);
  $mpdf->SetAuthor((string)env('APP_NAME', 'SOL'));
  $mpdf->WriteHTML($html);
  $mpdf->Output('recepcion_' . $plId . '.pdf', \Mpdf\Output\Destination::INLINE);

} catch (Throwable $e) {
  http_response_code(500);
  header('Content-Type: text/plain; charset=utf-8');
  echo env('APP_ENV') === 'local'
    ? ('Error generando PDF: ' . $e->getMessage())
    : 'No se pudo generar el PDF de recepción';
}
