<?php
declare(strict_types=1);

/**
 * API: Confirmar Salida consolidando desde staging (so_import_row)
 * Ruta: api/operaciones/so_confirm_commit.php
 * Método: POST
 *
 * Entrada:
 * - batch_id    (int, requerido)
 * - cliente_id  (int, requerido)
 *
 * Proceso:
 * - Crea so_pedido (estado RECIBIDO) con cliente_ref/cliente_id
 * - Agrupa filas del batch por destinatario (cod/nombre) y doc (tipo/numero)
 * - Crea so_pedido_dest para cada grupo
 * - Resuelve producto por alias (pl_producto_alias) o por SKU en para_productos; si no existe, lo crea
 * - Crea ítems (so_pedido_dest_item) con expected_uv/expected_uc
 * - Vincula staging (so_import_row.pedido_id/pedido_dest_id/item_id) y marca OK
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function respond_json(array $payload, int $code = 200): void {
    http_response_code($code);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE);
    exit;
}

// Helper: cache de columnas
function hasColumn(PDO $pdo, string $table, string $col): bool {
    static $cache = [];
    $key = $table.'|'.$col;
    if (array_key_exists($key, $cache)) return $cache[$key];
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?");
    $stmt->execute([$table, $col]);
    return $cache[$key] = ((int)$stmt->fetchColumn() > 0);
}

// Helper: truncar cadenas respetando multibyte
function trunc_str(?string $s, int $max): string {
    $s = (string)$s;
    if ($max <= 0) return '';
    if (function_exists('mb_substr')) return mb_substr($s, 0, $max);
    return substr($s, 0, $max);
}

// Helper: asegurar que pl_producto_alias tenga cliente_id y un índice único (cliente_id, sku_cliente)
function ensureAliasSchemaWithClienteId(PDO $pdo): void {
    // Agregar columna cliente_id si falta
    if (!hasColumn($pdo, 'pl_producto_alias', 'cliente_id')) {
        try {
            $pdo->exec("ALTER TABLE pl_producto_alias ADD COLUMN cliente_id BIGINT UNSIGNED NULL AFTER id");
        } catch (Throwable $e) {
            // Ignorar si otro proceso ya lo agregó o si no tenemos permisos
        }
    }
    // Asegurar índice único por (cliente_id, sku_cliente)
    try {
        $stmt = $pdo->query("SELECT COUNT(1) FROM INFORMATION_SCHEMA.STATISTICS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'pl_producto_alias' AND INDEX_NAME = 'uk_alias_cli_id'");
        $exists = (int)$stmt->fetchColumn() > 0;
        if (!$exists) {
            // Crear índice único; NULL en cliente_id permite múltiples NULL (MySQL), está bien para legacy
            $pdo->exec("ALTER TABLE pl_producto_alias ADD UNIQUE KEY uk_alias_cli_id (cliente_id, sku_cliente)");
        }
    } catch (Throwable $e) {
        // Ignorar si ya existe o no hay permisos
    }
}

/** Asegura catálogo mínimo de estados de pedido de salida */
function ensureSoPedidoEstados(PDO $pdo): void {
    static $done = false;
    if ($done) return;
    $defaults = [
        ['RECIBIDO', 'Recibido', 1],
        ['EN_PREPARACION', 'En preparación', 2],
        ['PARCIAL', 'Parcialmente preparado', 3],
        ['PREPARADO', 'Preparado', 4],
        ['EN_EMBARQUE', 'En embarque', 5],
        ['DESPACHADO', 'Despachado', 6],
        ['CERRADO', 'Cerrado', 7],
        ['CANCELADO', 'Cancelado', 9],
    ];
    $stmt = $pdo->prepare("INSERT INTO so_pedido_estado (code, nombre, orden) VALUES (?,?,?) ON DUPLICATE KEY UPDATE nombre=VALUES(nombre), orden=VALUES(orden)");
    foreach ($defaults as $row) {
        try {
            $stmt->execute($row);
        } catch (Throwable $e) {
            // Ignorar fallos de permisos; si falta la tabla se lanzará más adelante.
        }
    }
    $done = true;
}

/** Construye un cliente_ref legible (CLI-XXX) desde razón social y/o RUC */
function build_cliente_ref_from_razon(string $razon, ?string $ruc = null): string {
    $s = $razon;
    if (function_exists('iconv')) {
        $tmp = @iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $s);
        if ($tmp !== false) $s = $tmp;
    }
    $s = strtoupper($s);
    $s = preg_replace('/[^A-Z0-9 ]+/', ' ', $s);
    $tokens = preg_split('/\s+/', trim($s)) ?: [];
    $stop = ['SA','SAS','SRL','SAC','LTDA','LTD','CORP','CORPORATION','COMPANIA','CIA','DE','LA','EL','LOS','LAS','Y'];
    $alias = '';
    foreach ($tokens as $t) { if ($t === '' || in_array($t, $stop, true)) continue; $alias = $t; break; }
    if ($alias === '') {
        $ruc = (string)$ruc;
        $ruc = preg_replace('/[^0-9A-Z]+/','', strtoupper($ruc ?? 'GEN'));
        $alias = $ruc !== '' ? $ruc : 'GEN';
    }
    return 'CLI-' . $alias;
}

try {
    if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
        respond_json(['ok' => false, 'error' => 'Método no permitido.'], 405);
    }

    $batchId   = (int)($_POST['batch_id'] ?? 0);
    $clienteId = (int)($_POST['cliente_id'] ?? 0);

    if ($batchId <= 0) respond_json(['ok'=>false, 'error'=>'batch_id es requerido'], 422);
    if ($clienteId <= 0) respond_json(['ok'=>false, 'error'=>'cliente_id es requerido'], 422);

    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec("SET NAMES utf8mb4");

    // Cliente ref
    $stCli = $pdo->prepare("SELECT razon_social, ruc FROM para_clientes WHERE id=? LIMIT 1");
    $stCli->execute([$clienteId]);
    $cli = $stCli->fetch(PDO::FETCH_ASSOC) ?: ['razon_social' => 'Generico', 'ruc' => null];
    $clienteRef = build_cliente_ref_from_razon((string)$cli['razon_social'], (string)($cli['ruc'] ?? ''));

    // Estado RECIBIDO
    ensureSoPedidoEstados($pdo);
    $estadoId = (int)$pdo->query("SELECT id FROM so_pedido_estado WHERE code='RECIBIDO' LIMIT 1")->fetchColumn();
    if ($estadoId <= 0) respond_json(['ok'=>false,'error'=>'Catálogo so_pedido_estado incompleto (RECIBIDO).'], 500);

    // Aliasing para productos: SIEMPRE por cliente_id (auto-ajustar esquema si falta)
    ensureAliasSchemaWithClienteId($pdo);
    $aliasUsesClienteId   = hasColumn($pdo, 'pl_producto_alias', 'cliente_id');
    $aliasHasProductoDesc = hasColumn($pdo, 'pl_producto_alias', 'producto_desc');
    if (!$aliasUsesClienteId) {
        respond_json([
            'ok' => false,
            'error' => 'Esquema de alias no compatible',
            'message' => 'No fue posible asegurar columna cliente_id en pl_producto_alias. Este sistema usa siempre cliente_id.'
        ], 500);
    }

    // Validar: el batch no debe contener errores
    $stBatch = $pdo->prepare("SELECT rows_error FROM so_import_batch WHERE id=? LIMIT 1");
    $stBatch->execute([$batchId]);
    $rowsErrCnt = (int)$stBatch->fetchColumn();
    if ($rowsErrCnt > 0) {
        respond_json([
            'ok' => false,
            'error' => 'No se puede confirmar: hay filas con error en el batch.',
            'rows_error' => $rowsErrCnt,
        ], 422);
    }

    // Leer staging (todas las filas)
    $stRows = $pdo->prepare("SELECT id, rownum,
        JSON_UNQUOTE(JSON_EXTRACT(raw,'$.cod'))                 AS cod,
        JSON_UNQUOTE(JSON_EXTRACT(raw,'$.razon_social'))        AS razon_social,
        JSON_UNQUOTE(JSON_EXTRACT(raw,'$.nombre_destinatario')) AS destinatario,
        JSON_UNQUOTE(JSON_EXTRACT(raw,'$.direccion'))           AS direccion,
        JSON_UNQUOTE(JSON_EXTRACT(raw,'$.telefono'))            AS telefono,
        JSON_UNQUOTE(JSON_EXTRACT(raw,'$.sku_cliente'))         AS sku,
        CAST(JSON_UNQUOTE(JSON_EXTRACT(raw,'$.uv_cajas')) AS UNSIGNED)   AS uv_cajas,
        CAST(JSON_UNQUOTE(JSON_EXTRACT(raw,'$.uc_sueltas')) AS UNSIGNED) AS uc_sueltas,
        JSON_UNQUOTE(JSON_EXTRACT(raw,'$.numero_doc'))          AS numero_doc,
        JSON_UNQUOTE(JSON_EXTRACT(raw,'$.tipo_factura'))        AS tipo_factura,
        JSON_UNQUOTE(JSON_EXTRACT(raw,'$.fecha_salida'))        AS fecha_salida,
        JSON_UNQUOTE(JSON_EXTRACT(raw,'$.pre_embarque'))        AS pre_embarque
      FROM so_import_row WHERE batch_id=? ORDER BY rownum ASC, id ASC");
    $stRows->execute([$batchId]);
    $rows = $stRows->fetchAll(PDO::FETCH_ASSOC) ?: [];
    if (!$rows) respond_json(['ok'=>false,'error'=>'No hay filas en staging para el batch indicado.'], 404);

    // Agrupar por pre_embarque (si viene). Si no, un solo grupo '_DEFAULT_'
    $pedidoGroups = [];
    foreach ($rows as $r) {
        $grp = trim((string)($r['pre_embarque'] ?? ''));
        if ($grp === '') $grp = '_DEFAULT_';
        if (!isset($pedidoGroups[$grp])) $pedidoGroups[$grp] = [];
        $pedidoGroups[$grp][] = $r;
    }

    $pdo->beginTransaction();

    $pedidosResult = [];

    // Preparar statements comunes
    $stInsPed = $pdo->prepare("INSERT INTO so_pedido (codigo, cliente_ref, cliente_id, fecha_pedido, estado_id, observacion, import_batch_id)
                               VALUES (?,?,?,?,?,?,?)");

    // Helpers para producto (exclusivo cliente_id)
    $stFindAliasById = $pdo->prepare("SELECT producto_id FROM pl_producto_alias WHERE cliente_id=? AND sku_cliente=? LIMIT 1");
    $stUpsertAliasId = $aliasHasProductoDesc
        ? $pdo->prepare("INSERT INTO pl_producto_alias (cliente_id, sku_cliente, producto_id, producto_desc) VALUES (?,?,?,?) ON DUPLICATE KEY UPDATE producto_id=VALUES(producto_id), producto_desc=VALUES(producto_desc)")
        : $pdo->prepare("INSERT INTO pl_producto_alias (cliente_id, sku_cliente, producto_id) VALUES (?,?,?) ON DUPLICATE KEY UPDATE producto_id=VALUES(producto_id)");
    $stFindProdBySku   = $pdo->prepare("SELECT id FROM para_productos WHERE sku=? LIMIT 1");
    $stNewProd         = $pdo->prepare("INSERT INTO para_productos (sku, denominacion) VALUES (?, ?)");

    // Helper: resolver destinatario (por cod o nombre/razon_social + cliente) con esquema dinámico
    $stFindDestByCod   = $pdo->prepare("SELECT id FROM para_destinatarios WHERE cliente_id=? AND cod=? LIMIT 1");
    $destHasRazon      = hasColumn($pdo, 'para_destinatarios', 'razon_social');
    $destHasNombre     = hasColumn($pdo, 'para_destinatarios', 'nombre');
    $destHasFantasia   = hasColumn($pdo, 'para_destinatarios', 'fantasia');
    $destHasTelefono   = hasColumn($pdo, 'para_destinatarios', 'telefono');
    $destHasDireccion  = hasColumn($pdo, 'para_destinatarios', 'direccion');
    $destHasCiudad     = hasColumn($pdo, 'para_destinatarios', 'ciudad_id');
    $findByNameSql     = $destHasRazon
        ? "SELECT id FROM para_destinatarios WHERE cliente_id=? AND (razon_social=? OR fantasia=?) LIMIT 1"
        : "SELECT id FROM para_destinatarios WHERE cliente_id=? AND (nombre=? OR fantasia=?) LIMIT 1";
    $stFindDestByName  = $pdo->prepare($findByNameSql);
    // Armar INSERT según columnas disponibles, priorizando cubrir NOT NULL
    if ($destHasRazon && $destHasNombre && $destHasTelefono && $destHasDireccion && $destHasCiudad && $destHasFantasia) {
        $stInsDest = $pdo->prepare("INSERT INTO para_destinatarios (cliente_id, cod, nombre, fantasia, razon_social, telefono, direccion, ciudad_id) VALUES (?,?,?,?,?,?,?,?)");
    } elseif ($destHasRazon && $destHasNombre) {
        $stInsDest = $pdo->prepare("INSERT INTO para_destinatarios (cliente_id, cod, nombre, razon_social) VALUES (?,?,?,?)");
    } elseif ($destHasRazon) {
        $stInsDest = $pdo->prepare("INSERT INTO para_destinatarios (cliente_id, cod, razon_social) VALUES (?,?,?)");
    } else {
        $stInsDest = $pdo->prepare("INSERT INTO para_destinatarios (cliente_id, cod, nombre) VALUES (?,?,?)");
    }

    $resolveDest = function(?string $cod = null, ?string $nombre = null, ?string $dir = null, ?string $tel = null) use ($clienteId, $stFindDestByCod, $stFindDestByName, $stInsDest, $pdo, $destHasRazon, $destHasNombre, $destHasFantasia, $destHasTelefono, $destHasDireccion, $destHasCiudad): int {
        $cod = trim((string)$cod);
        $nombre = trim((string)$nombre);
        if ($cod !== '') {
            $stFindDestByCod->execute([$clienteId, $cod]);
            $id = (int)$stFindDestByCod->fetchColumn();
            if ($id > 0) return $id;
        }
        if ($nombre !== '') {
            $stFindDestByName->execute([$clienteId, $nombre, $nombre]);
            $id = (int)$stFindDestByName->fetchColumn();
            if ($id > 0) return $id;
        }
        // crear
        $nomRaw = $nombre !== '' ? $nombre : ($cod !== '' ? $cod : 'DEST-' . date('Ymd-His'));
        $codSafe = $cod !== '' ? trunc_str($cod, 20) : null;
        $nomSafe = trunc_str($nomRaw, 100);
        $fantSafe = trunc_str($nomRaw, 100);
        $razonSafe = trunc_str($nomRaw, 20);
        $dirSafe = ($dir !== null && $dir !== '') ? (string)$dir : 'S/D';
        $telNum = (int)preg_replace('/[^0-9]/','', (string)$tel);
        $telSafe = $telNum >= 0 ? $telNum : 0;
        $ciudadSafe = 0; // sin FK conocida, usar 0

        if ($destHasRazon && $destHasNombre && $destHasTelefono && $destHasDireccion && $destHasCiudad && $destHasFantasia) {
            $stInsDest->execute([$clienteId, $codSafe, $nomSafe, $fantSafe, $razonSafe, $telSafe, $dirSafe, $ciudadSafe]);
        } elseif ($destHasRazon && $destHasNombre) {
            $stInsDest->execute([$clienteId, $codSafe, $nomSafe, $razonSafe]);
        } elseif ($destHasRazon) {
            $stInsDest->execute([$clienteId, $codSafe, $razonSafe]);
        } else {
            $stInsDest->execute([$clienteId, $codSafe, $nomSafe]);
        }
        return (int)$pdo->lastInsertId();
    };

    foreach ($pedidoGroups as $grpKey => $grpRows) {
        // Determinar fecha del pedido (primera no nula del grupo)
        $pedidoFecha = null;
        foreach ($grpRows as $gr) { if (!empty($gr['fecha_salida'])) { $pedidoFecha = (string)$gr['fecha_salida']; break; } }
        if ($pedidoFecha === null) $pedidoFecha = date('Y-m-d');

        // Código por grupo (mantener único)
        $pedidoCodigo = 'SO-' . date('Ymd-His') . ($grpKey !== '_DEFAULT_' ? ('-' . preg_replace('/[^A-Za-z0-9]+/','', (string)$grpKey)) : '');

        $stInsPed->execute([$pedidoCodigo, $clienteRef, $clienteId, $pedidoFecha, $estadoId, null, $batchId]);
        $pedidoId = (int)$pdo->lastInsertId();

        // Agrupar por destinatario+doc dentro del grupo
        $groups = []; // key => ['dest_id'=>, 'doc_tipo'=>, 'doc_numero'=>, 'items'=> sku=>[uv,uc], 'rows'=>[]]
        foreach ($grpRows as $r) {
        $sku = trim((string)($r['sku'] ?? ''));
        $uv  = (int)($r['uv_cajas'] ?? 0);
        $uc  = (int)($r['uc_sueltas'] ?? 0);
        $destNombre = trim((string)($r['destinatario'] ?? ''));
        $cod = trim((string)($r['cod'] ?? ''));
        $dir = trim((string)($r['direccion'] ?? ''));
        $tel = trim((string)($r['telefono'] ?? ''));
        $docT = trim((string)($r['tipo_factura'] ?? ''));
        $docN = trim((string)($r['numero_doc'] ?? ''));

        if ($destNombre === '' && $cod === '') { $destNombre = 'DESTINATARIO'; }
        $destId = $resolveDest($cod !== '' ? $cod : null, $destNombre, $dir, $tel);

        $gkey = $destId . '|' . $docT . '|' . $docN;
        if (!isset($groups[$gkey])) {
            $groups[$gkey] = [
                'dest_id' => $destId,
                'doc_tipo' => ($docT !== '' ? $docT : null),
                'doc_numero' => ($docN !== '' ? $docN : null),
                'items' => [],
                'rows' => [],
            ];
        }
        if ($sku !== '' && ($uv > 0 || $uc > 0)) {
            if (!isset($groups[$gkey]['items'][$sku])) $groups[$gkey]['items'][$sku] = ['uv' => 0, 'uc' => 0];
            $groups[$gkey]['items'][$sku]['uv'] += $uv;
            $groups[$gkey]['items'][$sku]['uc'] += $uc;
        }
        $groups[$gkey]['rows'][] = $r; // para linkear luego
        }

        // Insertar destinos e ítems
    $stInsDestRow = $pdo->prepare("INSERT INTO so_pedido_dest (pedido_id, destinatario_id, doc_tipo, doc_numero, nota) VALUES (?,?,?,?,NULL)");
    $stFindItem   = $pdo->prepare("SELECT id FROM so_pedido_dest_item WHERE pedido_dest_id=? AND producto_id=? AND (lote_codigo <=> ?) LIMIT 1");
    $stInsItem    = $pdo->prepare("INSERT INTO so_pedido_dest_item (pedido_dest_id, producto_id, lote_codigo, expected_uv, expected_uc) VALUES (?,?,?,?,?)");
    $stUpdItem    = $pdo->prepare("UPDATE so_pedido_dest_item SET expected_uv = expected_uv + ?, expected_uc = expected_uc + ? WHERE id=?");

    $itemIdMap = []; // key: pd_id|sku => item_id
    $pedidoDestIds = [];

        foreach ($groups as $gkey => $g) {
        // crear pedido_dest
        $stInsDestRow->execute([$pedidoId, $g['dest_id'], $g['doc_tipo'], $g['doc_numero']]);
        $pdId = (int)$pdo->lastInsertId();
        $pedidoDestIds[$gkey] = $pdId;

        // por cada SKU del grupo: resolver producto y crear/acumular ítem
        foreach ($g['items'] as $sku => $qty) {
            $prodId = 0;
            $stFindAliasById->execute([$clienteId, $sku]);
            $prodId = (int)$stFindAliasById->fetchColumn();
            if ($prodId <= 0) {
                $stFindProdBySku->execute([$sku]);
                $prodId = (int)$stFindProdBySku->fetchColumn();
            }
            if ($prodId <= 0) {
                $den = 'SKU ' . $sku;
                $stNewProd->execute([$sku, $den]);
                $prodId = (int)$pdo->lastInsertId();
            }
            // asegurar alias
            if ($aliasHasProductoDesc) $stUpsertAliasId->execute([$clienteId, $sku, $prodId, null]);
            else $stUpsertAliasId->execute([$clienteId, $sku, $prodId]);

            // upsert ítem (sin lote)
            $stFindItem->execute([$pdId, $prodId, null]);
            $itId = (int)$stFindItem->fetchColumn();
            if ($itId > 0) {
                $stUpdItem->execute([(int)$qty['uv'], (int)$qty['uc'], $itId]);
            } else {
                $stInsItem->execute([$pdId, $prodId, null, (int)$qty['uv'], (int)$qty['uc']]);
                $itId = (int)$pdo->lastInsertId();
            }
            $itemIdMap[$pdId . '|' . $sku] = $itId;
        }
        }

        // Vincular staging
        $stUpdRow = $pdo->prepare("UPDATE so_import_row SET pedido_id=:pid, pedido_dest_id=:pdid, item_id=:iid, status='OK', error_msg=NULL WHERE id=:id");
        foreach ($groups as $gkey => $g) {
            $pdId = (int)$pedidoDestIds[$gkey];
            foreach ($g['rows'] as $r) {
                $sku = trim((string)($r['sku'] ?? ''));
                $iid = ($sku !== '' && isset($itemIdMap[$pdId . '|' . $sku])) ? (int)$itemIdMap[$pdId . '|' . $sku] : null;
                $stUpdRow->execute([
                    ':pid'  => $pedidoId,
                    ':pdid' => $pdId,
                    ':iid'  => $iid,
                    ':id'   => (int)$r['id'],
                ]);
            }
        }

        // Resumen por pedido
        $stAgg = $pdo->prepare("SELECT COUNT(*) AS items, SUM(expected_uv) AS exp_uv, SUM(expected_uc) AS exp_uc
                                 FROM so_pedido_dest_item WHERE pedido_dest_id IN (
                                   SELECT id FROM so_pedido_dest WHERE pedido_id=?
                                 )");
        $stAgg->execute([$pedidoId]);
        $agg = $stAgg->fetch(PDO::FETCH_ASSOC) ?: ['items'=>0,'exp_uv'=>0,'exp_uc'=>0];
        $pedidosResult[] = [
            'so_id'   => $pedidoId,
            'codigo'  => $pedidoCodigo,
            'fecha'   => $pedidoFecha,
            'summary' => [
                'destinatarios' => count($groups),
                'items'         => (int)$agg['items'],
                'expected_uv'   => (int)$agg['exp_uv'],
                'expected_uc'   => (int)$agg['exp_uc'],
            ],
            'pre_embarque' => ($grpKey === '_DEFAULT_' ? null : $grpKey),
        ];
    }

    $pdo->commit();

    // Respuesta: múltiples pedidos. Mantener compat con so_id (primero)
    $first = $pedidosResult[0] ?? null;
    respond_json([
        'ok'          => true,
        'cliente_ref' => $clienteRef,
        'pedidos'     => $pedidosResult,
        'so_id'       => $first['so_id'] ?? null,
        'message'     => 'Salidas consolidadas por pre-embarque.'
    ]);

} catch (Throwable $e) {
    if (!empty($pdo) && $pdo->inTransaction()) $pdo->rollBack();
    respond_json([
        'ok' => false,
        'error' => 'No se pudo confirmar la salida',
        'message' => $e->getMessage(),
    ], 500);
}
