<?php
declare(strict_types=1);

/**
 * API: Diagnóstico de stock por pedido para preparación (UV/UC)
 * Ruta: api/operaciones/so_diag_stock.php
 * Método: GET
 *
 * Parámetros:
 * - so_id (int) obligatorio
 * - limit (int) opcional, top N candidatos FEFO por item (default 3)
 *
 * Respuesta JSON:
 * {
 *   ok: true,
 *   deposito: { id, code },
 *   totals: { expected_uv, prepared_uv, pending_uv, expected_uc, prepared_uc, pending_uc },
 *   items: [
 *     {
 *       producto_id, sku, denominacion, lote_codigo,
 *       need_uv, need_uc,
 *       pick_uv, pick_uc, other_uv, other_uc,
 *       cands_pick: [ { pos_code, lote_codigo, venc, qty_uv, qty_uc } ],
 *       cands_other: [ { pos_code, lote_codigo, venc, qty_uv, qty_uc } ]
 *     }
 *   ]
 * }
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function respond(array $p, int $c=200){ http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }

try {
  $soId = isset($_GET['so_id']) ? (int)$_GET['so_id'] : 0;
  $limit = isset($_GET['limit']) ? max(1, (int)$_GET['limit']) : 3;
  $depCodeOverride = isset($_GET['deposito_code']) ? trim((string)$_GET['deposito_code']) : '';
  $depIdOverride = isset($_GET['deposito_id']) ? (int)$_GET['deposito_id'] : 0;
  if ($soId <= 0) respond(['ok'=>false,'error'=>'so_id requerido'], 422);

  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $pdo->exec('SET NAMES utf8mb4');

  // Resolver depósito: override explícito > desde PRE > fallback DEP1
  $depRow = null;
  if ($depIdOverride > 0) {
    $st = $pdo->prepare('SELECT id, code FROM wh_deposito WHERE id=? LIMIT 1');
    $st->execute([$depIdOverride]);
    $depRow = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  } elseif ($depCodeOverride !== '') {
    $st = $pdo->prepare('SELECT id, code FROM wh_deposito WHERE code=? LIMIT 1');
    $st->execute([$depCodeOverride]);
    $depRow = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  }
  if (!$depRow) {
    $st = $pdo->prepare('SELECT d.id, d.code FROM so_preembarque pre JOIN wh_deposito d ON d.id=pre.deposito_id WHERE pre.pedido_id=? ORDER BY pre.id DESC LIMIT 1');
    $st->execute([$soId]);
    $depRow = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  }
  if (!$depRow) {
    $st = $pdo->query("SELECT id, code FROM wh_deposito WHERE code='DEP1' LIMIT 1");
    $depRow = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  }
  if (!$depRow) respond(['ok'=>false,'error'=>'No se pudo resolver el depósito (faltan datos de wh_deposito).'], 500);
  $depId = (int)$depRow['id'];

  // Items del pedido con necesidades actuales
  $sqlItems = 'SELECT i.id AS item_id, i.producto_id, i.lote_codigo, 
                      GREATEST(i.expected_uv - i.prepared_uv,0) AS need_uv,
                      GREATEST(i.expected_uc - i.prepared_uc,0) AS need_uc,
                      p.sku, p.denominacion
                 FROM so_pedido_dest d
                 JOIN so_pedido_dest_item i ON i.pedido_dest_id=d.id
                 JOIN para_productos p ON p.id=i.producto_id
                WHERE d.pedido_id = ?
             ORDER BY i.id ASC';
  $st = $pdo->prepare($sqlItems);
  $st->execute([$soId]);
  $items = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

  // Totales del pedido (para alinear con cabecera de Preparación)
  $sqlTotals = 'SELECT 
                  COALESCE(SUM(i.expected_uv),0) AS expected_uv,
                  COALESCE(SUM(i.prepared_uv),0) AS prepared_uv,
                  COALESCE(SUM(i.expected_uc),0) AS expected_uc,
                  COALESCE(SUM(i.prepared_uc),0) AS prepared_uc
                FROM so_pedido_dest d
                JOIN so_pedido_dest_item i ON i.pedido_dest_id=d.id
               WHERE d.pedido_id=?';
  $st = $pdo->prepare($sqlTotals);
  $st->execute([$soId]);
  $totRow = $st->fetch(PDO::FETCH_ASSOC) ?: ['expected_uv'=>0,'prepared_uv'=>0,'expected_uc'=>0,'prepared_uc'=>0];

  $outItems = [];
  foreach ($items as $it) {
    $prodId = (int)$it['producto_id'];
    $loteCode = $it['lote_codigo'] !== null ? (string)$it['lote_codigo'] : null;

    // Factor de conversión: UC por UV (caja) si existe en para_producto_pack
    $ucPerUv = 0; // 0 = desconocido/no definido
    try {
      $stF = $pdo->prepare("SELECT COALESCE(MAX(unidades_por_uv),0) FROM para_producto_pack WHERE producto_id=?");
      $stF->execute([$prodId]);
      $ucPerUv = (int)($stF->fetchColumn() ?: 0);
    } catch (Throwable $e) {
      $ucPerUv = 0;
    }

    // Totales de stock en PICKING
    $params = [$depId, $prodId];
    $condLote = '';
    if ($loteCode !== null && $loteCode !== '') {
      $condLote = ' AND EXISTS(SELECT 1 FROM wh_lote lx WHERE lx.id=s.lote_id AND lx.codigo=?) ';
      $params[] = $loteCode;
    }
    $sqlPick = "SELECT COALESCE(SUM(s.qty_uv),0) AS uv, COALESCE(SUM(s.qty_uc),0) AS uc
                  FROM wh_stock s
                  JOIN wh_posicion p ON p.id=s.posicion_id
                  JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PICKING'
                 WHERE s.deposito_id=? AND s.producto_id=? $condLote ";
  $st = $pdo->prepare($sqlPick);
  $st->execute($params);
  $pick = $st->fetch(PDO::FETCH_ASSOC) ?: ['uv'=>0,'uc'=>0];
  $pickUv = (int)($pick['uv'] ?? 0);
  $pickUc = (int)($pick['uc'] ?? 0);
  $pickUvEquiv = ($ucPerUv > 0) ? intdiv($pickUc, $ucPerUv) : 0; // UV equivalentes si se arma caja desde UC

    // Totales de stock en OTROS ambientes (excluye PREP, PICKING, CUARENTENA)
    $params = [$depId, $prodId];
    if ($loteCode !== null && $loteCode !== '') { $params[] = $loteCode; }
    $sqlOther = "SELECT COALESCE(SUM(s.qty_uv),0) AS uv, COALESCE(SUM(s.qty_uc),0) AS uc
                   FROM wh_stock s
                   JOIN wh_posicion p ON p.id=s.posicion_id
                   JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code NOT IN ('PREP','PICKING','CUARENTENA')
                  WHERE s.deposito_id=? AND s.producto_id=? $condLote ";
  $st = $pdo->prepare($sqlOther);
  $st->execute($params);
  $other = $st->fetch(PDO::FETCH_ASSOC) ?: ['uv'=>0,'uc'=>0];
  $otherUv = (int)($other['uv'] ?? 0);
  $otherUc = (int)($other['uc'] ?? 0);
  $otherUvEquiv = ($ucPerUv > 0) ? intdiv($otherUc, $ucPerUv) : 0;

    // Candidatos FEFO (PICKING)
    $params = [$depId, $prodId];
    if ($loteCode !== null && $loteCode !== '') { $params[] = $loteCode; $params[] = $loteCode; }
    $sqlCandPick = "SELECT pf.code AS pos_code, l.codigo AS lote_codigo, l.fecha_vencimiento AS venc,
                           s.qty_uv, s.qty_uc
                      FROM wh_stock s
                      JOIN wh_posicion pf ON pf.id=s.posicion_id
                      JOIN wh_ambiente a ON a.id=pf.ambiente_id AND a.code='PICKING'
                      LEFT JOIN wh_lote l ON l.id=s.lote_id
                     WHERE s.deposito_id=? AND s.producto_id=? " .
                     ($loteCode !== null && $loteCode !== '' ? " AND EXISTS(SELECT 1 FROM wh_lote lx WHERE lx.id=s.lote_id AND lx.codigo=?) " : '') .
                    " AND (s.qty_uv>0 OR s.qty_uc>0)
                     ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC, pf.code ASC
                     LIMIT " . (int)$limit;
    $st = $pdo->prepare($sqlCandPick);
    $st->execute($params);
    $candPick = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

    // Candidatos FEFO (OTROS)
    $params = [$depId, $prodId];
    if ($loteCode !== null && $loteCode !== '') { $params[] = $loteCode; $params[] = $loteCode; }
    $sqlCandOther = "SELECT pf.code AS pos_code, l.codigo AS lote_codigo, l.fecha_vencimiento AS venc,
                            s.qty_uv, s.qty_uc
                       FROM wh_stock s
                       JOIN wh_posicion pf ON pf.id=s.posicion_id
                       JOIN wh_ambiente a ON a.id=pf.ambiente_id AND a.code NOT IN ('PREP','PICKING','CUARENTENA')
                       LEFT JOIN wh_lote l ON l.id=s.lote_id
                      WHERE s.deposito_id=? AND s.producto_id=? " .
                      ($loteCode !== null && $loteCode !== '' ? " AND EXISTS(SELECT 1 FROM wh_lote lx WHERE lx.id=s.lote_id AND lx.codigo=?) " : '') .
                     " AND (s.qty_uv>0 OR s.qty_uc>0)
                      ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC, pf.code ASC
                      LIMIT " . (int)$limit;
    $st = $pdo->prepare($sqlCandOther);
    $st->execute($params);
    $candOther = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $outItems[] = [
      'producto_id' => $prodId,
      'sku' => (string)$it['sku'],
      'denominacion' => (string)$it['denominacion'],
      'lote_codigo' => $loteCode,
      'need_uv' => (int)$it['need_uv'],
      'need_uc' => (int)$it['need_uc'],
      'pick_uv' => $pickUv,
      'pick_uc' => $pickUc,
      'pick_uv_equiv_from_uc' => $pickUvEquiv,
      'other_uv' => $otherUv,
      'other_uc' => $otherUc,
      'other_uv_equiv_from_uc' => $otherUvEquiv,
      'uc_per_uv' => $ucPerUv,
      'cands_pick' => $candPick,
      'cands_other' => $candOther,
    ];
  }

  // Métrica rápida: cuántos items esperan UV>0 pero no existe UV en stock (picking + otros)
  $uvShortage = 0; $ucOnly = 0; $uvShortageButEquivPossible = 0;
  foreach ($outItems as $r) {
    $needsUv = ($r['need_uv'] ?? 0) > 0;
    $hasUv = ($r['pick_uv'] ?? 0) + ($r['other_uv'] ?? 0) > 0;
    $hasUc = ($r['pick_uc'] ?? 0) + ($r['other_uc'] ?? 0) > 0;
    if ($needsUv && !$hasUv) $uvShortage++;
    if (!$hasUv && $hasUc) $ucOnly++;
    $equivUv = ($r['pick_uv_equiv_from_uc'] ?? 0) + ($r['other_uv_equiv_from_uc'] ?? 0);
    if ($needsUv && !$hasUv && $equivUv > 0) $uvShortageButEquivPossible++;
  }

  // Resumen de pendientes a nivel pedido
  $pendingUv = max(0, (int)$totRow['expected_uv'] - (int)$totRow['prepared_uv']);
  $pendingUc = max(0, (int)$totRow['expected_uc'] - (int)$totRow['prepared_uc']);
  $itemsNeedUv = 0; $itemsNeedUc = 0;
  foreach ($outItems as $r) {
    if (($r['need_uv'] ?? 0) > 0) $itemsNeedUv++;
    if (($r['need_uc'] ?? 0) > 0) $itemsNeedUc++;
  }

  respond([
    'ok' => true,
    'deposito' => [ 'id' => $depId, 'code' => (string)$depRow['code'] ],
    'totals' => [
      'expected_uv' => (int)$totRow['expected_uv'],
      'prepared_uv' => (int)$totRow['prepared_uv'],
      'pending_uv'  => (int)$pendingUv,
      'expected_uc' => (int)$totRow['expected_uc'],
      'prepared_uc' => (int)$totRow['prepared_uc'],
      'pending_uc'  => (int)$pendingUc,
      'items_need_uv' => (int)$itemsNeedUv,
      'items_need_uc' => (int)$itemsNeedUc
    ],
    'items' => $outItems,
    'metrics' => [
      'items_uv_need_but_no_uv_stock' => $uvShortage,
      'items_only_uc_available' => $ucOnly,
      'items_uv_short_but_equiv_from_uc_possible' => $uvShortageButEquivPossible
    ]
  ]);

} catch (Throwable $e) {
  respond(['ok'=>false,'error'=>'Error en diagnóstico','message'=>$e->getMessage()], 500);
}

?>
