<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';
require_once $ROOT . '/app/Support/EmbarqueDocs.php';
require_once $ROOT . '/app/Support/InventarioReport.php';

function respond(array $payload, int $code = 200): void {
    http_response_code($code);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

function embarque_norm_sku(?string $sku): string {
    $sku = trim((string)$sku);
    return $sku === '' ? '' : strtoupper($sku);
}

function embarque_fetch_product_conversions(PDO $pdo, array $skus): array {
    $normalized = [];
    foreach ($skus as $sku) {
        $norm = embarque_norm_sku($sku);
        if ($norm !== '') {
            $normalized[$norm] = $sku;
        }
    }
    if (empty($normalized)) {
        return [];
    }

    $skuColumn = inventario_detect_column($pdo, 'para_productos', ['sku', 'codigo', 'cod', 'sku_cliente']);
    if (!$skuColumn) {
        return [];
    }

    $placeholders = implode(',', array_fill(0, count($normalized), '?'));
    $sql = 'SELECT id, `' . $skuColumn . '` AS sku FROM para_productos WHERE `' . $skuColumn . '` IN (' . $placeholders . ')';
    $stmt = $pdo->prepare($sql);
    $index = 1;
    foreach ($normalized as $norm => $original) {
        $stmt->bindValue($index++, $original, PDO::PARAM_STR);
    }
    $stmt->execute();

    $skuToProduct = [];
    $productIds = [];
    foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) ?: [] as $row) {
        $pid = isset($row['id']) ? (int)$row['id'] : 0;
        $skuVal = isset($row['sku']) ? trim((string)$row['sku']) : '';
        $norm = embarque_norm_sku($skuVal);
        if ($pid <= 0 || $norm === '') {
            continue;
        }
        $skuToProduct[$norm] = [
            'producto_id' => $pid,
            'sku' => $skuVal,
        ];
        $productIds[] = $pid;
    }

    if (empty($productIds)) {
        return [];
    }

    $info = inventario_fetch_product_info($pdo, $productIds);

    $result = [];
    foreach ($skuToProduct as $norm => $meta) {
        $pid = $meta['producto_id'];
        if (!isset($info[$pid])) {
            continue;
        }
        $unitsPerCase = $info[$pid]['unidades_por_caja'];
        $casesPerPallet = $info[$pid]['cajas_por_pallet'];
        $unitsPerPallet = null;
        if ($unitsPerCase !== null && $casesPerPallet !== null) {
            $unitsPerPallet = (float)$unitsPerCase * (float)$casesPerPallet;
        }
        $result[$norm] = [
            'sku' => $meta['sku'],
            'producto_id' => $pid,
            'unidades_por_caja' => $unitsPerCase !== null ? (float)$unitsPerCase : null,
            'cajas_por_pallet' => $casesPerPallet !== null ? (float)$casesPerPallet : null,
            'unidades_por_pallet' => $unitsPerPallet !== null ? (float)$unitsPerPallet : null,
        ];
    }

    return $result;
}

function embarque_calculate_total_unidades(array $row, ?array $conversion): array {
    $unitsPerCase = null;
    $casesPerPallet = null;
    $unitsPerPallet = null;

    if ($conversion !== null) {
        if (array_key_exists('unidades_por_caja', $conversion) && $conversion['unidades_por_caja'] !== null) {
            $unitsPerCase = (float)$conversion['unidades_por_caja'];
        }
        if (array_key_exists('cajas_por_pallet', $conversion) && $conversion['cajas_por_pallet'] !== null) {
            $casesPerPallet = (float)$conversion['cajas_por_pallet'];
        }
        if (array_key_exists('unidades_por_pallet', $conversion) && $conversion['unidades_por_pallet'] !== null) {
            $unitsPerPallet = (float)$conversion['unidades_por_pallet'];
        }
    }

    $effectiveUnitsPerCase = ($unitsPerCase !== null && $unitsPerCase > 0) ? $unitsPerCase : 1.0;
    $effectiveCasesPerPallet = ($casesPerPallet !== null && $casesPerPallet > 0) ? $casesPerPallet : 1.0;

    if ($unitsPerPallet === null && $unitsPerCase !== null && $casesPerPallet !== null) {
        $unitsPerPallet = $unitsPerCase * $casesPerPallet;
    }

    $pallets = max(0, (int)($row['pallets'] ?? 0));
    $cajas = max(0, (int)($row['cajas_sueltas'] ?? 0));
    $unidadesSueltas = max(0, (int)($row['unidades_sueltas'] ?? 0));
    $unidadesDanadas = max(0, (int)($row['unidades_danadas'] ?? 0));

    $totalCajas = ($pallets * $effectiveCasesPerPallet) + $cajas;
    $unitsFromCajas = $totalCajas * $effectiveUnitsPerCase;
    $total = $unitsFromCajas + $unidadesSueltas + $unidadesDanadas;
    $unidadesPicking = max(0.0, $total - $unidadesDanadas);

    return [
        'total' => $total,
        'unidades_a_picking' => $unidadesPicking,
        'unidades_por_caja' => $unitsPerCase,
        'cajas_por_pallet' => $casesPerPallet,
        'unidades_por_pallet' => $unitsPerPallet,
    ];
}

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
$allowedMotivos = ['', 'Motivo 1', 'Motivo 2', 'Motivo 3'];

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');
    $hasFacturaColumn = embarque_column_exists($pdo, 'so_embarque_devoluciones', 'factura');

    if ($method === 'GET' && isset($_GET['lookup_sku'])) {
        $lookupSku = trim((string)($_GET['lookup_sku'] ?? ''));
        if ($lookupSku === '') {
            respond(['ok' => false, 'error' => 'lookup_sku requerido'], 422);
        }
        $convMap = embarque_fetch_product_conversions($pdo, [$lookupSku]);
        $conversion = null;
        if (!empty($convMap)) {
            $first = reset($convMap);
            if ($first !== false) {
                $conversion = [
                    'sku' => $first['sku'] ?? $lookupSku,
                    'producto_id' => $first['producto_id'] ?? null,
                    'unidades_por_caja' => $first['unidades_por_caja'] ?? null,
                    'cajas_por_pallet' => $first['cajas_por_pallet'] ?? null,
                    'unidades_por_pallet' => $first['unidades_por_pallet'] ?? null,
                ];
            }
        }
        respond(['ok' => true, 'conversion' => $conversion]);
    }

    if ($method === 'GET') {
        $embarqueId = isset($_GET['embarque_id']) ? (int)$_GET['embarque_id'] : 0;
        if ($embarqueId <= 0) {
            respond(['ok' => false, 'error' => 'embarque_id requerido'], 422);
        }
        $selectCols = ['id'];
        if ($hasFacturaColumn) {
            $selectCols[] = 'factura';
        }
        $selectCols = array_merge($selectCols, ['sku', 'pallets', 'cajas_sueltas', 'unidades_sueltas', 'unidades_danadas', 'motivo', 'lote']);
        $stmt = $pdo->prepare('SELECT ' . implode(', ', $selectCols) . ' FROM so_embarque_devoluciones WHERE embarque_id = :emb ORDER BY id ASC');
        $stmt->execute(['emb' => $embarqueId]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $rows = array_map(function(array $row) {
            return [
                'id' => (int)($row['id'] ?? 0),
                'factura' => isset($row['factura']) ? (string)$row['factura'] : '',
                'sku' => (string)($row['sku'] ?? ''),
                'pallets' => (int)($row['pallets'] ?? 0),
                'cajas_sueltas' => (int)($row['cajas_sueltas'] ?? 0),
                'unidades_sueltas' => (int)($row['unidades_sueltas'] ?? 0),
                'unidades_danadas' => (int)($row['unidades_danadas'] ?? 0),
                'motivo' => $row['motivo'] ?? '',
                'lote' => $row['lote'] ?? ''
            ];
        }, $rows);

        $skuList = array_map(static function(array $row) {
            return $row['sku'] ?? '';
        }, $rows);
        $conversionMap = embarque_fetch_product_conversions($pdo, $skuList);

        $totalUnidades = 0.0;
        $totalUnidadesPicking = 0.0;
        foreach ($rows as &$row) {
            $skuKey = embarque_norm_sku($row['sku'] ?? '');
            $conv = $skuKey !== '' && isset($conversionMap[$skuKey]) ? $conversionMap[$skuKey] : null;
            $calc = embarque_calculate_total_unidades($row, $conv);
            $row['total_unidades'] = (int) round($calc['total']);
            $row['unidades_a_picking'] = max(0, (int) round($calc['unidades_a_picking'] ?? 0));
            $row['unidades_por_caja'] = $calc['unidades_por_caja'];
            $row['cajas_por_pallet'] = $calc['cajas_por_pallet'];
            $row['unidades_por_pallet'] = $calc['unidades_por_pallet'];
            $totalUnidades += $row['total_unidades'];
            $totalUnidadesPicking += $row['unidades_a_picking'];
        }
        unset($row);

        $exposedConversions = [];
        foreach ($conversionMap as $entry) {
            if (!is_array($entry)) {
                continue;
            }
            $sku = $entry['sku'] ?? null;
            if ($sku === null) {
                continue;
            }
            $exposedConversions[$sku] = [
                'sku' => $sku,
                'producto_id' => $entry['producto_id'] ?? null,
                'unidades_por_caja' => $entry['unidades_por_caja'] ?? null,
                'cajas_por_pallet' => $entry['cajas_por_pallet'] ?? null,
                'unidades_por_pallet' => $entry['unidades_por_pallet'] ?? null,
            ];
        }

        respond([
            'ok' => true,
            'data' => $rows,
            'totals' => [
                'total_unidades' => (int) round($totalUnidades),
                'total_unidades_a_picking' => (int) round($totalUnidadesPicking),
            ],
            'conversions' => $exposedConversions,
        ]);
    }

    if ($method === 'POST') {
        $body = json_decode(file_get_contents('php://input') ?: '[]', true) ?: [];
        $embarqueId = isset($body['embarque_id']) ? (int)$body['embarque_id'] : 0;
        if ($embarqueId <= 0) {
            respond(['ok' => false, 'error' => 'embarque_id requerido'], 422);
        }
        $rowId = isset($body['row_id']) ? (int)$body['row_id'] : 0;
        $data = isset($body['data']) && is_array($body['data']) ? $body['data'] : [];

        $factura = trim((string)($data['factura'] ?? ''));
        if ($factura === '') {
            $factura = null;
        } elseif (strlen($factura) > 100) {
            $factura = substr($factura, 0, 100);
        }

        $sku = trim((string)($data['sku'] ?? ''));
        if ($sku === '') {
            $sku = null;
        } elseif (strlen($sku) > 100) {
            $sku = substr($sku, 0, 100);
        }

        $pallets = max(0, (int)($data['pallets'] ?? 0));
        $cajas = max(0, (int)($data['cajas_sueltas'] ?? 0));
        $unidades = max(0, (int)($data['unidades_sueltas'] ?? 0));
        $danadas = max(0, (int)($data['unidades_danadas'] ?? 0));
        $motivo = (string)($data['motivo'] ?? '');
        if (!in_array($motivo, $allowedMotivos, true)) {
            if ($motivo !== '') {
                respond(['ok' => false, 'error' => 'Motivo inválido'], 422);
            }
            $motivo = '';
        }
        $lote = trim((string)($data['lote'] ?? ''));
        if (strlen($lote) > 100) {
            $lote = substr($lote, 0, 100);
        }

        if ($rowId > 0) {
            $verify = $pdo->prepare('SELECT id FROM so_embarque_devoluciones WHERE id = :id AND embarque_id = :emb LIMIT 1');
            $verify->execute(['id' => $rowId, 'emb' => $embarqueId]);
            if (!$verify->fetch(PDO::FETCH_ASSOC)) {
                respond(['ok' => false, 'error' => 'Registro no encontrado'], 404);
            }
            $updateParts = [];
            $params = [
                'sku' => $sku,
                'pallets' => $pallets,
                'cajas' => $cajas,
                'unidades' => $unidades,
                'danadas' => $danadas,
                'motivo' => $motivo,
                'lote' => $lote,
                'id' => $rowId,
                'emb' => $embarqueId
            ];
            if ($hasFacturaColumn) {
                $updateParts[] = 'factura = :factura';
                $params['factura'] = $factura;
            }
            $updateParts[] = 'sku = :sku';
            $updateParts[] = 'pallets = :pallets';
            $updateParts[] = 'cajas_sueltas = :cajas';
            $updateParts[] = 'unidades_sueltas = :unidades';
            $updateParts[] = 'unidades_danadas = :danadas';
            $updateParts[] = 'motivo = :motivo';
            $updateParts[] = 'lote = :lote';
            $upd = $pdo->prepare('UPDATE so_embarque_devoluciones SET ' . implode(', ', $updateParts) . ' WHERE id = :id AND embarque_id = :emb');
            $upd->execute($params);
            respond(['ok' => true, 'id' => $rowId]);
        }
        $insertCols = ['embarque_id', 'sku', 'pallets', 'cajas_sueltas', 'unidades_sueltas', 'unidades_danadas', 'motivo', 'lote'];
        $insertVals = [':emb', ':sku', ':pallets', ':cajas', ':unidades', ':danadas', ':motivo', ':lote'];
        $params = [
            'emb' => $embarqueId,
            'sku' => $sku,
            'pallets' => $pallets,
            'cajas' => $cajas,
            'unidades' => $unidades,
            'danadas' => $danadas,
            'motivo' => $motivo,
            'lote' => $lote
        ];
        if ($hasFacturaColumn) {
            array_splice($insertCols, 1, 0, 'factura');
            array_splice($insertVals, 1, 0, ':factura');
            $params['factura'] = $factura;
        }
        $ins = $pdo->prepare('INSERT INTO so_embarque_devoluciones (' . implode(', ', $insertCols) . ')
                               VALUES (' . implode(', ', $insertVals) . ')');
        $ins->execute($params);
        $newId = (int)$pdo->lastInsertId();
        respond(['ok' => true, 'id' => $newId], 201);
    }

    if ($method === 'DELETE') {
        $body = json_decode(file_get_contents('php://input') ?: '[]', true) ?: [];
        $embarqueId = isset($body['embarque_id']) ? (int)$body['embarque_id'] : 0;
        $rowId = isset($body['row_id']) ? (int)$body['row_id'] : 0;
        if ($embarqueId <= 0 || $rowId <= 0) {
            respond(['ok' => false, 'error' => 'embarque_id y row_id requeridos'], 422);
        }
        $del = $pdo->prepare('DELETE FROM so_embarque_devoluciones WHERE id = :id AND embarque_id = :emb LIMIT 1');
        $del->execute(['id' => $rowId, 'emb' => $embarqueId]);
        if ($del->rowCount() === 0) {
            respond(['ok' => false, 'error' => 'Registro no encontrado'], 404);
        }
        respond(['ok' => true]);
    }

    respond(['ok' => false, 'error' => 'Método no permitido'], 405);
} catch (Throwable $e) {
    respond(['ok' => false, 'error' => 'Error interno', 'message' => $e->getMessage()], 500);
}
