<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';
require_once $ROOT . '/app/Support/ApiHelpers.php';

function respond(array $payload, int $code = 200): void {
    http_response_code($code);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

try {
    $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
    if (!in_array($method, ['GET', 'POST'], true)) {
        respond(['ok' => false, 'error' => 'Método no permitido'], 405);
    }

    if ($method === 'POST') {
        $raw = file_get_contents('php://input') ?: '';
        $body = json_decode($raw, true);
        if (!is_array($body)) {
            $body = $_POST;
        }
        $embarqueId = isset($body['embarque_id']) ? (int)$body['embarque_id'] : 0;
        $removeRetorno = !empty($body['remove_retorno']);
    } else {
        $embarqueId = isset($_GET['embarque_id']) ? (int)$_GET['embarque_id'] : 0;
        $removeRetorno = false;
    }

    if ($embarqueId <= 0) {
        respond(['ok' => false, 'error' => 'embarque_id requerido'], 422);
    }

    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');

    $shouldLock = ($method === 'POST');
    if ($shouldLock) {
        $pdo->beginTransaction();
    }

    $sqlCur = "SELECT e.estado_id, e.movil_id, ee.code AS estado_code, ee.orden AS estado_orden\n                                FROM so_embarque e\n                                JOIN so_embarque_estado ee ON ee.id = e.estado_id\n                               WHERE e.id = ? LIMIT 1";
    if ($shouldLock) {
        $sqlCur .= ' FOR UPDATE';
    }
    $stCur = $pdo->prepare($sqlCur);
    $stCur->execute([$embarqueId]);
    $cur = $stCur->fetch(PDO::FETCH_ASSOC);
    if (!$cur) {
        if ($shouldLock && $pdo->inTransaction()) {
            $pdo->rollBack();
        }
        respond(['ok' => false, 'error' => 'No existe el embarque'], 404);
    }

    $curCode = strtoupper((string)($cur['estado_code'] ?? ''));
    $curOrden = isset($cur['estado_orden']) ? (int)$cur['estado_orden'] : 0;

    $stPrev = $pdo->prepare("SELECT id, code FROM so_embarque_estado WHERE orden < ? AND activo = 1 ORDER BY orden DESC LIMIT 1");
    $stPrev->execute([$curOrden]);
    $prev = $stPrev->fetch(PDO::FETCH_ASSOC);
    if ($method === 'GET') {
        respond([
            'ok' => true,
            'current_code' => $curCode,
            'previous_code' => $prev ? strtoupper((string)$prev['code']) : null,
        ]);
    }

    if (!$prev) {
        if ($shouldLock && $pdo->inTransaction()) {
            $pdo->rollBack();
        }
        respond(['ok' => false, 'error' => 'No hay un estado anterior disponible'], 409);
    }

    $prevId = (int)$prev['id'];
    $prevCode = strtoupper((string)$prev['code']);
    $movilId = isset($cur['movil_id']) && $cur['movil_id'] !== null ? (int)$cur['movil_id'] : null;
    $hasOcupadoColumn = columnExists($pdo, 'para_moviles', 'ocupado');

    try {
        if ($removeRetorno && $curCode === 'FINALIZADO') {
            $stRet = $pdo->prepare("SELECT id FROM so_retorno WHERE embarque_id = ? ORDER BY id DESC LIMIT 1");
            $stRet->execute([$embarqueId]);
            $retId = $stRet->fetchColumn();
            if ($retId) {
                $pdo->prepare('DELETE FROM so_retorno WHERE id = ?')->execute([(int)$retId]);
            }
        }

        $upd = $pdo->prepare('UPDATE so_embarque SET estado_id = ? WHERE id = ?');
        $upd->execute([$prevId, $embarqueId]);

        if ($hasOcupadoColumn && $movilId && in_array($curCode, ['VUELTA','FINALIZADO'], true)) {
            $occ = $pdo->prepare('UPDATE para_moviles SET ocupado = 1, updated_at = NOW() WHERE id = ?');
            $occ->execute([$movilId]);
        }

        if ($shouldLock && $pdo->inTransaction()) {
            $pdo->commit();
        }
        respond([
            'ok' => true,
            'estado_code' => $prevCode,
            'previous_code' => $curCode,
        ]);
    } catch (Throwable $e) {
        if ($shouldLock && $pdo->inTransaction()) {
            $pdo->rollBack();
        }
        throw $e;
    }
} catch (Throwable $e) {
    respond(['ok' => false, 'error' => 'Error interno', 'message' => $e->getMessage()], 500);
}
