<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';
require_once $ROOT . '/app/Support/EmbarqueDocs.php';

function respond(array $payload, int $code = 200): void {
    http_response_code($code);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

try {
    $embarqueId = isset($_GET['embarque_id']) ? (int)$_GET['embarque_id'] : 0;
    if ($embarqueId <= 0) {
        respond(['ok' => false, 'error' => 'embarque_id requerido'], 422);
    }

    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');

    $context = embarque_collect_destinatarios_docs($pdo, $embarqueId);
    $byDest = $context['destinatarios'] ?? [];
    $docsByPedido = $context['pedido_docs'] ?? [];
    $pedidoMeta = $context['pedido_meta'] ?? [];
    $pedidoItems = $context['pedido_items'] ?? [];

    $facturaMap = [];

    $destByPedido = [];
    foreach ($byDest as $destId => $info) {
        if (!empty($info['pedido_dest_ids'])) {
            foreach (array_keys($info['pedido_dest_ids']) as $pid) {
                $destByPedido[(int)$pid] = [
                    'destinatario_id' => (int)$destId,
                    'destinatario' => $info['destinatario'] ?? null,
                ];
            }
        }
    }

    foreach ($docsByPedido as $pid => $docs) {
        $pid = (int)$pid;
        $meta = $pedidoMeta[$pid] ?? [];
        $dest = $destByPedido[$pid] ?? [
            'destinatario_id' => $meta['destinatario_id'] ?? null,
            'destinatario' => $meta['destinatario'] ?? null,
        ];
        foreach ($docs as $doc) {
            $candidate = $doc['factura'] ?? $doc['doc_numero'] ?? $doc['label'] ?? '';
            $candidate = trim((string)$candidate);
            if ($candidate === '') {
                continue;
            }
            $key = strtoupper($candidate);
            if (!isset($facturaMap[$key])) {
                $facturaMap[$key] = [
                    'value' => $candidate,
                    'label' => $doc['label'] ?? $candidate,
                    'doc_tipo' => $doc['doc_tipo'] ?? null,
                    'doc_numero' => $doc['doc_numero'] ?? null,
                    'factura' => $doc['factura'] ?? $candidate,
                    'destinatario_id' => $dest['destinatario_id'] ?? null,
                    'destinatario' => $dest['destinatario'] ?? null,
                    'pedido_dest_ids' => [],
                    'items' => [],
                ];
            }
            $facturaMap[$key]['pedido_dest_ids'][] = $pid;
        }
    }

    // Incluir facturas previamente registradas en rendiciones o devoluciones, por si no existen docs asociados
    $includeExisting = static function (PDO $pdo, int $embarqueId, string $table, string $column, array &$facturaMap): void {
        $sql = 'SELECT DISTINCT ' . $column . ' AS factura FROM ' . $table . ' WHERE embarque_id = ?';
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$embarqueId]);
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $fact = trim((string)($row['factura'] ?? ''));
            if ($fact === '') {
                continue;
            }
            $key = strtoupper($fact);
            if (!isset($facturaMap[$key])) {
                $facturaMap[$key] = [
                    'value' => $fact,
                    'label' => $fact,
                    'doc_tipo' => 'FACTURA',
                    'doc_numero' => $fact,
                    'factura' => $fact,
                    'destinatario_id' => null,
                    'destinatario' => null,
                    'pedido_dest_ids' => [],
                    'items' => [],
                ];
            }
        }
    };

    if (embarque_table_exists($pdo, 'so_embarque_rendiciones')) {
        $includeExisting($pdo, $embarqueId, 'so_embarque_rendiciones', 'factura', $facturaMap);
    }
    if (embarque_table_exists($pdo, 'so_embarque_devoluciones') && embarque_column_exists($pdo, 'so_embarque_devoluciones', 'factura')) {
        $includeExisting($pdo, $embarqueId, 'so_embarque_devoluciones', 'factura', $facturaMap);
    }

    foreach ($facturaMap as $key => &$factura) {
        $items = [];
        $seen = [];
        foreach ($factura['pedido_dest_ids'] as $pid) {
            $pid = (int)$pid;
            if (empty($pedidoItems[$pid])) {
                continue;
            }
            foreach ($pedidoItems[$pid] as $item) {
                if (!is_array($item)) {
                    continue;
                }
                $sku = isset($item['sku']) ? trim((string)$item['sku']) : '';
                $skuKey = $sku !== '' ? strtoupper($sku) : '';
                if ($sku === '') {
                    continue;
                }
                $itemId = isset($item['item_id']) ? (int)$item['item_id'] : 0;
                $uniqueKey = $itemId > 0 ? ('id:' . $itemId) : ($skuKey !== '' ? ('sku:' . $skuKey) : null);
                if ($uniqueKey === null) {
                    continue;
                }
                if (isset($seen[$uniqueKey])) {
                    continue;
                }
                $seen[$uniqueKey] = true;
                $descripcion = isset($item['descripcion']) ? trim((string)$item['descripcion']) : '';
                $label = isset($item['label']) ? trim((string)$item['label']) : '';
                if ($label === '') {
                    if ($sku !== '') {
                        $label = $sku;
                    } elseif ($descripcion !== '') {
                        $label = $descripcion;
                    } elseif ($itemId > 0) {
                        $label = 'Item #' . $itemId;
                    }
                }
                $items[] = [
                    'item_id' => $itemId > 0 ? $itemId : null,
                    'pedido_dest_id' => $pid,
                    'producto_id' => isset($item['producto_id']) ? (int)$item['producto_id'] : null,
                    'sku' => $sku,
                    'descripcion' => $descripcion !== '' ? $descripcion : null,
                    'label' => $label !== '' ? $label : null,
                ];
            }
        }
        $factura['items'] = $items;
    }
    unset($factura);

    $facturas = array_values(array_map(static function (array $item): array {
        $item['pedido_dest_ids'] = array_values(array_unique(array_map('intval', $item['pedido_dest_ids'] ?? [])));
        return $item;
    }, $facturaMap));

    usort($facturas, static function (array $a, array $b): int {
        return strcmp($a['value'], $b['value']);
    });

    respond(['ok' => true, 'facturas' => $facturas]);
} catch (Throwable $e) {
    respond([
        'ok' => false,
        'error' => 'Error interno',
        'message' => $e->getMessage(),
    ], 500);
}
